﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.IO;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;
    using LegacyFormat = NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat;

    /// <summary>
    /// RVL サウンドセットドキュメントをサウンドセットドキュメントに変換します。
    /// </summary>
    public class SoundSetRvlToSoundSetDocumentConverterCommon : SoundSetRvlToSoundSetDocumentConverter
    {
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits;
        private IObjectFactory<Type, Component> componentFactory;

        public SoundSetRvlToSoundSetDocumentConverterCommon(
            SoundIntermediateOutputTraits intermediateOutputTraits,
            IObjectFactory<Type, Component> componentFactory)
            : base(intermediateOutputTraits, componentFactory)
        {
            this.intermediateOutputTraits = intermediateOutputTraits;
            this.componentFactory = componentFactory;
        }

        //-----------------------------------------------------------------
        // サウンドセットアイテムのコンバート
        //-----------------------------------------------------------------

        protected override StreamSoundBase ConvertStreamSound(LegacyFormat.Nw4rSoundSetStrm source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Stream Sound: ({0})", source.Label);

            if (source.TrackList.Count == 0)
            {
                throw new Exception("invalid stream sound found.");
            }

            StreamSound dest = componentFactory.Create(typeof(StreamSoundBase)) as StreamSound;
            {
                dest.ActorPlayer = source.ActorPlayer;
                dest.ColorIndex = source.ColorIndex;
                dest.Comment = source.Comment == null ? string.Empty : source.Comment;
                dest.Encoding = this.ConvertWaveEncoding(source.TrackList[0].FileFormat);
                dest.IsEnabled = source.Enabled;
                dest.Name = source.Label;
                dest.PanCurve = this.ConvertPanCurve(source.PanCurve);
                dest.PanMode = this.ConvertPanMode(source.PanMode);
                dest.PlayerPriority = source.PlayerPrio;
                dest.PlayerReference = source.Player;
                dest.Sound3D = this.ConvertSound3D(source.Sound3DParam);
                dest.UserParameter = source.UserParam;
                dest.Volume = source.Volume;
                dest.RemoteFilter = source.RemoteFilter;
            }

            foreach (LegacyFormat.Nw4rSoundSetStrm.Track track in source.TrackList)
            {
                dest.Children.Add(this.ConvertStreamSoundTrack(track));
            }

            return dest;
        }

        protected override SequenceSoundBase ConvertSequenceSound(LegacyFormat.Nw4rSoundSetSeq source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Sequence Sound: ({0})", source.Label);
            SequenceSound dest = componentFactory.Create(typeof(SequenceSoundBase)) as SequenceSound;
            {
                dest.ActorPlayer = source.ActorPlayer;
                dest.ChannelPriority = source.ChannelPrio;
                dest.ColorIndex = source.ColorIndex;
                dest.Comment = source.Comment == null ? string.Empty : source.Comment;
                dest.FilePath = source.FullPath;
                dest.FileType = this.ConvertSequenceSoundFileType(source.FileType);
                dest.IsEnabled = source.Enabled;
                dest.Name = source.Label;
                dest.PlayerPriority = source.PlayerPrio;
                dest.PlayerReference = source.Player;
                dest.ReleasePriorityFixed = source.ReleasePrioFixFlag;
                dest.Sound3D = this.ConvertSound3D(source.Sound3DParam);
                dest.StartPosition = source.StartPosition == null ? string.Empty : source.StartPosition;
                dest.UserParameter = source.UserParam;
                dest.Volume = source.Volume;
                dest.RemoteFilter = source.RemoteFilter;
            }

            if (dest.FileType == SequenceSoundFileType.Text)
            {
                dest.FilePath = Path.ChangeExtension(
                    dest.FilePath,
                    this.intermediateOutputTraits.TextSequenceSoundFileExtension
                    );
            }

            dest.SoundSetBankReferences.Add(
                new ComponentReference()
                {
                    TargetName = source.Bank,
                }
                );

            return dest;
        }

        protected override WaveSoundBase ConvertWaveSound(LegacyFormat.Nw4rSoundSetWaveSound source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Wave Sound: ({0})", source.Label);
            WaveSound dest = componentFactory.Create(typeof(WaveSoundBase)) as WaveSound;
            {
                dest.ActorPlayer = source.ActorPlayer;
                dest.ChannelPriority = source.ChannelPrio;
                dest.ColorIndex = source.ColorIndex;
                dest.Comment = source.Comment == null ? string.Empty : source.Comment;
                dest.Encoding = this.ConvertWaveEncoding(source.FileFormat);
                dest.FilePath = source.FullPath;
                dest.IsEnabled = source.Enabled;
                dest.Name = source.Label;
                dest.Pan = source.Pan;
                dest.PanCurve = this.ConvertPanCurve(source.PanCurve);
                dest.PanMode = this.ConvertPanMode(source.PanMode);
                dest.Pitch = (float)source.Pitch;
                dest.PlayerPriority = source.PlayerPrio;
                dest.PlayerReference = source.Player;
                dest.ReleaseEnvelope = source.AdsrEnvelope.Release;
                dest.ReleasePriorityFixed = source.ReleasePrioFixFlag;
                dest.Sends = new SendsCommon()
                {
                    AuxASend = source.FxSendA,
                    AuxBSend = source.FxSendB,
                    AuxCSend = source.FxSendC,
                    MainSend = source.MainSend,
                };
                dest.Sound3D = this.ConvertSound3D(source.Sound3DParam);
                dest.SurroundPan = source.SurroundPan;
                dest.UserParameter = source.UserParam;
                dest.Volume = source.Volume;
                dest.RemoteFilter = source.RemoteFilter;
            }

            return dest;
        }
    }
}
