﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.Foundation.ComponentModel;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using System;
    using System.Collections.Generic;
    using System.Linq;

    public class SoundProjectServiceCommon : SoundProjectService
    {
        private readonly string ChannelCountParameterName = ProjectParameterNames.StreamSoundTrack.ChannelCount;
        private readonly string NoSaveChannelCountParameterName = "@" + ProjectParameterNames.StreamSoundTrack.ChannelCount;

        private WeakCacheTable<SoundSetDocument, OperationExecutor> operationExecutors = new WeakCacheTable<SoundSetDocument, OperationExecutor>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="documentService">ドキュメントサービス。</param>
        /// <param name="intermediateOutputTraits">サウンド中間出力の特性を指定します。</param>
        public SoundProjectServiceCommon(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits)
            : base(documentService, intermediateOutputTraits)
        {
            documentService.DocumentSaving += OnDocumentSaving;
            documentService.DocumentSaved += OnDocumentSaved;
        }

        public OperationExecutor GetOperationExecutor(SoundSetDocument document)
        {
            return this.operationExecutors.GetOrAddValue(document, () => new OperationExecutor(document.OperationHistory));
        }

        protected override void OnSoundSetDocumentOpened(SoundDocumentResourceEventArgs e)
        {
            this.ResolveWaveSoundResourceReferences(
                this.DocumentService.Documents
                .OfType<SoundSetDocument>()
                .First(document => document.Resource == e.Resource));

            base.OnSoundSetDocumentOpened(e);
        }

        protected override void OnProjectDocumentOpened(SoundDocumentResourceEventArgs e)
        {
            this.ResolveSoundArchiveOutputType();
            base.OnProjectDocumentOpened(e);
        }

        protected override void OnProjectReloaded(ProjectReloadEventArgs e)
        {
            if (e.IsReloaded == true)
            {
                this.ResolveSoundArchiveOutputType();
            }

            base.OnProjectReloaded(e);
        }

        /// <summary>
        /// サウンドプロジェクトリソースに保存します。
        /// </summary>
        protected override void SaveInternal()
        {
            this.DocumentService.Documents
                .OfType<SoundSetDocument>()
                .Select(document => document.SoundSet)
                .ForEach(soundSet => this.UpdateWaveSoundResourceReferences(soundSet));

            base.SaveInternal();
        }

        private void ResolveWaveSoundResourceReferences(SoundSetDocument document)
        {
            // SoundSet.ItemResources の waveSoundResourceData 参照情報を解決し、WaveSound.WaveSoundResoruce に設定します。
            foreach (var waveSoundResourceData in document.SoundSet.ItemResources.Values.OfType<WaveSoundResourceData>())
            {
                var targetWaveSound = ((ICollection<Component>)document.SoundSet.Children)
                    .OfType<WaveSoundSetPack>()
                    .SelectMany<WaveSoundSetPack, WaveSoundSetBase>(pack => pack.Children.Cast<WaveSoundSetBase>())
                    .SelectMany<WaveSoundSetBase, WaveSound>(waveSoundSet => waveSoundSet.Children.Cast<WaveSound>())
                    .Where(item => item.Name == waveSoundResourceData.Name)
                    .FirstOrDefault();

                if (targetWaveSound != null)
                {
                    targetWaveSound.WaveSoundResource = new WaveSoundResource(
                        waveSoundResourceData,
                        this.GetOperationExecutor(document)
                        );
                }
            }
        }

        /// <summary>
        /// SoundProjectの読み込み時にstring, SoundArchiveOutputTypeだった組み合わせを SoundSet, SoundArchiveOutputTypeに変換します。
        /// </summary>
        private void ResolveSoundArchiveOutputType()
        {
            this.ProjectDocument.Project.SoundArchiveOutputTypes.Clear();

            foreach (var pair in this.Project.UnresolvedSoundArchiveOutputTypes)
            {
                var soundSet = this.SoundSets
                    .Where(s => s.Name == pair.Key)
                    .FirstOrDefault();
                if (soundSet != null)
                {
                    this.AddSoundArchiveOutputType(soundSet, pair.Value);
                }
            }
        }

        private void UpdateWaveSoundResourceReferences(SoundSet soundSet)
        {
            soundSet.ItemResources.Clear();

            // WaveSound.WaveSoundResoruce を SoundSet.ItemResources に書き出します。
            ((ICollection<Component>)soundSet.Children)
                .OfType<WaveSoundSetPack>()
                .SelectMany<WaveSoundSetPack, WaveSoundSetBase>(pack => pack.Children.Cast<WaveSoundSetBase>())
                .SelectMany<WaveSoundSetBase, WaveSound>(waveSoundSet => waveSoundSet.Children.Cast<WaveSound>())
                .Where(waveSound => waveSound.WaveSoundResource != null)
                .ForEach(delegate (WaveSound waveSound)
                {
                    waveSound.WaveSoundResource.DataModel.Name = waveSound.Name;
                    soundSet.ItemResources.Add(waveSound.Name, waveSound.WaveSoundResource.DataModel);
                });
        }

        private void OnDocumentSaving(object sender, DocumentEventArgs e)
        {
            SoundSetDocument doc = e.Document as SoundSetDocument;

            if (doc != null)
            {
                // ChannelCount パラメータ名を一時的に変更して保存対象から外します。
                StreamSoundTrackRenameParameters(doc, this.ChannelCountParameterName, this.NoSaveChannelCountParameterName);
            }
        }

        private void OnDocumentSaved(object sender, DocumentEventArgs e)
        {
            SoundSetDocument doc = e.Document as SoundSetDocument;

            if (doc != null)
            {
                // 一時的に変更して保存対象から外した ChannelCount パラメータ名を元に戻します。
                StreamSoundTrackRenameParameters(doc, this.NoSaveChannelCountParameterName, this.ChannelCountParameterName);
            }
        }

        private void StreamSoundTrackRenameParameters(SoundSetDocument doc, string oldKey, string newKey)
        {
            var pack = doc.SoundSet.Children.Cast<Component>().Where(c => c is StreamSoundPack).First();

            foreach (StreamSoundBase streamSound in pack.Children)
            {
                foreach (StreamSoundTrackBase track in streamSound.Children)
                {
                    if (track.Parameters.ContainsKey(oldKey) == true)
                    {
                        IParameterValue value = track.Parameters.GetValue(oldKey);
                        track.Parameters.RemoveValue(oldKey);
                        track.Parameters.AddValue(newKey, value);
                    }
                }
            }
        }
    }
}
