﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using Core;
    using Core.IO;
    using Projects;
    using ToolDevelopmentKit;

    /// <summary>
    /// コンポーネントのファイルIDを作成します。
    /// </summary>
    internal class FileIDFactory<TComponent> : IObjectFactory<Component, FileID>
        where TComponent : Component
    {
        private string basePath;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="basePath">相対パスの基点となるパスを指定します。</param>
        public FileIDFactory(string basePath)
        {
            Ensure.Argument.NotNull(basePath);
            this.basePath = basePath;
        }

        /// <summary>
        /// ファイルパスを使用しないファイルIDのためのコンストラクタです。
        /// </summary>
        protected FileIDFactory()
        {
        }

        /// <summary>
        /// 指定されたキーからオブジェクトのインスタンスを作成します。
        /// </summary>
        /// <param name="key">キーを指定します。</param>
        /// <returns>生成したオブジェクトのインスタンスを返します。</returns>
        public FileID Create(Component key)
        {
            Ensure.Argument.NotNull(key);
            Ensure.Argument.True(key is TComponent);
            return this.CreateInternal(key as TComponent);
        }

        /// <summary>
        /// コンポーネントのファイルIDを作成します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <returns>ファイルIDを返します。</returns>
        protected virtual FileID CreateInternal(TComponent component)
        {
            Assertion.Argument.NotNull(component);
            return new FileID(this.GetRelativeFilePath(component));
        }

        /// <summary>
        /// コンポーネントの相対ファイルパスを取得します。
        /// </summary>
        /// <param name="component">コンポーネントを指定します。</param>
        /// <param name="basePath">相対パスの基点となるパスを指定します。</param>
        /// <returns>相対パスを返します。</returns>
        protected string GetRelativeFilePath(Component component)
        {
            Ensure.Argument.NotNull(component);

            if (basePath == null)
            {
                throw new Exception("internal error : basePath must not be null.");
            }

            string filePath = component.GetFilePathForConvert();
            if (filePath == null || filePath.Length == 0)
            {
                return string.Empty;
            }

            return PathEx.MakeRelative(filePath, basePath);
        }
    }
}
