﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Linq;
    using System.Text;
    using Core.Collections;
    using Projects;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Collections;
    using System.IO;

    /// <summary>
    /// サウンドアーカイブバイナリ構築用のコンテキスト情報を格納します。
    /// </summary>
    public class SoundArchiveContext : ComponentContext
    {
        private SoundProjectConversionSettings settings;

        private ItemList<StreamSoundBase> streamSounds = new ItemList<StreamSoundBase>();
        private ItemList<WaveSoundBase> waveSounds = new ItemList<WaveSoundBase>();
        private ItemList<SequenceSoundBase> sequenceSounds = new ItemList<SequenceSoundBase>();
        private ItemList<WaveSoundSetBase> waveSoundSets = new ItemList<WaveSoundSetBase>();
        private ItemList<SequenceSoundSetBase> sequenceSoundSets = new ItemList<SequenceSoundSetBase>();
        private ItemList<SoundSetBankBase> soundSetBanks = new ItemList<SoundSetBankBase>();
        private ItemList<WaveArchiveBase> waveArchives = new ItemList<WaveArchiveBase>();
        private ItemList<PlayerBase> players = new ItemList<PlayerBase>();
        private ItemList<GroupBase> groups = new ItemList<GroupBase>();
        private ItemList<SoundSetItem> allItems = new ItemList<SoundSetItem>();

        private ItemList<SoundSetItem> stringTableItems = new ItemList<SoundSetItem>();

        private Dictionary<SoundSetBankBase, Bank> bankDictionary = new Dictionary<SoundSetBankBase, Bank>();
        private Dictionary<Bank, string> bankFilePathDictionary = new Dictionary<Bank, string>();

        private List<RegisteringOutputInfo> registeringOutputs = new List<RegisteringOutputInfo>();

        public SoundArchiveContext(
            SoundProjectConversionTraits traits, SoundProjectConversionSettings settings, string projectFilePath)
            : base(traits, projectFilePath)
        {
            Ensure.Argument.NotNull(settings);
            this.settings = settings;

            streamSounds.CollectionChanged += OnItemCollectionChanged;
            waveSounds.CollectionChanged += OnItemCollectionChanged;
            sequenceSounds.CollectionChanged += OnItemCollectionChanged;
            waveSoundSets.CollectionChanged += OnItemCollectionChanged;
            sequenceSoundSets.CollectionChanged += OnItemCollectionChanged;
            soundSetBanks.CollectionChanged += OnItemCollectionChanged;
            waveArchives.CollectionChanged += OnItemCollectionChanged;
            players.CollectionChanged += OnItemCollectionChanged;
            groups.CollectionChanged += OnItemCollectionChanged;
        }

        public new SoundProjectConversionTraits Traits
        {
            get { return base.Traits as SoundProjectConversionTraits; }
        }

        public SoundProjectConversionSettings Settings
        {
            get { return this.settings; }
        }

        public SoundProject Project { get; set; }

        public SoundSet AddonSoundSet { get; set; }

        // HACK : ★IReadOnlyKeyedList<> に変更する
        public IEnumerable<SoundSetItem> AllItems
        {
            get { return this.allItems; }
        }

        public IEnumerable<Sound> AllSounds
        {
            get
            {
                foreach (Sound streamSound in this.StreamSounds)
                {
                    yield return streamSound;
                }

                foreach (Sound waveSound in this.WaveSounds)
                {
                    yield return waveSound;
                }

                foreach (Sound sequenceSound in this.SequenceSounds)
                {
                    yield return sequenceSound;
                }
            }
        }

        public IEnumerable<SoundSetItem> AllSoundGroups
        {
            get
            {
                foreach (SoundSetItem soundGroup in this.WaveSoundSets)
                {
                    yield return soundGroup;
                }

                foreach (SoundSetItem soundGroup in this.SequenceSoundSets)
                {
                    yield return soundGroup;
                }
            }
        }

        public IKeyedList<string, StreamSoundBase> StreamSounds
        {
            get { return this.streamSounds; }
        }

        public IKeyedList<string, WaveSoundBase> WaveSounds
        {
            get { return this.waveSounds; }
        }

        public IKeyedList<string, SequenceSoundBase> SequenceSounds
        {
            get { return this.sequenceSounds; }
        }

        public IKeyedList<string, WaveSoundSetBase> WaveSoundSets
        {
            get { return this.waveSoundSets; }
        }

        public IKeyedList<string, SequenceSoundSetBase> SequenceSoundSets
        {
            get { return this.sequenceSoundSets; }
        }

        public IKeyedList<string, SoundSetBankBase> SoundSetBanks
        {
            get { return this.soundSetBanks; }
        }

        public IKeyedList<string, WaveArchiveBase> WaveArchives
        {
            get { return this.waveArchives; }
        }

        public IKeyedList<string, PlayerBase> Players
        {
            get { return this.players; }
        }

        public IKeyedList<string, GroupBase> Groups
        {
            get { return this.groups; }
        }

        public IKeyedList<string, SoundSetItem> StringTableItems
        {
            get { return this.stringTableItems; }
        }

        public IDictionary<SoundSetBankBase, Bank> BankDictionary
        {
            get { return this.bankDictionary; }
        }

        public IDictionary<Bank, string> BankFilePathDictionary
        {
            get { return this.bankFilePathDictionary; }
        }

        /// <summary>
        /// 登録完了前の出力リストを取得します。
        /// </summary>
        /// <remarks>
        /// コンバートの処理順序を遅延するために暫定措置として利用されています。
        /// 本来、***OutputFactory で出力を登録したいのですが、その場合、不要なキャッシュがクリアされる前に処理されるために、
        /// 出力名が重複して不要なポストフィックス（番号）が付加されてしまう問題が発生します。
        /// 出力の登録タイミングをキャッシュクリア後に遅延させることで、この問題に対処しています。
        /// より良い方法が見つかれば、廃止を検討してください。
        /// </remarks>
        internal IList<RegisteringOutputInfo> RegisteringOutputs
        {
            get { return this.registeringOutputs; }
        }

        internal IOutputItem SoundArchiveOutput { get; set; }
        internal IOutputItem SoundArchiveMapOutput { get; set; }

        internal string GetSoundArchiveName()
        {
            Assertion.Operation.ObjectNotNull(this.Project);
            return this.AddonSoundSet == null ? this.Project.Name : this.AddonSoundSet.Name;
        }

        public bool IsTargetSoundSet(SoundSet soundSet)
        {
            if (this.AddonSoundSet == null)
            {
                SoundArchiveOutputTypes outputType;

                if (!this.Project.SoundArchiveOutputTypes.TryGetValue(soundSet, out outputType))
                {
                    return true;
                }

                return outputType == SoundArchiveOutputTypes.SoundArchive;
            }
            else
            {
                return this.AddonSoundSet == soundSet;
            }
        }

        /// <summary>
        /// 外部サウンドアーカイブのバンクかどうかを調べます。
        /// </summary>
        /// <param name="soundSetBank">バンクを指定します。</param>
        /// <returns>外部サウンドアーカイブのバンクなら true、それ以外なら false を返します。</returns>
        public bool IsExternalSoundArchiveBank(SoundSetBankBase soundSetBank)
        {
            return !this.BankDictionary.ContainsKey(soundSetBank) && this.AddonSoundSet != null;
        }

        public string GetWavePreprocessExeFullPath()
        {
            if (this.Project == null)
            {
                return string.Empty;
            }

            return this.Project.DoUseBuiltInWavePreprocessExe
                ? this.Traits.BuiltInWavePreprocessExePath
                : this.Project.GetWavePreprocessExeFullPath(Path.GetDirectoryName(this.ProjectFilePath));
        }

        private void OnItemCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (SoundSetItem item in e.NewItems)
                    {
                        this.allItems.Add(item);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (SoundSetItem item in e.OldItems)
                    {
                        this.allItems.Remove(item);
                    }
                    break;

                default:
                    throw new NotImplementedException();
            }
        }

        private class ItemList<TItem> : KeyedListDecorator<string, TItem>
            where TItem : SoundSetItem
        {
            public ItemList()
                : base(new ObservableList<TItem>(), GetItemName)
            {
            }

            private static string GetItemName(TItem item)
            {
                return item.Name;
            }
        }
    }
}
