﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Collections.Generic;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    internal class SoundSetBankProcessorFactory : ComponentSetup<SoundArchiveContext, SoundSetBankBase>
    {
        public const string OutputID_Include = ComponentContext.IntermediateOutputPrefix + "Include";

        private SoundArchiveBuilder builder;

        public SoundSetBankProcessorFactory(SoundArchiveBuilder builder)
        {
            Ensure.Argument.NotNull(builder);
            this.builder = builder;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, SoundSetBankBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            // メインサウンドアーカイブのバンクを追加サウンドアーカイブで参照する場合はバンク情報のみを出力するので、スキップする
            if (context.IsExternalSoundArchiveBank(component))
            {
                return;
            }

            IOutputItem includeOutputItem = this.GetIncludeOutputItem(component);

            foreach (KeyValuePair<string, WaveArchiveBase> item in component.GetWaveArchiveDictionary())
            {
                IOutputItem outputItem = this.GetOutputItem(component, item.Key);
                this.AddProcessor(context, component, item.Key, item.Value, outputItem, includeOutputItem);

                // BankInclude ファイルは１回だけ生成します。
                includeOutputItem = null;
            }
        }

        private void AddProcessor(
            SoundArchiveContext context, SoundSetBankBase soundSetBank, string outputID,
            WaveArchiveBase waveArchive, IOutputItem outputItem, IOutputItem includeOutputItem)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(soundSetBank);
            Assertion.Argument.NotNull(outputID);
            Assertion.Argument.NotNull(outputItem);

            IOutput output = soundSetBank.GetOutputTarget();

            BankContext bankContext = new BankContext(
                new BankConversionTraits(context.Traits.IsLittleEndian, context.Traits.MaxFileName),
                context.GetWavePreprocessExeFullPath());

            BankService bankService = this.builder.BankServiceDictionary[soundSetBank.FilePath];

            BankBuilder builder = new BankBuilder(this.builder.FileManager);
            builder.Initialize(this.builder.SoundProjectService, bankService, waveArchive, soundSetBank.Name);
            builder.Build(bankContext, output, outputItem, includeOutputItem);

            foreach (OutputLine line in bankContext.Logger.Lines)
            {
                context.Logger.AddLine(this.ConvertLine(line, soundSetBank));
            }

            if (context.IsFailed)
            {
                return;
            }

            foreach (VelocityRegion velocityRegion in bankService.VelocityRegions)
            {
                //  Instrument が無効なら処理しません。
                if (!velocityRegion.Instrument.IsConvertTarget())
                {
                    continue;
                }

                IOutput waveOutput = velocityRegion.GetOutputTarget();
                if (context.ContainsProcessor(output, string.Empty)) { continue; }

                context.AddComponentProcessor(
                    waveOutput,
                    string.Empty,
                    bankContext.GetProcessor(waveOutput, string.Empty)
                    );

                context.GetProcessor(waveOutput, string.Empty).RelatedComponents.Add(soundSetBank);

                // VelocityRegion の波形ファイル登録
                context.AddWaveFile(velocityRegion, waveOutput);
            }

            BankProcessor bankProcessor = bankContext.GetProcessor(output, string.Empty) as BankProcessor;

            // 出力ファイルパスを設定するために、SoundSetBankBase を渡します。
            if (context.Settings.IsConvertParts)
            {
                bankProcessor.SoundSetBank = soundSetBank;
            }

            context.AddComponentProcessor(output, outputID, bankProcessor);
        }

        private OutputLine ConvertLine(OutputLine line, SoundSetBankBase soundSetBank)
        {
            Assertion.Argument.NotNull(line);
            Assertion.Argument.NotNull(soundSetBank);

            List<Component> components = new List<Component>();

            components.AddRange(line.Components);
            components.Add(soundSetBank);

            line.Components = components.ToArray();

            return line;
        }

        private IOutputItem GetOutputItem(SoundSetBankBase soundSetBank, string outputID)
        {
            Assertion.Argument.NotNull(soundSetBank);
            Assertion.Argument.NotNull(outputID);
            return soundSetBank.GetOutputTarget().ItemDictionary[outputID];
        }

        private IOutputItem GetIncludeOutputItem(SoundSetBankBase soundSetBank)
        {
            Assertion.Argument.NotNull(soundSetBank);
            return soundSetBank.GetOutputTarget().ItemDictionary[OutputID_Include];
        }
    }
}
