﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text.RegularExpressions;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Resources;
    using NintendoWare.ToolDevelopmentKit;

    internal class StreamSoundOutputFactory : ComponentSetup<SoundArchiveContext, StreamSoundBase>
    {
        private const string InvalidFileNamePattern = "[^!-~]";
        private const string streamDirectoryNameForPC = "_forPC";

        private readonly FileManager fileManager;

        public StreamSoundOutputFactory(FileManager fileManager)
        {
            Ensure.Argument.NotNull(fileManager);
            this.fileManager = fileManager;
        }

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, StreamSoundBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            IOutput outputTarget = this.fileManager.GetOutput(component);
            component.SetOutputTarget(outputTarget);

            if (component.GetAdtsHeader() != null)
            {
                this.AddOutputBinaryForPC(context, outputTarget, component);
            }

            string fileName = CreateFileName(component);

            string validExternalFilePath = Path.Combine(
                context.Project.ValidExternalFileOutputDirectoryPath,
                fileName
                ).Replace('\\', '/');

            if (validExternalFilePath.Length + 1 >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, validExternalFilePath),
                        component));
            }

            context.RegisteringOutputs.Add(
                new RegisteringOutputInfo(
                    component,
                    outputTarget,
                    context.Project.ExternalFileOutputDirectoryPath,
                    validExternalFilePath)
                    );

            // プリフェッチ
            if (component.IsPrefetchEnabled)
            {
                IOutput prefetchOutputTarget = this.fileManager.GetOutput(component, ".Prefetch");
                component.SetPrefetchOutputTarget(prefetchOutputTarget);

                this.AddPrefetchOutput(context, prefetchOutputTarget, component);
                context.AddFile(component, prefetchOutputTarget);
            }
            else
            {
                // プリフェッチが無効なのにグループ登録されている場合はエラー
                var attachedGroups = component.GetAttachedGroups();
                if (attachedGroups.Count > 0)
                {
                    context.Logger.AddLine(
                        new Logs.ErrorLine(MessageResource.Message_GroupMustNotContainsStreamSoundsThatHaveNoPrefetch, component));
                }
            }
        }

        private void AddPrefetchOutput(SoundArchiveContext context, IOutput outputTarget, StreamSoundBase streamSound)
        {
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(streamSound);

            string fileName = this.CreatePrefetchFileName(streamSound);

            if (fileName.Length >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, fileName),
                        streamSound));
            }

            // Include ファイルは複数の CacheItem から出力される可能性があるため、
            // isAutoNameCorrection を false にします。
            this.fileManager.RegisterCacheItem(
                outputTarget,
                string.Empty,
                fileName,
                true);
        }

        private void AddOutputBinaryForPC(SoundArchiveContext context, IOutput outputTarget, StreamSoundBase streamSound)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(outputTarget);
            Assertion.Argument.NotNull(streamSound);

            var filePath = Path.Combine(context.Project.ValidExternalFileOutputDirectoryPath, streamDirectoryNameForPC);
            filePath = Path.Combine(filePath, this.CreateFileNameForPC(streamSound));

            if (filePath.Length + 1 >= context.Traits.MaxFileName - 1)
            {
                context.Logger.AddLine(
                    new Logs.ErrorLine(
                        string.Format(MessageResource.Message_FilePathTooLong, filePath),
                        streamSound));
            }

            this.fileManager.RegisterOutputItem(
                outputTarget,
                StreamSoundProcessorFactory.OutputID_BinaryForPC,
                filePath);
        }

        /// <summary>
        /// ストリームサウンドバイナリのファイル名を作成します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreateFileName(StreamSoundBase streamSound)
        {
            Assertion.Argument.NotNull(streamSound);
            Ensure.Argument.StringNotEmpty(
                this.fileManager.BinaryOutputTraits.StreamSoundBinaryFileExtension);

            // シングルトラックストリームサウンドの場合、
            // 波形ファイル名から出力ファイル名を生成する
            // 同じ波形ファイルなら、ファイル共有される
            if (!streamSound.HasMultiTracks())
            {
                StreamSoundTrackBase track = streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;

                if (streamSound.Encoding == WaveEncoding.NoConvert)
                {
                    return Path.GetFileName(track.FilePath);
                }

                string baseFileName = Regex.Replace(
                    Path.GetFileNameWithoutExtension(track.FilePath),
                    InvalidFileNamePattern,
                    "_");

                if (baseFileName.Length == 0)
                {
                    throw new Exception("internal error : invalid file path.");
                }

                if (streamSound.IsResampleEnabled)
                {
                    return string.Format(
                        "{0}.{1}.{2}.{3}",
                        baseFileName,
                        this.GetEncoding(streamSound.Encoding),
                        streamSound.SampleRate,
                        this.fileManager.BinaryOutputTraits.StreamSoundBinaryFileExtension);
                }
                else
                {
                    return string.Format(
                        "{0}.{1}.{2}",
                        baseFileName,
                        this.GetEncoding(streamSound.Encoding),
                        this.fileManager.BinaryOutputTraits.StreamSoundBinaryFileExtension);
                }
            }

            // マルチトラックストリームサウンドの場合、
            // サウンド名を出力ファイル名にする
            // 同じ波形ファイルの組み合わせでも、ファイル共有されない
            string fileName = streamSound.Name;

            if (fileName.Length == 0)
            {
                throw new Exception("internal error : invalid stream sound name.");
            }

            switch (streamSound.Encoding)
            {
                case WaveEncoding.Adpcm:
                case WaveEncoding.Pcm16:
                case WaveEncoding.Pcm8:
                    return Path.ChangeExtension(
                        fileName,
                        this.fileManager.BinaryOutputTraits.StreamSoundBinaryFileExtension);

                case WaveEncoding.NoConvert:
                    return Path.ChangeExtension(
                        fileName,
                        Path.GetExtension((streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase).FilePath));
            }

            throw new Exception("internal error : invalid wave encoding.");
        }

        private string CreateFileNameForPC(StreamSoundBase streamSound)
        {
            Assertion.Argument.NotNull(streamSound);
            Ensure.Argument.StringNotEmpty(
                this.fileManager.BinaryOutputTraits.StreamSoundBinaryFileExtension);

            StreamSoundTrackBase track = streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;
            Assertion.Operation.True(streamSound.Encoding == WaveEncoding.NoConvert);

            // シングルトラックストリームサウンドの場合、波形ファイル名から出力ファイル名を生成する
            // マルチトラックストリームサウンドの場合、サウンド名を出力ファイル名にする
            string baseFileName = Regex.Replace(
                Path.GetFileNameWithoutExtension(track.FilePath),
                InvalidFileNamePattern,
                "_");

            if (streamSound.HasMultiTracks())
            {
                baseFileName = streamSound.Name;
            }

            if (baseFileName.Length == 0)
            {
                throw new Exception("internal error : invalid file path.");
            }

            // PC 用バイナリファイルは PCM16。
            return string.Format(
                "{0}.{1}.{2}",
                baseFileName,
                this.GetEncoding(WaveEncoding.Pcm16),
                this.fileManager.BinaryOutputTraits.StreamSoundBinaryFileExtension);
        }

        /// <summary>
        /// ストリームサウンドプリフェッチバイナリのファイル名を作成します。
        /// </summary>
        /// <param name="streamSound">ストリームサウンドを指定します。</param>
        /// <returns>ファイル名を返します。</returns>
        private string CreatePrefetchFileName(StreamSoundBase streamSound)
        {
            // シングルトラックストリームサウンドの場合、
            // 波形ファイル名から出力ファイル名を生成する
            // 同じ波形ファイルなら、ファイル共有される
            if (!streamSound.HasMultiTracks())
            {
                StreamSoundTrackBase track = streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase;

                if (streamSound.Encoding == WaveEncoding.NoConvert)
                {
                    return Path.GetFileName(track.FilePath);
                }

                string baseFileName = Regex.Replace(
                    Path.GetFileNameWithoutExtension(track.FilePath),
                    InvalidFileNamePattern,
                    "_");

                if (baseFileName.Length == 0)
                {
                    throw new Exception("internal error : invalid file path.");
                }

                return string.Format(
                    "{0}.{1}.{2}",
                    baseFileName,
                    this.GetEncoding(streamSound.Encoding),
                    this.fileManager.BinaryOutputTraits.StreamSoundPrefetchBinaryFileExtension);

            }

            // マルチトラックストリームサウンドの場合、
            // サウンド名を出力ファイル名にする
            // 同じ波形ファイルの組み合わせでも、ファイル共有されない
            string fileName = streamSound.Name;

            if (fileName.Length == 0)
            {
                throw new Exception("internal error : invalid stream sound name.");
            }

            switch (streamSound.Encoding)
            {
                case WaveEncoding.Adpcm:
                case WaveEncoding.Pcm16:
                case WaveEncoding.Pcm8:
                    return string.Format(
                        "{0}.{1}.{2}",
                        fileName,
                        this.GetEncoding(streamSound.Encoding),
                        this.fileManager.BinaryOutputTraits.StreamSoundPrefetchBinaryFileExtension);

                case WaveEncoding.NoConvert:
                    return Path.ChangeExtension(
                        fileName,
                        Path.GetExtension((streamSound.Children.FirstOrDefault<Component>(item => item.IsEnabled) as StreamSoundTrackBase).FilePath));
            }

            throw new Exception("internal error : invalid wave encoding.");
        }

        private string GetEncoding(WaveEncoding waveEncodig)
        {
            switch (waveEncodig)
            {
                case WaveEncoding.Adpcm:
                    return "dspadpcm";

                case WaveEncoding.Pcm16:
                    return "pcm16";

                case WaveEncoding.Pcm8:
                    return "pcm8";
            }

            throw new Exception("internal error : invalid wave encoding.");
        }
    }
}
