﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System.Collections.Generic;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Collections;
    using System;

    /// <summary>
    /// サウンドセットバンクに関連するアイテムを生成します。
    /// </summary>
    internal class SoundSetBankItemGenerator : ComponentSetup<SoundArchiveContext, SoundSetBankBase>
    {
        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, SoundSetBankBase component)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(component);

            Bank bank;

            if (!context.BankDictionary.TryGetValue(component, out bank))
            {
                // メインサウンドアーカイブのバンクを追加サウンドアーカイブで参照する場合はバンク情報のみを出力するので、スキップする
                // それ以外の場合はここにくることはないはずなので、例外を投げる
                if (context.AddonSoundSet != null)
                {
                    return;
                }
                else
                {
                    throw new InvalidOperationException("invalid bank.");
                }
            }

            // 部分コンバートの場合は、必ずオリジナル波形アーカイブを生成します。
            if (context.Settings.IsConvertParts)
            {
                this.GenerateOriginalWaveArchive(context, component, bank);
                return;
            }

            if (this.IsGenerateOriginalWaveArchive(component))
            {
                this.GenerateOriginalWaveArchive(context, component, bank);
            }

            if (component.WaveArchiveReference == WaveArchiveConsts.AutoShared)
            {
                this.DelegateGroupWaveArchives(component, bank);
            }
        }

        private bool DelegateGroupWaveArchives(SoundSetBankBase soundSetBank, Bank bank)
        {
            Assertion.Argument.NotNull(soundSetBank);
            Assertion.Argument.NotNull(bank);

            bool isDelegated = true;

            foreach (GroupBase group in soundSetBank.GetAttachedGroups())
            {
                switch (group.OutputType)
                {
                    case GroupOutputType.Embedding:
                    case GroupOutputType.UserManagement:
                        break;

                    default:
                        continue;
                }

                WaveArchiveBase groupWaveArchive = group.GetTargetWaveArchive();

                soundSetBank.GetWaveArchiveDictionary().Add(group.Name, groupWaveArchive);
                DelegateWaves(bank, groupWaveArchive);

                isDelegated = false;
            }

            return isDelegated;
        }

        /// <summary>
        /// 指定されたバンクに含まれる全てのベロシティリージョンを
        /// 指定された波形アーカイブに関連付けます。
        /// </summary>
        /// <param name="bank">バンクを指定します。</param>
        /// <param name="waveArchive">関連付ける先の波形アーカイブを指定します。</param>
        private void DelegateWaves(Bank bank, WaveArchiveBase waveArchive)
        {
            Assertion.Argument.NotNull(bank);
            Assertion.Argument.NotNull(waveArchive);

            WaveArchiveEx.ItemList waveArchiveItems = waveArchive.GetItems();

            foreach (VelocityRegion velocityRegion in bank.GetVelocityRegions())
            {
                if (!velocityRegion.IsHierarchicalConvertTarget())
                {
                    continue;
                }

                if (waveArchiveItems.Contains(velocityRegion))
                {
                    continue;
                }

                waveArchiveItems.Add(velocityRegion);
            }
        }

        private bool IsGenerateOriginalWaveArchive(SoundSetBankBase soundSetBank)
        {
            Assertion.Argument.NotNull(soundSetBank);

            switch (soundSetBank.WaveArchiveReference)
            {
                case WaveArchiveConsts.AutoIndividual:
                    // 自動（個別）の場合は、必ずオリジナル波形アーカイブを作ります。
                    return true;

                case WaveArchiveConsts.AutoShared:
                    // 自動（共有）の場合で、
                    // どのグループにも含まれない or １つ以上のリンクグループにふくまれる
                    // 場合は、オリジナル波形アーカイブを作ります。
                    ICollection<GroupBase> groups = soundSetBank.GetAttachedGroups();

                    if (groups.Count == 0)
                    {
                        return true;
                    }

                    foreach (GroupBase group in groups)
                    {
                        if (group.OutputType == GroupOutputType.Link)
                        {
                            return true;
                        }
                    }
                    return false;
            }

            // その他、手動共有の場合は、オリジナル波形アーカイブを作りません。
            return false;
        }

        /// <summary>
        /// サウンドセットバンク用のオリジナル波形アーカイブを生成します。
        /// </summary>
        /// <param name="context">サウンドアーカイブコンテキストを指定します。</param>
        /// <param name="soundSetBank">サウンドセットバンクを指定します。</param>
        /// <param name="bank">バンクを指定します。</param>
        private void GenerateOriginalWaveArchive(
            SoundArchiveContext context, SoundSetBankBase soundSetBank, Bank bank)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(soundSetBank);
            Assertion.Argument.NotNull(bank);

            WaveArchiveBase waveArchive = new WaveArchiveBase()
            {
                Name = string.Format(
                    "{0}_WaveArchive{1}",
                    soundSetBank.Name, SoundArchiveContext.AutoGeneratedNamePostfix),
            };

            soundSetBank.GetWaveArchiveDictionary().Add(string.Empty, waveArchive);
            DelegateWaves(bank, waveArchive);

            waveArchive.ID = new BinaryItemID(ItemType.WaveArchive, (uint)context.WaveArchives.Count);
            context.WaveArchives.Add(waveArchive);
        }
    }
}
