﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using NintendoWare.SoundFoundation.Binarization;
    using NintendoWare.SoundFoundation.Codecs;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Resources;
    using NintendoWare.ToolDevelopmentKit;
    using WaveFormats = NintendoWare.SoundFoundation.FileFormats.Wave;

    internal class WaveProcessor<TComponent> : ComponentProcessor<ComponentContext, TComponent>
        where TComponent : Component
    {
        private const int MaxChannel = 2;

        private readonly Func<TComponent, int?> getTargetSampleRateFunc;
        private readonly Func<TComponent, int?> getTargetChannelCountFunc;

        private List<string> warningMessages;

        private bool? hasOriginalLoop;
        private int? originalLoopStart;
        private int? originalLoopEnd;

        public WaveProcessor(TComponent component, IOutputItem outputItem, Func<TComponent, int?> getTargetSampleRateFunc, Func<TComponent, int?> getTargetChannelCountFunc)
            : base(component, outputItem)
        {
            Ensure.Argument.NotNull(getTargetSampleRateFunc);
            this.getTargetSampleRateFunc = getTargetSampleRateFunc;
            this.getTargetChannelCountFunc = getTargetChannelCountFunc;
            warningMessages = new List<string>();
        }

        public string PreprocessExePath { get; set; }

        protected override void ProcessInternal(ComponentContext context)
        {
            Ensure.Argument.NotNull(context);

            try
            {
                using (Stream stream = this.OutputTargetItem.OpenWrite())
                {
                    Write(context, context.CreateBinaryWriter(stream));
                }
            }
            catch
            {
                try
                {
                    if (File.Exists(this.OutputTargetItem.Path))
                    {
                        File.Delete(this.OutputTargetItem.Path);
                    }
                }
                catch
                {
                }

                throw;
            }
        }

        protected override void OutputLog(ComponentContext context, Component[] components)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(components);

            context.Logger.AddLine(new InformationLine(
                string.Format("[WAVE] {0} ({1}) > {2}",
                Path.GetFileName((components[0] as TComponent).GetFilePathForConvert()),
                this.GetEncoding(components[0] as TComponent).ToText(),
                Path.GetFileName(this.OutputTargetItem.Path)
                )));
        }

        private void Write(ComponentContext context, BinaryWriter writer)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(writer);

            var sourceFilePath = (this.TargetComponent as TComponent).GetFilePathForConvert();
            var targetFilePath = this.Preprocess(context, sourceFilePath);

            try
            {
                WaveEncoding encoding = GetEncoding(this.TargetComponent as TComponent);

                using (WaveFormats.WaveFileReader reader = WaveFormats.WaveFileReader.CreateInstance(targetFilePath))
                {
                    reader.LoopEndFrameModified += OnLoopEndFrameModified;

                    WaveFormats.WaveFile srcFile;

                    try
                    {
                        srcFile = reader.Open(targetFilePath);
                    }
                    catch (Exception e)
                    {
                        ErrorLine errorLine = new ErrorLine(e.Message, this.TargetComponent);
                        throw new ConversionException(errorLine);
                    }

                    if (srcFile.ChannelCount > MaxChannel)
                    {
                        ErrorLine errorLine = new ErrorLine(
                            string.Format(
                               Resources.MessageResource.Message_InvalidWaveChannels,
                               srcFile.ChannelCount,
                               MaxChannel,
                               sourceFilePath),
                           this.TargetComponent);
                        throw new ConversionException(errorLine);
                    }

                    WaveFormat waveFormat = new WaveFormat()
                    {
                        BitsPerSample = srcFile.SampleBit,
                        ChannelCount = srcFile.ChannelCount,
                        Encoding = Encoding.Pcm,
                        IsLittleEndian = srcFile is WaveFileWav,
                        IsSigned = !(srcFile is WaveFileWav && srcFile.SampleBit == 8),
                        SamplingRate = srcFile.SampleRate,
                        OriginalLoopStartFrame = this.originalLoopStart ?? (int)srcFile.LoopStartFrame,
                        OriginalLoopEndFrame = this.originalLoopEnd ?? (int)srcFile.LoopEndFrame,
                        LoopStartFrame = this.originalLoopStart ?? (int)srcFile.LoopStartFrame,
                        LoopEndFrame = this.originalLoopEnd ?? (int)srcFile.LoopEndFrame,
                        HasLoop = this.hasOriginalLoop ?? srcFile.IsLoop,
                    };

                    NintendoWareWaveEncoder encoder = this.CreateEncoder(context.Traits.IsLittleEndian);

                    using (var waveStream = new WaveStream(waveFormat, reader.OpenDataStream()))
                    {
                        CodecOutput[] codecOutputs = encoder.Run(waveStream);

                        // ループなしの場合は、全範囲をループ範囲とします。
                        foreach (CodecOutput codecOutput in codecOutputs)
                        {
                            if (codecOutput.Format.HasLoop)
                            {
                                continue;
                            }

                            codecOutput.Format = new WaveFormat(codecOutput.Format)
                            {
                                OriginalLoopStartFrame = 0,
                                OriginalLoopEndFrame = 0,
                                LoopStartFrame = 0,
                                LoopEndFrame = (int)srcFile.FrameCount,
                            };
                        }

                        WaveBinary file = new WaveFileBuilder(
                            context.Traits.BinaryFileInfo.WaveSignature,
                            context.Traits.BinaryFileInfo.WaveVersion).
                            Build(encoding, codecOutputs);

                        DomElement fileElement = new DomBuilder().Build(file);
                        new DomWriter(writer).Run(new DomObjectWriter(), fileElement);
                    }
                }
            }
            catch
            {
                if (sourceFilePath != targetFilePath && File.Exists(targetFilePath))
                {
                    File.Delete(targetFilePath);
                }
                throw;
            }

            foreach (string message in this.warningMessages)
            {
                context.Logger.AddLine(new WarningLine(message));
            }
            this.warningMessages.Clear();
        }

        private string Preprocess(ComponentContext context, string filePath)
        {
            var result = filePath;

            using (WaveFormats.WaveFileReader reader = WaveFormats.WaveFileReader.CreateInstance(filePath))
            {
                WaveFormats.WaveFile srcFile;

                try
                {
                    srcFile = reader.Open(filePath);
                }
                catch (Exception e)
                {
                    ErrorLine errorLine = new ErrorLine(e.Message, this.TargetComponent);
                    throw new ConversionException(errorLine);
                }

                var targetSampleRate = this.getTargetSampleRateFunc((TComponent)this.TargetComponent);
                var targetChannelCount = this.getTargetChannelCountFunc((TComponent)this.TargetComponent);

                // サンプルレートが指定されている場合
                if (targetSampleRate.HasValue == true)
                {
                    // ダウンサンプルになっていない（アップサンプルは禁止）
                    if (srcFile.SampleRate < targetSampleRate)
                    {
                        var message = string.Format(MessageResourceCommon.Message_UpsampleIsNotSupported, targetSampleRate, srcFile.SampleRate);
                        throw new ConversionException(new ErrorLine(message, this.TargetComponent));
                    }

                    // サンプルレートが範囲外
                    if (targetSampleRate < ConversionTraits.MinSampleRate || ConversionTraits.MaxSampleRate < targetSampleRate)
                    {
                        var message = string.Format(
                            MessageResourceCommon.Message_SampleRateMustBeValidRange,
                            ConversionTraits.MinSampleRate,
                            ConversionTraits.MaxSampleRate);
                        throw new ConversionException(new ErrorLine(message, this.TargetComponent));
                    }
                }

                // チャンネル数が指定されている場合（＝現時点ではモノラル化が指定されている場合）
                if (targetChannelCount.HasValue == true)
                {
                    // 現時点では Assert のチェックをするだけ。モノラル化のみ。
                    Debug.Assert(targetChannelCount == 1);
                }

                if (targetSampleRate.HasValue == true || targetChannelCount.HasValue == true)
                {
                    if (string.IsNullOrEmpty(this.PreprocessExePath))
                    {
                        throw new ConversionException(
                            new ErrorLine(MessageResourceCommon.Message_SoxExePathIsEmpty, this.TargetComponent));
                    }

                    if (!File.Exists(this.PreprocessExePath))
                    {
                        var message = string.Format(MessageResourceCommon.Message_SoxExeNotFound, this.PreprocessExePath);
                        throw new ConversionException(new ErrorLine(message, this.TargetComponent));
                    }

                    var sox = new SoxExecutor(this.PreprocessExePath)
                    {
                        InputPath = filePath,
                        OutputPath = this.GetTempFileName("wav"),
                        TargetComponents = new Component[] { this.TargetComponent },
                    };

                    if (targetSampleRate.HasValue == true)
                    {
                        sox.SampleRate = targetSampleRate.Value;
                    }

                    if (targetChannelCount.HasValue == true)
                    {
                        sox.ChannelCount = targetChannelCount.Value;
                    }

                    if (!sox.Run(context))
                    {
                        throw new ConversionException(
                            new ErrorLine(MessageResourceCommon.Message_FailedToPreprocessWaveFile, this.TargetComponent));
                    }

                    int sampleRate = srcFile.SampleRate;
                    if (targetSampleRate.HasValue == true)
                    {
                        sampleRate = targetSampleRate.Value;
                    }

                    this.originalLoopStart = (int)(srcFile.LoopStartFrame * sampleRate / srcFile.SampleRate);
                    this.originalLoopEnd = (int)(srcFile.LoopEndFrame * sampleRate / srcFile.SampleRate);
                    this.hasOriginalLoop = srcFile.IsLoop && this.originalLoopStart < this.originalLoopEnd;

                    result = sox.OutputPath;
                }
            }

            return result;
        }

        private string GetTempFileName(string extension)
        {
            while (true)
            {
                var filePath = Path.ChangeExtension(Path.GetTempFileName(), extension);

                if (!File.Exists(filePath))
                {
                    return filePath;
                }
            }
        }

        private NintendoWareWaveEncoder CreateEncoder(bool isLittleEndian)
        {
            var encoding = GetEncoding(this.TargetComponent as TComponent);

            switch (encoding)
            {
                case WaveEncoding.Adpcm:
                    // TODO : ★ImaAdpcmEncoder 対応。
                    return new NintendoWareDspAdpcmEncoder();

                case WaveEncoding.Pcm16:
                    return new NintendoWareLinearPcmEncoder()
                    {
                        BitsPerSample = 16,
                        IsLittleEndian = isLittleEndian,
                    };
            }

            throw new ConversionException(
                new ErrorLine(
                    string.Format(Resources.MessageResourceCommon.Message_InvalidWaveEncoding, encoding.ToString()),
                    this.TargetComponent));
        }

        private WaveEncoding GetEncoding(Component component)
        {
            return (WaveEncoding)component.Parameters[ProjectParameterNames.WaveEncoding].Value;
        }

        private void OnLoopEndFrameModified(object send, LoopEndFrameModifiedEventArgs e)
        {
            string format = string.Format(
                MessageResource.Message_LoopEndFrameModified,
                e.LoopEndFrame,
                e.FrameCount,
                e.FilePath);

            this.warningMessages.Add(format);
        }
    }
}
