﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Core.Parameters;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListCtrlCore : OperatableControl, IInplaceOwner
    {
        private HeaderCtrl _HeaderCtrl = null;

        private ExchangeOperator _ExchangeOperator = null;
        private SelectItemOperator _SelectItemOperator = null;
        private SelectSubItemOperator _SelectSubItemOperator = null;
        private MoveOperator _MoveOperator = null;

        private IListItem _CaretItem = null;
        private string _CaretColumnName = null;

        private bool _EnabledCaretMarker = false;
        private IListItem _CaretMarkerItem = null;
        private string _CaretMarkerName = null;

        private bool _EnabledCardinal = false;
        private IListItem _CardinalItem = null;
        private string _CardinalName = null;

        private bool _EnabledRubberBand = false;
        private RubberBandKinds _RubberBandKind = RubberBandKinds.None;
        private bool _RubberBandToggle = false;
        private int _RubberBandBeginIndex = -1;
        private int _RubberBandEndIndex = -1;
        private int _RubberBandBeginY = -1;
        private int _RubberBandEndY = -1;

        private bool _EnabledDragMarker = false;
        private IListItem _DragItem = null;
        private ListItemInPosition _DragItemInPosition = ListItemInPosition.None;

        private InplaceEditor _ActiveEditor = null;
        private IListItem _EditingItem = null;
        private string _EditingName = null;

        private Brush _TextBrush = null;
        private Brush _SelectedTextBrush = null;
        private Brush _ItemSelectedFtBrush = null;
        private Brush _ItemSelectedFfBrush = null;
        private Brush _SubItemSelectedFtBrush = null;
        private Brush _SubItemSelectedFfBrush = null;
        private Brush _ItemBackBrush = null;
        private Pen _ItemBorderPen = null;
        private int _RightEdgeClientX = -1;

        private bool _ReserveBeginEditByKey = false;
        private bool _ReserveBeginByEditBeginKey = false;
        private bool _ReserveImmediateEdit = false;
        private Keys _ReserveImmediateEditKey;
        private bool _ReservePressedShiftKey;
        private StringBuilder _ReserveKey = new StringBuilder();

        private Dictionary<string, ColumnInfo> _ColumnInfos = null;
        private ListItemDashLineDictionary _DashLineDictionary = null;

        // NOTE: ツールチップ・アイコンは動的に変更できないようなので、それぞれインスタンスを作ります。
        private readonly ToolTip _toolTipError = new ToolTip()
        {
            ToolTipIcon = ToolTipIcon.Error,
        };
        private readonly ToolTip _toolTipWarning = new ToolTip()
        {
            ToolTipIcon = ToolTipIcon.Warning,
        };
        private readonly ToolTip _toolTipNormal = new ToolTip();
        private ListCollisionResult _lastMouseListCollisionResult;

#if DEBUG
        private int _DrawCounter = 0;
#endif

        ///
        private enum RubberBandKinds
        {
            None,
            Item,
            SubItem,
        }

        public event EventHandler VerticalScrollValueChanged;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListCtrlCore()
        {
            CreateGraphicContext();

            _ExchangeOperator = new ExchangeOperator(this);
            _SelectItemOperator = new SelectItemOperator(this);
            _SelectSubItemOperator = new SelectSubItemOperator(this);
            _MoveOperator = new MoveOperator(this);
            CurrentOperator = _ExchangeOperator;

            HorizontalScrollBarSmallChange = 4;
            VerticalScrollBarSmallChange = 1;
            VerticalScrollBar.ValueChanged += OnVerticalScrollBarValueChanged;

            _ColumnInfos = new Dictionary<string, ColumnInfo>();
            UpdateColumnInfos();

            _DashLineDictionary = new ListItemDashLineDictionary();
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                _toolTipError.Dispose();
                _toolTipWarning.Dispose();
                _toolTipNormal.Dispose();
            }

            base.Dispose(disposing);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListCtrl OwnerCtrl { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public HeaderCtrl HeaderCtrl
        {
            get { return _HeaderCtrl; }
            set
            {
                if (_HeaderCtrl != null)
                {
                    _HeaderCtrl.OrderChanged -= OnHeaderChanged;
                    _HeaderCtrl.SortOrderChanged -= OnHeaderSortChanged;
                    _HeaderCtrl.PositionChanged -= OnHeaderPositionChanged;
                    _HeaderCtrl.ItemLengthChanged -= OnHeaderChanged;
                    _HeaderCtrl.ItemVisibleChanged -= OnHeaderChanged;
                    _HeaderCtrl.Resize -= OnHeaderResized;
                    _HeaderCtrl.VSplitDoubleClicked -= OnHeaderAdjustColumnWidth;
                    _HeaderCtrl.GotFocus -= OnHeaderGotFocus;
                }

                _HeaderCtrl = value;

                if (_HeaderCtrl != null)
                {
                    _HeaderCtrl.OrderChanged += OnHeaderChanged;
                    _HeaderCtrl.SortOrderChanged += OnHeaderSortChanged;
                    _HeaderCtrl.PositionChanged += OnHeaderPositionChanged;
                    _HeaderCtrl.ItemLengthChanged += OnHeaderChanged;
                    _HeaderCtrl.ItemVisibleChanged += OnHeaderChanged;
                    _HeaderCtrl.Resize += OnHeaderResized;
                    _HeaderCtrl.VSplitDoubleClicked += OnHeaderAdjustColumnWidth;
                    _HeaderCtrl.GotFocus += OnHeaderGotFocus;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IHeaderSource HeaderSource
        {
            get { return _HeaderCtrl.Source; }
            set
            {
                _HeaderCtrl.Source = value;
                UpdateColumnInfos();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ResizeHeaderHeight(int height)
        {
            HeaderCtrl.Bounds = new Rectangle(0, 0, HeaderCtrl.Width, height);
            Bounds = new Rectangle(0, height, OwnerCtrl.Width, OwnerCtrl.Height - height);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void InitializeCaret()
        {
            _CaretItem = null;
            _CaretColumnName = null;

            CaretMarker();
            _EnabledCardinal = false;
            _CardinalItem = null;
            _CardinalName = null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void GetCaret(out IListItem item, out string columnName)
        {
            item = _CaretItem;
            columnName = _CaretColumnName;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetCaret()
        {
            IListItemInfo[] infos = ItemInfos;

            if (infos.Length <= 0)
            {
                return;
            }

            MoveCaret(infos[0].Item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetCaret(IListItem item)
        {
            MoveCaret(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetCaret(IListItem item, string columnName)
        {
            Caret(item, columnName);
        }

        /// <summary>
        ///
        /// </summary>
        public void SetCaret(IListItem item, string columnName, bool enabledFollow)
        {
            Caret(item, columnName, enabledFollow);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Sort()
        {
            if (HeaderCtrl.Source == null ||
                HeaderCtrl.Source.SortItem == null)
            {
                return;
            }

            ListColumnStyle style = null;
            string name = null;
            SortOrder sortOrder;
            Type type = null;

            name = HeaderCtrl.Source.SortItem.Name;
            sortOrder = HeaderCtrl.Source.SortOrder;

            if ((style = GetColumnStyle(name)) == null)
            {
                return;
            }

            type = style.SubItemType;

            ItemsSource.Sort(name, type, sortOrder);

            /// ItemsSource.Sort()によってアイテムの選択が無くなるのでイベントを発生させます。
            OnSelectChanged();

            UpdateColumnInfos();
            UpdateScrollBar();
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        /// 表示情報が更新された時に呼ばれる
        /// </summary>
        public void UpdatedInfos()
        {
            //
            if (GetItemInfo(_CaretItem) == null)
            {
                _CaretItem = null;
                _CaretColumnName = null;
            }

            //
            UpdateColumnInfos();
            UpdateScrollBar();
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ClearDashLine()
        {
            _DashLineDictionary.Clear();
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetDashLineBySelectedSubItem()
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            ListItemSelectedState state = null;

            _DashLineDictionary.Clear();

            foreach (IListItem item in selectedItems.Keys)
            {

                state = selectedItems[item];
                foreach (string name in state.SubSelected)
                {
                    _DashLineDictionary.Mark(item, name);
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ReadOnly
        {
            get { return OwnerCtrl.ReadOnly; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool AllowDragItem
        {
            get { return OwnerCtrl.AllowDragItem; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ClickedOutsideUnselectItem
        {
            get { return OwnerCtrl.ClickedOutsideUnselectItem; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool FollowSelectedItemToCaret
        {
            get { return OwnerCtrl.FollowSelectedItemToCaret; }
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムを選択した時に、アイテムも選択されるのか？
        /// </summary>
        public bool InterlockSelectedItemAndSubItem
        {
            get { return OwnerCtrl.InterlockSelectedItemAndSubItem; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool StopCaretLeftAndRight
        {
            get { return OwnerCtrl.StopCaretLeftAndRight; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool MoveNewLineCaretLeftAndRight
        {
            get { return OwnerCtrl.MoveNewLineCaretLeftAndRight; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool MoveNextLineCaretAfterEndEditByEnter
        {
            get { return OwnerCtrl.MoveNextLineCaretAfterEndEditByEnter; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DecideFileDropPosition
        {
            get { return OwnerCtrl.DecideFileDropPosition; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool SuspendTransaction
        {
            get { return OwnerCtrl.SuspendTransaction; }
        }

        /// <summary>
        /// 編集中なのか調べる
        /// </summary>
        public bool IsEditing
        {
            get
            {
                return _ActiveEditor != null ? true : false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// スクロールバーの更新
        /// </summary>
        public override void UpdateScrollBar()
        {
            IListItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                HorizontalScrollBar.Visible = false;
                VerticalScrollBar.Visible = false;
                return;
            }

            //
            bool redraw = false;
            int itemCount = 0;
            int fullDisplayedItemCount = 0;
            int effectiveItemWidth = HorizontalWidth;
            int displayedItemWidth = DisplayedHorizontalWidth;

            itemCount = infos.Length;
            fullDisplayedItemCount = GetFullyDisplayedItemCount(infos, VerticalScrollBar.Value);

            if (displayedItemWidth < 0) { displayedItemWidth = 0; }

            HorizontalScrollBar.Visible =
                displayedItemWidth >= effectiveItemWidth ? false : true;
            VerticalScrollBar.Visible =
                fullDisplayedItemCount >= itemCount ? false : true;

            //
            base.UpdateScrollBar();

            //
            if (AdjustHorizontalScrollBarValue() != false)
            {
                redraw = true;
            }

            if (AdjustVerticalScrollBarValue() != false)
            {
                fullDisplayedItemCount = GetFullyDisplayedItemCount
                    (infos, VerticalScrollBar.Value);
                redraw = true;
            }

            //
            HorizontalScrollBar.Minimum = 0;
            HorizontalScrollBar.Maximum = effectiveItemWidth;
            HorizontalScrollBar.SmallChange = HorizontalScrollBarSmallChange;
            HorizontalScrollBar.LargeChange = displayedItemWidth;

            VerticalScrollBar.Minimum = 0;
            VerticalScrollBar.Maximum = itemCount == 0 ? 0 : itemCount - 1;
            VerticalScrollBar.SmallChange = VerticalScrollBarSmallChange;
            VerticalScrollBar.LargeChange = fullDisplayedItemCount;

            //
            if (redraw != false)
            {
                Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        /// ドラック中、スクロールバーの矢印を押した時の水平方向のスクロールの量の
        /// 取得/設定
        /// </summary>
        public int HorizontalScrollBarSmallChange { get; set; }

        ///--------------------------------
        /// <summary>
        /// ドラック中、スクロールバーの矢印を押した時の垂直方向のスクロールの量の
        /// 取得/設定
        /// </summary>
        public int VerticalScrollBarSmallChange { get; set; }

        /// <summary>
        /// 編集を終了します。
        /// </summary>
        public void EndEdit()
        {
            if (this._ActiveEditor == null)
            {
                return;
            }

            Debug.WriteLine("EndEdit");
            Trace.WriteLine("EndEdit");

            InplaceEditor editor = this._ActiveEditor;
            this._ActiveEditor = null;

            editor.EndEdit();
            this.BeginInvoke(new MethodInvoker(editor.Dispose));
            editor = null;

            CancelReserveBeginEditByKey();
            OnEditEnded();
        }

        /// <summary>
        /// 編集をキャンセルします。
        /// </summary>
        public bool CancelEdit()
        {
            if (this._ActiveEditor == null)
            {
                return false;
            }

            Debug.WriteLine("CancelEdit");
            Trace.WriteLine("CancelEdit");

            InplaceEditor editor = this._ActiveEditor;
            this._ActiveEditor = null;

            editor.CancelEdit();
            this.BeginInvoke(new MethodInvoker(editor.Dispose));
            editor = null;

            //
            CancelReserveBeginEditByKey();
            OnEditEnded();

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットの移動
        /// </summary>
        public bool MoveCaret(CaretMove direction)
        {
            switch (direction)
            {
                case CaretMove.Left: return MoveCaretPreviousSubItem(true);
                case CaretMove.Right: return MoveCaretNextSubItem(true);
                case CaretMove.Up: return MoveCaretPreviousItem();
                case CaretMove.Down: return MoveCaretNextItem();

                case CaretMove.LeftWithoutNewLine: return MoveCaretPreviousSubItem(false);
                case CaretMove.RightWithoutNewLine: return MoveCaretNextSubItem(false);

                case CaretMove.MostLeftInClient: return MoveCaretMostPreviousSubItemInClient();
                case CaretMove.MostRightInClient: return MoveCaretMostNextSubItemInClient();
                case CaretMove.MostUpInClient: return MoveCaretMostPreviousItemInClient();
                case CaretMove.MostDownInClient: return MoveCaretMostNextItemInClient();

                case CaretMove.MostLeft: return MoveCaretMostPreviousSubItem();
                case CaretMove.MostRight: return MoveCaretMostNextSubItem();
                case CaretMove.MostUp: return MoveCaretMostPreviousItem();
                case CaretMove.MostDown: return MoveCaretMostNextItem();

                case CaretMove.PageUp: return MoveCaretPreviousPageItem();
                case CaretMove.PageDown: return MoveCaretNextPageItem();
            }
            return false;
        }

        ///
        public void ShowItem(IListItem item)
        {
            MoveCaret(item);
        }

        ///--------------------------------
        /// <summary>
        /// クライアント領域に表示されているアイテムの取得
        /// </summary>
        public IEnumerable<IListItem> GetDisplayedItems()
        {
            return this.GetDisplayedItemInfos().Select(i => i.Item);
        }

        ///--------------------------------
        /// <summary>
        /// 基準アイテムをカレットで設定
        /// </summary>
        internal void SetCardinalItemByCaret()
        {
            SetCardinalItem(_CaretItem, _CaretColumnName);
        }

        ///--------------------------------
        /// <summary>
        /// オペレータの状態の変更
        /// </summary>
        protected void ChangeOperatorToExchange(Point beginPoint)
        {
            Debug.WriteLine("Operator changed to exchange");
            CurrentOperator = _ExchangeOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// オペレータの状態の変更
        /// </summary>
        protected void ChangeOperatorToSelectItem(Point beginPoint)
        {
            Debug.WriteLine("Operator changed to Select item");
            CurrentOperator = _SelectItemOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// オペレータの状態の変更
        /// </summary>
        protected void ChangeOperatorToSelectSubItem(Point beginPoint)
        {
            Debug.WriteLine("Operator changed to Select sub item");
            CurrentOperator = _SelectSubItemOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// オペレータの状態の変更
        /// </summary>
        protected void ChangeOperatorToMove(Point beginPoint)
        {
            Debug.WriteLine("Operator changed to Move");
            CurrentOperator = _MoveOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListItemsSource ItemsSource
        {
            get
            {
                if (OwnerCtrl != null)
                {
                    return OwnerCtrl.ItemsSource;
                }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListItemInfo[] ItemInfos
        {
            get
            {
                if (OwnerCtrl != null)
                {
                    return OwnerCtrl.ItemInfos;
                }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void DemandInvalidate()
        {
            OwnerCtrl.DemandInvalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListItemSelectedDictionary GetItemSelecteds()
        {
            return OwnerCtrl.GetItemSelecteds(this);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListItem[] GetSelectedItems()
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();

            List<IListItem> items = new List<IListItem>();
            foreach (IListItem item in ItemsSource.Items)
            {
                if (selectedItems.Keys.Contains(item) == true)
                {
                    items.Add(item);
                }
            }

            return items.ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// 直接指定キーが編集開始キーなのか調べる
        /// </summary>
        protected bool ContainsImmediateEditBeginKey(Keys key)
        {
            return OwnerCtrl.ContainsImmediateEditBeginKey(key);
        }

        ///--------------------------------
        /// <summary>
        /// 指定キーが編集開始キーなのか調べる
        /// </summary>
        protected bool ContainsEditBeginKey(Keys key)
        {
            return OwnerCtrl.ContainsEditBeginKey(key);
        }

        ///--------------------------------
        /// <summary>
        /// スタイルの取得
        /// </summary>
        protected ListColumnStyle GetColumnStyle(string name)
        {
            ListColumnStyle style = null;

            OwnerCtrl.ColumnStyles.TryGetValue(name, out style);
            return style;
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムの取得
        /// </summary>
        protected IListSubItem GetSubItem(string name)
        {
            IListSubItem subItem = null;

            OwnerCtrl.SubItems.TryGetValue(name, out subItem);
            return subItem;
        }

        ///--------------------------------
        /// <summary>
        /// エディタの取得
        /// </summary>
        protected InplaceEditor GetInplaceEditor(string name)
        {
            if (name == null)
            {
                return null;
            }

            InplaceEditor editor = null;

            OwnerCtrl.InplaceEditors.TryGetValue(name, out editor);
            return editor;
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムの取得
        /// </summary>
        protected IListSubItem GetSubItemByStyleName(string name)
        {
            ListColumnStyle style = null;
            IListSubItem subItem = null;

            //スタイルの取得
            if ((style = GetColumnStyle(name)) == null)
            {
                return null;
            }

            //サブアイテムの取得
            if ((subItem = GetSubItem(style.SubItemName)) == null)
            {
                return null;
            }

            return subItem;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected class ColumnInfo
        {
            public string Name { get; set; }
            public int ClientPosition { get; set; }
            public int Length { get; set; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ColumnInfo GetColumnInfo(string name)
        {
            if (_ColumnInfos.ContainsKey(name) == false)
            {
                return null;
            }
            return _ColumnInfos[name];
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void UpdateColumnInfos()
        {
            if (HeaderCtrl == null)
            {
                return;
            }

            List<string> list = new List<string>();
            ColumnInfo info = null;
            string[] names = null;
            string lastName = null;

            list.AddRange(HeaderCtrl.DisplayedRightFixedItemsName);
            list.AddRange(HeaderCtrl.DisplayedLeftFixedItemsName);

            //list.AddRange( HeaderCtrl.DisplayedFreeItemsName);
            names = HeaderCtrl.DisplayedFreeItemsName;
            list.AddRange(names);

            //
            _RightEdgeClientX = -1;
            if (names.Length > 0)
            {
                lastName = names[names.Length - 1];
            }

            //
            _ColumnInfos.Clear();
            foreach (string name in list)
            {

                info = new ColumnInfo();
                info.Name = name;
                info.ClientPosition = HeaderCtrl.GetPositionByName(name);
                info.Length = HeaderCtrl.GetLengthByName(name);

                _ColumnInfos.Add(name, info);

                //
                if (name == lastName)
                {
                    _RightEdgeClientX = info.ClientPosition + info.Length - 1;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ColumnCount
        {
            get { return _ColumnInfos.Count; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle ToClientRectangle(int x, int y, int width, int height)
        {
            return new Rectangle(x - ScrollLogicalX, y - ScrollLogicalY, width, height);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle ToClientRectangle(Rectangle rect)
        {
            rect.X -= ScrollLogicalX;
            rect.Y -= ScrollLogicalY;
            return rect;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ToClientY(int ly)
        {
            return ly - ScrollLogicalY;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ToClientHeight(int height)
        {
            return height;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ToLogicalX(int cx)
        {
            return cx + ScrollLogicalX;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ToLogicalY(int cy)
        {
            return cy + ScrollLogicalY;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Point ToLogicalPoint(Point point)
        {
            point.X += ScrollLogicalX;
            point.Y += ScrollLogicalY;
            return point;
        }

        ///--------------------------------
        /// <summary>
        /// 2点の座標から正規化された矩形を作成
        /// </summary>
        protected Rectangle NormalizedRectangle(Point begin, Point end)
        {
            Point lp = new Point();
            Point rp = new Point();

            lp.X = begin.X <= end.X ? begin.X : end.X;
            lp.Y = begin.Y <= end.Y ? begin.Y : end.Y;

            rp.X = begin.X >= end.X ? begin.X : end.X;
            rp.Y = begin.Y >= end.Y ? begin.Y : end.Y;

            return new Rectangle(lp.X, lp.Y, rp.X - lp.X, rp.Y - lp.Y);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void Normalize(ref int minValue, ref int maxValue)
        {
            int min = Math.Min(minValue, maxValue);
            int max = Math.Max(minValue, maxValue);
            minValue = min;
            maxValue = max;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ScrollLogicalX
        {
            get { return HorizontalScrollBar.Value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ScrollLogicalY
        {
            get
            {
                IListItemInfo[] infos = ItemInfos;
                int index = 0;

                if (infos == null) { return 0; }

                index = VerticalScrollBar.Value;
                if (index < 0 || index >= infos.Length)
                {
                    return 0;
                }

                return infos[index].Y;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int HorizontalWidth
        {
            get { return HeaderCtrl.TotalFreeItemsLength; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int DisplayedHorizontalWidth
        {
            get
            {
                return ClientRectangle.Width -
                    HeaderCtrl.TotalLeftFixedItemsLength -
                    HeaderCtrl.TotalRightFixedItemsLength;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 垂直方向のスクロールしてデータを表示する領域の幅の取得
        /// </summary>
        protected virtual int DisplayedClientHeight
        {
            get { return ClientRectangleWithoutScrollBar.Height; }
        }

        ///--------------------------------
        /// <summary>
        /// 横スクロールの位置を補正する
        /// 補正された場合には trueを返す
        /// </summary>
        protected bool AdjustHorizontalScrollBarValue()
        {
            int value = HorizontalScrollBar.Value;
            int min = HorizontalScrollBar.Minimum;
            int max = HorizontalScrollBar.Maximum;
            int clientWidth = DisplayedHorizontalWidth;
            //int       logicalWidth = max - min;
            int offset = 0;

            if (value + clientWidth > max)
            {
                offset = (value + clientWidth) - max;
                value -= offset;
            }

            if (HorizontalScrollBar.Value == value)
            {
                return false;
            }

            if (value < min || value > max)
            {
                value = 0;
            }

            HorizontalScrollBar.Value = value;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロールの位置を補正する
        /// 補正された場合には trueを返す
        /// </summary>
        protected bool AdjustVerticalScrollBarValue()
        {
            int value = GetAdjustVerticalScrollBarValue();

            if (VerticalScrollBar.Value == value)
            {
                return false;
            }

            VerticalScrollBar.Value = value;
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int GetAdjustVerticalScrollBarValue()
        {

            IListItemInfo[] infos = ItemInfos;
            int index = VerticalScrollBar.Value;
            int min = VerticalScrollBar.Minimum;
            int max = infos.Length;
            int height = DisplayedClientHeight;
            int h = 0;

            while (index > min)
            {
                h = 0;
                for (int loop = index - 1; loop < max; loop++)
                {
                    h += infos[loop].Item.Length;
                    if (h > height)
                    {
                        return index;
                    }
                }
                index--;
            }
            return index;
        }

        ///--------------------------------
        /// <summary>
        /// 指定ID番号のアイテムが見えるように垂直スクロールバーの位置の値を調整する
        /// </summary>
        protected void AdjustVerticalScrollBarByItem(IListItem item)
        {
            IListItemInfo[] infos = ItemInfos;
            int position = VerticalScrollBar.Value;
            int min = 0;
            int max = 0;
            int index = -1;

            if (infos == null)
            {
                return;
            }

            min = position;
            max = position + GetFullyDisplayedItemCount(infos, position);

            if ((index = GetIndexOnItem(item)) < 0)
            {
                return;
            }

            if (index == min - 1)
            {
                ScrollVerticalScrollBarRelative(-1);
                return;
            }

            if (index == max)
            {
                ScrollVerticalScrollBarRelative(1);
                return;
            }

            if (index == max + 1)
            {
                ScrollVerticalScrollBarRelative(2);
                return;
            }

            if (index < min - 1 || index > max)
            {
                ScrollVerticalScrollBarAbsolute(index);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int GetDisplayedItemCount(IListItemInfo[] infos, int beginIndex)
        {
            int height = ClientRectangleWithoutScrollBar.Height;
            int count = 0;

            for (int index = beginIndex; index < infos.Length; index++)
            {
                count++;

                height -= infos[index].Item.Length;
                if (height <= 0)
                {
                    break;
                }
            }
            return count;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int GetFullyDisplayedItemCount(IListItemInfo[] infos, int beginIndex)
        {
            int height = ClientRectangleWithoutScrollBar.Height;
            int count = 0;

            for (int index = beginIndex; index < infos.Length; index++)
            {
                if (height - infos[index].Item.Length < 0)
                {
                    break;
                }

                count++;

                height -= infos[index].Item.Length;
                if (height <= 0)
                {
                    break;
                }
            }
            return count;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void CreateGraphicContext()
        {
            Color itemSelectedFtColor = Color.FromArgb
                ((SystemColors.Highlight.R * 192 / 255) + (Color.White.R * (255 - 192) / 255),
                 (SystemColors.Highlight.G * 192 / 255) + (Color.White.G * (255 - 192) / 255),
                 (SystemColors.Highlight.B * 192 / 255) + (Color.White.B * (255 - 192) / 255));

            Color subItemSelectedFtColor = Color.FromArgb
                (SystemColors.Highlight.R * 7 / 8,
                  SystemColors.Highlight.G * 7 / 8,
                  SystemColors.Highlight.B * 7 / 8);

            Color itemSelectedFfColor = Color.FromArgb
                ((SystemColors.ControlDark.R * 160 / 255) + (Color.White.R * (255 - 160) / 255),
                 (SystemColors.ControlDark.G * 160 / 255) + (Color.White.G * (255 - 160) / 255),
                 (SystemColors.ControlDark.B * 160 / 255) + (Color.White.B * (255 - 160) / 255));

            // アイテムブラシ
            _ItemSelectedFtBrush = new SolidBrush(itemSelectedFtColor);
            _ItemSelectedFfBrush = new SolidBrush(itemSelectedFfColor);

            // サブアイテムブラシ
            _SubItemSelectedFtBrush = new SolidBrush(subItemSelectedFtColor);
            _SubItemSelectedFfBrush = SystemBrushes.ControlDark;

            //
            _TextBrush = new SolidBrush(SystemColors.WindowText);
            _SelectedTextBrush = new SolidBrush(SystemColors.HighlightText);

            //
            _ItemBackBrush = new SolidBrush(SystemColors.Window);
            _ItemBorderPen = new Pen(SystemColors.ControlDark);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanDropFiles(string[] filePaths, object dropTarget)
        {
            return OwnerCtrl.CanDropFiles(filePaths, (IListItem)dropTarget);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool DropFiles(string[] filePaths, object dropTarget)
        {
            return OwnerCtrl.DropFiles(filePaths, (IListItem)dropTarget);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool DropFiles(string[] filePaths, object dropTarget, DropPosition dropPosition)
        {
            return OwnerCtrl.DropFiles(filePaths, (IListItem)dropTarget, dropPosition);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void QueryDragOver(DragEventArgs e, object dropTarget, DropPosition dropPosition)
        {
            OwnerCtrl.QueryDragOver(e, (IListItem)dropTarget, dropPosition);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteDragDrop(object dropTarget, DropPosition dropPosition)
        {
            OwnerCtrl.ExecuteDragDrop((IListItem)dropTarget, dropPosition);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnHorizontalScrollValueChanged()
        {
            HeaderCtrl.Position = HorizontalScrollBar.Value;
            UpdateColumnInfos();

            if (IsEditing == true)
            {
                _ActiveEditor.EndEdit();
            }

            base.OnHorizontalScrollValueChanged();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnVerticalScrollValueChanged()
        {
            if (IsEditing == true)
            {
                _ActiveEditor.EndEdit();
            }

            base.OnVerticalScrollValueChanged();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnHorizontalScrolled(ScrollEventArgs e)
        {
            HeaderCtrl.Position = e.NewValue;
            UpdateColumnInfos();

            base.OnHorizontalScrolled(e);
        }

        ///--------------------------------
        /// <summary>
        /// カレットが設定できるのか調べる
        /// </summary>
        protected bool CanPutCaret(string name)
        {
            ListColumnStyle style = GetColumnStyle(name);
            return style != null ? style.CanPutCaret : false;
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムが選択できるのか調べる
        /// </summary>
        protected bool CanSelectSubItem(string name)
        {
            ListColumnStyle style = GetColumnStyle(name);
            return style != null ? style.CanSelect : false;
        }

        ///--------------------------------
        /// <summary>
        /// アイテム、サブアイテムの選択解除
        /// </summary>
        protected void UnselectItem()
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            selectedItems.Clear();
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの選択
        /// </summary>
        protected void SelectItem(IListItem item, bool b)
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            selectedItems.Select(item, b);
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの選択
        /// </summary>
        protected void SelectItems(IListItem[] items, bool b)
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();

            foreach (IListItem item in items)
            {
                selectedItems.Select(item, b);
            }
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの反転選択
        /// </summary>
        protected void ToggleSelectItem(IListItem item)
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            selectedItems.ToggleSelect(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SelectCardinalTo(IListItem item)
        {
            if (_EnabledCardinal == false)
            {
                return;
            }

            IListItemInfo beginInfo = GetItemInfo(_CardinalItem);
            IListItemInfo endInfo = GetItemInfo(item);
            int beginY = 0;
            int endY = 0;

            if (beginInfo == null || endInfo == null)
            {
                return;
            }

            beginY = beginInfo.Y;
            endY = endInfo.Y;
            Normalize(ref beginY, ref endY);

            SelectItem(beginY, endY, true);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SelectCardinalTo(IListItem item, string name)
        {
            SelectSubItemCardinalTo(item, name);
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの選択
        /// </summary>
        protected IListItem[] SelectItem(int beginLogY, int endLogY, bool b)
        {
            IListItem[] items = GetIntersectItems(beginLogY, endLogY);

            foreach (IListItem item in items)
            {
                SelectItem(item, b);
            }
            return items;
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの選択
        /// </summary>
        protected IListItem[] SelectItem(int logY, bool b)
        {
            return SelectItem(logY, logY, b);
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの反転選択
        /// </summary>
        protected IListItem[] ToggleSelectItem(int beginLogY, int endLogY)
        {
            IListItem[] items = GetIntersectItems(beginLogY, endLogY);

            foreach (IListItem item in items)
            {
                ToggleSelectItem(item);
            }
            return items;
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムの反転選択
        /// </summary>
        protected void ToggleSelectSubItem(IListItem item, string name)
        {
            if (CanSelectSubItem(name) == false)
            {
                return;
            }

            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            selectedItems.ToggleSubSelect(item, name);

            if (InterlockSelectedItemAndSubItem != false)
            {
                selectedItems.ToggleSelect(item);
            }
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムの選択
        /// </summary>
        protected void SelectSubItem(IListItem item, string name, bool b)
        {
            if (CanSelectSubItem(name) == false)
            {
                return;
            }

            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            selectedItems.SubSelect(item, name, b);

            if (InterlockSelectedItemAndSubItem != false)
            {
                selectedItems.Select(item, b);
            }
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムの反転選択
        /// </summary>
        protected IListItem[] ToggleSelectSubItem(int beginIndex, int endIndex, int beginLogY, int endLogY)
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            IListItem[] items = GetIntersectItems(beginLogY, endLogY);
            string name = null;

            foreach (IListItem item in items)
            {
                for (int index = beginIndex; index <= endIndex; index++)
                {
                    name = ToName(index);
                    selectedItems.ToggleSubSelect(item, name);
                }

                if (InterlockSelectedItemAndSubItem != false)
                {
                    selectedItems.ToggleSelect(item);
                }
            }

            return items;
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムの選択
        /// </summary>
        protected IListItem[] SelectSubItem(int beginIndex, int endIndex, int beginLogY, int endLogY, bool b)
        {
            IListItem[] items = GetIntersectItems(beginLogY, endLogY);
            string name = null;

            foreach (IListItem item in items)
            {
                for (int index = beginIndex; index <= endIndex; index++)
                {
                    name = ToName(index);
                    SelectSubItem(item, name, b);
                }
            }
            return items;
        }

        ///--------------------------------
        /// <summary>
        /// 基準アイテム〜指定アイテムの間のサブアイテムを選択
        /// </summary>
        protected IListItem[] SelectSubItemCardinalTo(IListItem item, string name)
        {
            if (_EnabledCardinal == false)
            {
                return new IListItem[0];
            }

            int beginIndex = ToIndex(_CardinalName);
            int endIndex = ToIndex(name);
            IListItemInfo beginInfo = GetItemInfo(_CardinalItem);
            IListItemInfo endInfo = GetItemInfo(item);
            int beginY = 0;
            int endY = 0;

            if (beginIndex < 0 || beginInfo == null ||
                endIndex < 0 || endInfo == null)
            {
                return new IListItem[0];
            }

            beginY = beginInfo.Y;
            endY = endInfo.Y;
            Normalize(ref beginY, ref endY);
            Normalize(ref beginIndex, ref endIndex);

            return SelectSubItem(beginIndex, endIndex, beginY, endY, true);
        }

        ///--------------------------------
        /// <summary>
        /// 基準アイテム〜カレットの間のサブアイテムを選択
        /// </summary>
        protected IListItem[] SelectSubItemCardinalToCaret()
        {
            return SelectSubItemCardinalTo(_CaretItem, _CaretColumnName);
        }

        ///--------------------------------
        /// <summary>
        /// 基準アイテムの設定
        /// </summary>
        protected void SetCardinalItem(IListItem item, string name)
        {
            if (item == null || name == null)
            {
                return;
            }

            _EnabledCardinal = true;
            _CardinalItem = item;
            _CardinalName = name;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected string DefaultCaretName
        {
            get
            {
                ListColumnStyle style = null;

                foreach (string name in HeaderCtrl.ItemsName)
                {

                    //スタイルの取得
                    if ((style = GetColumnStyle(name)) == null)
                    {
                        continue;
                    }

                    //カレットが移動しても大丈夫なのか？
                    if (style.CanPutCaret != false)
                    {
                        return name;
                    }
                }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool SameAsCaret(IListItem item, string name)
        {
            return _CaretItem == item && _CaretColumnName == name ? true : false;
        }

        ///--------------------------------
        /// <summary>
        /// カレットマーカーをカレットにする
        /// </summary>
        protected bool CaretMarkerTo(bool enabledFollow)
        {
            if (_EnabledCaretMarker == false)
            {
                return false;
            }

            _CaretItem = _CaretMarkerItem;
            _CaretColumnName = _CaretMarkerName;

            OnCaretMoved(true, enabledFollow);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// カレットの設定
        /// </summary>
        protected bool Caret(IListItem item)
        {
            return Caret(item, _CaretColumnName);
        }

        ///--------------------------------
        /// <summary>
        /// カレットの設定
        /// MoveCaretと違って、スクロール位置の変更を行わない
        /// </summary>
        protected bool Caret(IListItem item, string name)
        {
            return Caret(item, name, true);
        }

        ///--------------------------------
        /// <summary>
        /// カレットの設定
        /// MoveCaretと違って、スクロール位置の変更を行わない
        /// </summary>
        protected bool Caret(IListItem item, string name, bool enabledFollow)
        {
            if (item == null || name == null || CanPutCaret(name) == false)
            {
                return false;
            }

            _CaretItem = item;
            _CaretColumnName = name;

            OnCaretMoved(enabledFollow);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// カレットの設定
        /// </summary>
        protected bool MoveCaret(int index)
        {
            return MoveCaret(index, _CaretColumnName);
        }

        ///--------------------------------
        /// <summary>
        /// カレットの設定
        /// </summary>
        protected bool MoveCaret(IListItem item)
        {
            if (_CaretColumnName == null)
            {
                if ((_CaretColumnName = DefaultCaretName) == null)
                {
                    return false;
                }
            }

            return MoveCaret(item, _CaretColumnName);
        }

        ///--------------------------------
        /// <summary>
        /// カレットの設定
        /// </summary>
        protected bool MoveCaret(int index, string name)
        {
            IListItemInfo[] infos = ItemInfos;

            if (index < 0 || index >= infos.Length ||
                name == null)
            {
                return false;
            }

            return MoveCaret(infos[index].Item, name);
        }

        ///--------------------------------
        /// <summary>
        /// カレットの設定
        /// </summary>
        protected bool MoveCaret(IListItem item, string name)
        {
            if (item == null || CanPutCaret(name) == false)
            {
                return false;
            }

            _CaretItem = item;
            _CaretColumnName = name;

            AdjustVerticalScrollBarByItem(_CaretItem);
            OnCaretMoved(true);
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnCaretMoved(bool moved)
        {
            OnCaretMoved(moved, true);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnCaretMoved(bool moved, bool enabledFollow)
        {
            OnCaretMoved(new ListCaretMovedEventArgs
                          (_CaretItem, _CaretColumnName, moved));

            //移動が失敗したのか？
            if (moved == false)
            {
                return;
            }

            //
            if (enabledFollow != false)
            {
                ///選択がカレットに追従するのか？
                if (OwnerCtrl.FollowSelectedItemToCaret != false)
                {
                    UnselectItem();
                    SelectSubItem(_CaretItem, _CaretColumnName, true);
                    OnSelectChanged();
                    OwnerCtrl.DemandInvalidate();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// キャレットが移動した時に呼ばれる
        /// </summary>
        protected virtual void OnCaretMoved(ListCaretMovedEventArgs e)
        {
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムとの衝突判定
        /// </summary>
        protected ListCollisionResult CollisionToSubItem(ListDrawDescriptor desc, IListItemInfo info, int lx, int ly, int cx, int cy, string name, int columnX, int columnWidth)
        {
            IListItem item = info.Item;
            IListSubItem subItem = null;
            ListColumnStyle style = null;
            Rectangle bounds = new Rectangle();

            //スタイルの取得
            if ((style = GetColumnStyle(name)) == null)
            {
                return new ListCollisionResult();
            }

            //サブアイテムの取得
            if ((subItem = GetSubItem(style.SubItemName)) == null)
            {
                return new ListCollisionResult();
            }

            //
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            bool selected = selectedItems.Selected(item);

            int x = cx - columnX;
            int y = ly - info.Y;

            bounds.X = 0;
            bounds.Y = 0;
            bounds.Width = columnWidth;
            bounds.Height = item.Length;

            desc.Bounds = bounds;
            desc.Selected = selected;
            desc.Parameter = item.GetConstValue(name);

            //パーツとの衝突判定
            foreach (IListPart part in subItem.GetParts(desc))
            {

                //Emptyの場合は、desc.Boundsと同じサイズなので Containsが常に trueになる
                if (part.Bounds == Rectangle.Empty)
                {
                    return new ListCollisionResult
                        (info, part, name, selected);
                }

                if (part.Bounds.Contains(x, y) != false)
                {
                    return new ListCollisionResult
                        (info, part, name, selected);
                }
            }

            return new ListCollisionResult();
        }

        ///--------------------------------
        /// <summary>
        /// アイテムとの衝突判定
        /// </summary>
        protected ListCollisionResult CollisionToItem(ListDrawDescriptor desc, IListItemInfo info, int lx, int ly, int cx, int cy)
        {
            int x = 0;
            int width = 0;

            foreach (ColumnInfo columnInfo in _ColumnInfos.Values)
            {
                x = columnInfo.ClientPosition;
                width = columnInfo.Length;

                if (cx >= x && cx < x + width)
                {
                    return CollisionToSubItem(desc, info, lx, ly, cx, cy,
                                               columnInfo.Name, x, width);
                }
            }

            return new ListCollisionResult();
        }

        ///--------------------------------
        /// <summary>
        /// 衝突判定
        /// </summary>
        protected ListCollisionResult Collision(int cx, int cy)
        {
            Graphics gc = null;
            ListCollisionResult result = null;
            ListDrawDescriptor desc = null;
            IListItemInfo[] infos = GetDisplayedItemInfos();
            IListItem item = null;
            int lx = 0;
            int ly = 0;

            if (infos.Length <= 0)
            {
                return new ListCollisionResult();
            }

            lx = ToLogicalX(cx);
            ly = ToLogicalY(cy);

            try
            {
                gc = CreateGraphics();
                desc = new ListDrawDescriptor(gc, Font, OwnerCtrl.Images);

                foreach (IListItemInfo info in infos)
                {
                    item = info.Item;

                    if (ly >= info.Y && ly < info.Y + item.Length)
                    {
                        if ((result = CollisionToItem(desc, info, lx, ly, cx, cy)) != null)
                        {
                            return result;
                        }
                    }
                }
            }

            finally { gc.Dispose(); }
            return new ListCollisionResult();
        }

        ///--------------------------------
        /// <summary>
        /// ドラック処理用の衝突判定
        /// 座標はクライアント座標を指定する
        /// </summary>
        protected ListCollisionResult CollisionForDrag(int cx, int cy)
        {
            IListItemInfo[] infos = GetDisplayedItemInfos();
            IListItem item = null;
            ListItemInPosition inPosition = ListItemInPosition.None;
            int lx = ToLogicalX(cx);
            int ly = ToLogicalY(cy);
            int cenY = 0;
            int uppY = 0;
            int lowY = 0;
            int y = 0;
            int height = 0;

            foreach (IListItemInfo info in infos)
            {

                item = info.Item;
                y = info.Y;
                height = item.Length;

                if (ly >= y && ly < y + height)
                {
                    cenY = y + (height / 2);
                    uppY = y + (height / 3) * 1;
                    lowY = y + (height / 3) * 2;

                    if (OwnerCtrl.DropOnDragItem != false)
                    {
                        inPosition = ListItemInPosition.Center;
                        if (ly < uppY) { inPosition = ListItemInPosition.Upper; }
                        if (ly > lowY) { inPosition = ListItemInPosition.Lower; }
                    }
                    else
                    {

                        inPosition = ly >= cenY ?
                            ListItemInPosition.Lower : ListItemInPosition.Upper;
                    }

                    return new ListCollisionResult(info, null, inPosition);
                }
            }

            return new ListCollisionResult();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected int ToIndex(string name)
        {
            if (name == null)
            {
                return -1;
            }

            return HeaderCtrl.GetIndexByName(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected string ToName(int index)
        {
            return HeaderCtrl.GetNameByIndex(index);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected string GetColumnNameByClientX(int cx)
        {
            int x = 0;
            int width = 0;

            foreach (ColumnInfo columnInfo in _ColumnInfos.Values)
            {
                x = columnInfo.ClientPosition;
                width = columnInfo.Length;

                if (cx >= x && cx < x + width)
                {
                    return columnInfo.Name;
                }
            }

            //
            string[] names = HeaderCtrl.ItemsName;

            if (cx < 0)
            {
                return names[0];
            }

            if (cx >= ClientRectangleWithoutScrollBar.Width)
            {
                return names[names.Length - 1];
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        /// クライアント領域に表示されているアイテムの取得
        /// </summary>
        protected IListItemInfo[] GetDisplayedItemInfos()
        {
            List<IListItemInfo> list = new List<IListItemInfo>();
            IListItemInfo[] infos = ItemInfos;
            int beginIndex = 0;
            int endIndex = 0;

            if (infos != null)
            {
                beginIndex = VerticalScrollBar.Value;
                endIndex = beginIndex + GetDisplayedItemCount(infos, beginIndex) - 1;

                for (int index = beginIndex; index <= endIndex; index++)
                {
                    list.Add(infos[index]);
                }
            }
            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool IntersectBounds(Rectangle bounds0, Rectangle bounds1, bool ignoreWidth)
        {
            if (ignoreWidth == false)
            {
                if (bounds0.Right < bounds1.Left ||
                    bounds0.Left > bounds1.Right)
                {
                    return false;
                }
            }

            if (bounds0.Bottom < bounds1.Top ||
                bounds0.Top >= bounds1.Bottom)
            {
                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 2点で指定される矩形範囲に掛かるアイテムの取得
        /// </summary>
        protected IListItem[] GetIntersectItems(int beginLogY, int endLogY)
        {
            List<IListItem> list = new List<IListItem>();
            IListItemInfo[] infos = ItemInfos;
            IListItem item = null;
            int by = 0;
            int ey = 0;
            //int                 height = 0;

            if (infos != null)
            {
                foreach (IListItemInfo info in infos)
                {
                    item = info.Item;

                    by = info.Y;
                    ey = by + item.Length - 1;
                    //height = item.Length;

                    //座標が選択範囲の始点座標より小さいのか？
                    //その場合には衝突判定をパスする
                    if (ey < beginLogY)
                    {
                        continue;
                    }

                    //座標が選択範囲の終点座標より大きいのか？
                    //その場合には衝突判定をパスする
                    if (by > endLogY)
                    {
                        break;
                    }

                    list.Add(item);
                }
            }

            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// ドラック中の自動スクロールを開始する境界の取得
        /// 引数はクライアント
        /// </summary>
        protected Rectangle DragScrollBorderRectangle
        {
            get
            {
                int wm = 48;
                int hm = 48;

                return new Rectangle(wm, hm, Width - wm * 2, Height - hm * 2);
            }
        }

        ///--------------------------------
        /// <summary>
        /// 横スクロールバーの相対スクロール
        /// </summary>
        protected bool ScrollHorizontalScrollBarRelative(int value)
        {
            int index = HorizontalScrollBar.Value;
            int count = DisplayedHorizontalWidth;
            int min = HorizontalScrollBar.Minimum;
            int max = HorizontalScrollBar.Maximum - count;

            if (ItemInfos == null)
            {
                return false;
            }

            if (max < 0)
            {
                return false;
            }

            if (value < 0)
            {
                if (HorizontalScrollBar.Value + value < min)
                {
                    value = min - HorizontalScrollBar.Value;
                }
            }

            if (value > 0)
            {
                if (HorizontalScrollBar.Value + value >= max)
                {
                    value = max - HorizontalScrollBar.Value + 1;
                }
            }

            if (value == 0)
            {
                return false;
            }

            HorizontalScrollBar.Value += value;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロールバーの相対スクロール
        /// </summary>
        protected bool ScrollVerticalScrollBarRelative(int value)
        {
            IListItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return false;
            }

            int index = VerticalScrollBar.Value;
            int count = GetFullyDisplayedItemCount(infos, index);
            int min = VerticalScrollBar.Minimum;
            int max = VerticalScrollBar.Maximum - count + 1;

            if (max < 0)
            {
                return false;
            }

            if (value < 0)
            {
                if (VerticalScrollBar.Value + value < min)
                {
                    value = min - VerticalScrollBar.Value;
                }
            }

            if (value > 0)
            {
                if (VerticalScrollBar.Value + value >= max)
                {
                    value = (max + 0) - VerticalScrollBar.Value;
                }
            }

            if (value == 0)
            {
                return false;
            }

            if (VerticalScrollBar.Value + value > VerticalScrollBar.Maximum)
            {
                VerticalScrollBar.Value = VerticalScrollBar.Maximum;
                return true;
            }

            VerticalScrollBar.Value += value;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロールバーの相対スクロール(ページスクロール)
        /// </summary>
        protected bool ScrollPageVerticalScrollBarRelative(int value)
        {
            IListItemInfo[] infos = ItemInfos;
            int relativeValue = 0;
            int index = VerticalScrollBar.Value;

            if (infos == null || value == 0 || _CaretItem == null)
            {
                return false;
            }

            relativeValue = GetFullyDisplayedItemCount(infos, index);

            if (value < 0)
            {
                relativeValue = -relativeValue;
            }

            ///
            IListItemInfo info = null;
            int offset = 0;
            bool result = false;

            if ((info = GetItemInfo(_CaretItem)) == null)
            {
                return false;
            }

            offset = GetIndexOnItem(info.Item) - index;

            result = ScrollVerticalScrollBarRelative(relativeValue);
            index = VerticalScrollBar.Value + offset;

            if (index < 0) { index = 0; }
            if (index > infos.Length - 1) { index = infos.Length - 1; }

            MoveCaret(index);
            return result;
        }

        ///--------------------------------
        /// <summary>
        /// 縦スクロールバーの絶対スクロール
        /// </summary>
        protected bool ScrollVerticalScrollBarAbsolute(int index)
        {
            IListItemInfo[] infos = ItemInfos;
            int min = 0;
            int max = 0;

            if (infos == null)
            {
                return false;
            }

            min = VerticalScrollBar.Minimum;
            max = VerticalScrollBar.Maximum -
                GetFullyDisplayedItemCount(infos, VerticalScrollBar.Value) + 1;

            if (index < min) { index = min; }
            if (index >= max) { index = max; }

            VerticalScrollBar.Value = index;
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ドラック時のスクロール
        /// </summary>
        protected void DragScroll(int cx, int cy)
        {
            Rectangle rect = DragScrollBorderRectangle;
            bool redraw = false;

            if (cx < rect.Left)
            {
                redraw = ScrollHorizontalScrollBarRelative(-HorizontalScrollBarSmallChange);
                //redraw = true;
            }

            if (cx > rect.Right)
            {
                redraw = ScrollHorizontalScrollBarRelative(HorizontalScrollBarSmallChange);
                //redraw = true;
            }

            if (cy < rect.Top)
            {
                redraw = ScrollVerticalScrollBarRelative(-VerticalScrollBarSmallChange);
                //redraw = true;
            }

            if (cy > rect.Bottom)
            {
                redraw = ScrollVerticalScrollBarRelative(VerticalScrollBarSmallChange);
                //redraw = true;
            }

            if (redraw != false)
            {
                Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        /// 正規化されたラバーバンドの取得
        /// 論理座標を返す
        /// </summary>
        protected Rectangle GetNormalizedRubberBand()
        {
            switch (_RubberBandKind)
            {
                case RubberBandKinds.Item:
                    return NormalizedRectangle(new Point(0, _RubberBandBeginY),
                                                new Point(0, _RubberBandEndY));

                case RubberBandKinds.SubItem:
                    return NormalizedRectangle(new Point(_RubberBandBeginIndex, _RubberBandBeginY),
                                                new Point(_RubberBandEndIndex, _RubberBandEndY));
            }
            return Rectangle.Empty;
        }

        ///--------------------------------
        /// <summary>
        /// ラバーバンドの設定
        /// </summary>
        protected void RubberBand(int beginLogY, int endLogY, bool toggle)
        {
            _EnabledRubberBand = true;
            _RubberBandToggle = toggle;
            _RubberBandBeginY = beginLogY;
            _RubberBandEndY = endLogY;

            _RubberBandKind = RubberBandKinds.Item;
        }

        ///--------------------------------
        /// <summary>
        /// ラバーバンドの設定
        /// </summary>
        protected void RubberBand(int beginIndex, int endIndex, int beginY, int endY, bool toggle)
        {
            _EnabledRubberBand = true;
            _RubberBandToggle = toggle;
            _RubberBandBeginIndex = beginIndex;
            _RubberBandEndIndex = endIndex;
            _RubberBandBeginY = beginY;
            _RubberBandEndY = endY;

            _RubberBandKind = RubberBandKinds.SubItem;
        }

        ///--------------------------------
        /// <summary>
        /// ラバーバンドの抹消
        /// </summary>
        protected void RubberBand()
        {
            _EnabledRubberBand = false;
        }

        ///--------------------------------
        /// <summary>
        /// ドラック用マーカーの設定
        /// </summary>
        protected void DragMarker(IListItem item, ListItemInPosition inPosition)
        {
            _EnabledDragMarker = true;
            _DragItem = item;
            _DragItemInPosition = inPosition;
        }

        ///--------------------------------
        /// <summary>
        /// ドラック用マーカーの消去
        /// </summary>
        protected void DragMarker()
        {
            _EnabledDragMarker = false;
            _DragItem = null;
            _DragItemInPosition = ListItemInPosition.None;
        }

        ///--------------------------------
        /// <summary>
        /// カレットマーカーの設定
        /// </summary>
        protected void CaretMarker(IListItem item)
        {
            string name = null;

            name = _CaretColumnName != null ? _CaretColumnName : DefaultCaretName;
            if (name == null)
            {
                return;
            }

            _EnabledCaretMarker = true;
            _CaretMarkerItem = item;
            _CaretMarkerName = name;
        }

        ///--------------------------------
        /// <summary>
        /// カレットマーカーの設定
        /// </summary>
        protected void CaretMarker(IListItem item, string name)
        {
            _EnabledCaretMarker = true;
            _CaretMarkerItem = item;
            _CaretMarkerName = name;
        }

        ///--------------------------------
        /// <summary>
        /// カレットマーカーの消去
        /// </summary>
        protected void CaretMarker()
        {
            _EnabledCaretMarker = false;
            _CaretMarkerItem = null;
            _CaretMarkerName = null;
        }

        ///--------------------------------
        /// <summary>
        /// 表示情報の配列のインデックスを取得
        /// </summary>
        protected int GetIndexOnItem(IListItem item)
        {
            IListItemInfo[] infos = ItemInfos;
            int index = 0;

            if (infos == null || item == null)
            {
                return -1;
            }

            foreach (IListItemInfo info in infos)
            {
                if (info.Item == item)
                {
                    return index;
                }
                index++;
            }
            return index;
        }

        ///--------------------------------
        /// <summary>
        /// カレットのあるサブアイテムを表示するクライアント矩形の取得
        /// </summary>
        protected virtual Rectangle GetCaretItemClientBounds()
        {
            if (_CaretItem == null || _CaretColumnName == null)
            {
                return Rectangle.Empty;
            }

            Rectangle bounds = GetSubItemBounds(_CaretItem, _CaretColumnName);
            //return ToClientRectangle( bounds);
            return bounds;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットのある表示情報の配列のインデックスを取得
        /// </summary>
        protected int IndexOnCaret
        {
            get { return GetIndexOnItem(_CaretItem); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool MoveCaretSubItemRelative(int direction, bool newLine)
        {
            int index = -1;

            if ((index = ToIndex(_CaretColumnName)) < 0)
            {
                return false;
            }

            ListColumnStyle style = null;
            string[] names = HeaderCtrl.ItemsName;
            int min = 0;
            int max = names.Length - 1;
            bool movePreviousItem = false;
            bool moveNextItem = false;
            string beforeName = names[index];
            string name = null;
            int offset = direction < 0 ? -1 : 1;
            int beforeIndex = -1;

            if (StopCaretLeftAndRight != false)
            {
                newLine = false;
            }

            //
            while (true)
            {
                index += offset;

                if (newLine != false)
                {
                    if (index < min) { index = max; movePreviousItem = true; }
                    if (index > max) { index = min; moveNextItem = true; }
                }
                else
                {

                    if (index < min) { index = min; }
                    if (index > max) { index = max; }
                }

                //無効なカラムをスキップ
                if (HeaderCtrl.IsVisibleItemByIndex(index) == false)
                {
                    if (index != min && index != max)
                    {
                        continue;
                    }
                    else //両端が無効な場合は何もしない。
                    {
                        return false;
                    }
                }

                name = names[index];

                //スタイルの取得
                if ((style = GetColumnStyle(name)) == null)
                {
                    return false;
                }

                //カレットが移動しても大丈夫なのか？
                if (style.CanPutCaret != false)
                {
                    break;
                }

                //繰り返しになっているのか？
                if (beforeIndex == index)
                {
                    return false;
                }
                beforeIndex = index;
            }

            _CaretColumnName = name;

            if (MoveNewLineCaretLeftAndRight != false)
            {
                if (movePreviousItem != false)
                {
                    //OnCaretMovedは MoveCaretPreviousItem()で呼ばれるので、ここでは呼ばない
                    if (MoveCaretPreviousItem() != false)
                    {
                        HeaderCtrl.FitDisplay(_CaretColumnName);
                        return true;
                    }

                    _CaretColumnName = beforeName;
                    return true;
                }

                if (moveNextItem != false)
                {
                    //OnCaretMovedは MoveCaretNextItem()で呼ばれるので、ここでは呼ばない
                    if (MoveCaretNextItem() != false)
                    {
                        HeaderCtrl.FitDisplay(_CaretColumnName);
                        return true;
                    }

                    _CaretColumnName = beforeName;
                    return true;
                }
            }

            HeaderCtrl.FitDisplay(_CaretColumnName);
            OnCaretMoved(true);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一つ前のサブアイテムに移動させる
        /// </summary>
        protected bool MoveCaretPreviousSubItem(bool newLine)
        {
            return MoveCaretSubItemRelative(-1, newLine);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一つ後のサブアイテムに移動させる
        /// </summary>
        protected bool MoveCaretNextSubItem(bool newLine)
        {
            return MoveCaretSubItemRelative(1, newLine);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一つ前のアイテムに移動させる
        /// </summary>
        protected bool MoveCaretPreviousItem()
        {
            IListItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return false;
            }

            int index = IndexOnCaret;
            int limit = 0;

            if (index <= limit)
            {
                OnCaretMoved(false);
                return false;
            }

            MoveCaret(infos[index - 1].Item);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一つ後のアイテムに移動させる
        /// </summary>
        protected bool MoveCaretNextItem()
        {
            IListItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return false;
            }

            int index = IndexOnCaret;
            int limit = infos.Length - 1;

            if (index < 0 || index >= limit)
            {
                OnCaretMoved(false);
                return false;
            }

            MoveCaret(infos[index + 1].Item);
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool MoveCaretPreviousPageItem()
        {
            if (ScrollPageVerticalScrollBarRelative(-1) == false)
            {
                return MoveCaretMostPreviousItem();
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool MoveCaretNextPageItem()
        {
            if (ScrollPageVerticalScrollBarRelative(1) == false)
            {
                return MoveCaretMostNextItem();
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一番前のアイテムに移動
        /// </summary>
        protected bool MoveCaretMostPreviousItem()
        {
            return MoveCaret(0);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットを一番後のアイテムに移動
        /// </summary>
        protected bool MoveCaretMostNextItem()
        {
            IListItemInfo[] infos = ItemInfos;
            if (infos == null) { return false; }
            return MoveCaret(infos.Length - 1);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットをクライアント領域の一番前のアイテムに移動
        /// </summary>
        protected bool MoveCaretMostPreviousItemInClient()
        {
            IListItemInfo info = null;
            int index = VerticalScrollBar.Value;

            if ((info = GetItemInfo(index)) == null)
            {
                return false;
            }

            if (info.Item == _CaretItem)
            {
                return ScrollPageVerticalScrollBarRelative(-1);
            }

            return Caret(info.Item);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットをクライアント領域の一番後のアイテムに移動
        /// </summary>
        protected bool MoveCaretMostNextItemInClient()
        {
            IListItemInfo[] infos = ItemInfos;
            IListItemInfo info = null;
            int index = VerticalScrollBar.Value;
            int count = 0;

            if (infos == null)
            {
                return false;
            }

            count = GetFullyDisplayedItemCount(infos, index) - 1;
            if ((index += count) >= infos.Length)
            {
                index = infos.Length - 1;
            }

            if ((info = GetItemInfo(index)) == null)
            {
                return false;
            }

            if (info.Item == _CaretItem)
            {
                return ScrollPageVerticalScrollBarRelative(1);
            }

            return Caret(info.Item);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットをサブアイテムに移動
        /// </summary>
        protected bool MoveCaretSubItem(string[] names, bool first)
        {
            int index = -1;

            if (names == null)
            {
                return false;
            }

            index = first != false ? 0 : names.Length - 1;
            if (MoveCaret(_CaretItem, names[index]) == false)
            {
                return false;
            }

            HeaderCtrl.FitDisplay(_CaretColumnName);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// キャレットをサブアイテムの一番前に移動させる
        /// </summary>
        protected bool MoveCaretMostPreviousSubItem()
        {
            return MoveCaretSubItem(HeaderCtrl.FreeItemsName, true);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットをサブアイテムの一番後に移動させる
        /// </summary>
        protected bool MoveCaretMostNextSubItem()
        {
            return MoveCaretSubItem(HeaderCtrl.FreeItemsName, false);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットをクライアント領域の一番前のサブアイテムに移動
        /// </summary>
        protected bool MoveCaretMostPreviousSubItemInClient()
        {
            return MoveCaretSubItem(HeaderCtrl.DisplayedFreeItemsName, true);
        }

        ///--------------------------------
        /// <summary>
        /// キャレットをクライアント領域の一番後のサブアイテムに移動
        /// </summary>
        protected bool MoveCaretMostNextSubItemInClient()
        {
            return MoveCaretSubItem(HeaderCtrl.DisplayedFreeItemsName, false);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool CanMoveSelectedItem()
        {
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool CanMoveSelectedItem(IListItem targetItem)
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();

            if (ItemsSource == null)
            {
                return false;
            }

            //
            foreach (IListItem item in selectedItems.Keys)
            {
                //if( item.CanMove == false ) { return false; } これもいる？
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 選択されているアイテムの指定アイテムの兄として移動
        /// </summary>
        protected bool MoveSelectedItemToPreviousSibling(IListItem targetItem)
        {
            if (CanMoveSelectedItem(targetItem) == false)
            {
                return false;
            }

            //
            IListItemsSource source = ItemsSource;
            if (SuspendTransaction == false)
            {
                source.BeginTransaction();
            }

            IListItem[] items = GetSelectedItems();
            foreach (IListItem item in items)
            {
                if (item == targetItem)
                {
                    continue;
                }

                int targetIndex = source.Items.IndexOf(targetItem);
                source.Items.Move(targetIndex, item);
            }

            if (SuspendTransaction == false)
            {
                source.EndTransaction();
            }

            SelectItems(items, true);
            return true;
        }

        /// <summary>
        /// 選択されているアイテムの指定アイテムの弟として移動
        /// </summary>
        protected bool MoveSelectedItemToNextSibling(IListItem targetItem)
        {
            if (CanMoveSelectedItem(targetItem) == false)
            {
                return false;
            }

            //
            IListItemsSource source = ItemsSource;
            if (SuspendTransaction == false)
            {
                source.BeginTransaction();
            }

            IListItem[] items = GetSelectedItems();
            foreach (IListItem item in items)
            {
                if (item == targetItem)
                {
                    continue;
                }

                int targetIndex = source.Items.IndexOf(targetItem) + 1;
                source.Items.Move(targetIndex, item);

                targetItem = item;
            }

            if (SuspendTransaction == false)
            {
                source.EndTransaction();
            }

            SelectItems(items, true);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ラバーバンドとの衝突判定
        /// </summary>
        protected bool CollisionToRubberBand(int logY, int logHeight)
        {
            Rectangle bounds = new Rectangle();
            Rectangle rubberBandBounds = GetNormalizedRubberBand();

            bounds.X = 0;
            bounds.Y = logY;
            bounds.Width = 0;
            bounds.Height = logHeight;

            return IntersectBounds(rubberBandBounds, bounds, true);
        }

        ///--------------------------------
        /// <summary>
        /// ラバーバンドとの衝突判定
        /// </summary>
        protected bool CollisionToRubberBand(string name, int logY, int logHeight)
        {
            Rectangle bounds = new Rectangle();
            Rectangle rubberBandBounds = GetNormalizedRubberBand();
            //string      name = null;
            int index = -1;

            //name = GetColumnNameByClientX( cx);
            index = ToIndex(name);

            bounds.X = index;
            bounds.Y = logY;
            bounds.Width = 0;
            bounds.Height = logHeight;

            return IntersectBounds(rubberBandBounds, bounds, false);
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムセットの描画
        /// サブアイテム = 1セル
        /// </summary>
        protected void DrawSubItem(ListDrawDescriptor desc, IListItemInfo info, string name)
        {
            ListItemSelectedDictionary selectedItems = GetItemSelecteds();
            IListPartDrawer partDrawer = null;
            IListItem item = info.Item;
            IListSubItem subItem = null;
            Rectangle bounds = new Rectangle();
            Rectangle subItemBounds = new Rectangle();
            int bx = 0;
            int by = 0;
            bool selected = false;
            bool subSelected = false;
            bool caret = false;

            ColumnInfo columnInfo = null;
            ListColumnStyle style = null;
            IConstParameterValue value = null;

            //カラム情報の取得
            if ((columnInfo = GetColumnInfo(name)) == null)
            {
                return;
            }

            //
            selected = selectedItems.Selected(item);
            subSelected = selectedItems.SubSelected(item, name);

            if (_EnabledCaretMarker != false)
            {
                caret = (item == _CaretMarkerItem && name == _CaretMarkerName) ? true : false;
            }
            else
            {
                caret = (item == _CaretItem && name == _CaretColumnName) ? true : false;
            }

            //
            bx = columnInfo.ClientPosition;
            by = ToClientY(info.Y);

            subItemBounds.X = bx;
            subItemBounds.Y = by;
            subItemBounds.Width = columnInfo.Length;
            subItemBounds.Height = item.Length;

            //スタイルの取得
            if ((style = GetColumnStyle(name)) == null)
            {
                return;
            }

            //サブアイテムの取得
            if ((subItem = GetSubItem(style.SubItemName)) == null)
            {
                return;
            }

            //ラバーバンドとの衝突判定
            if (_EnabledRubberBand != false)
            {
                bool temporarySelected = false;
                bool temporarySubSelected = false;

                switch (_RubberBandKind)
                {
                    case RubberBandKinds.Item:
                        temporarySelected = CollisionToRubberBand(info.Y, item.Length);
                        break;

                    case RubberBandKinds.SubItem:
                        if (InterlockSelectedItemAndSubItem != false)
                        {
                            temporarySelected = CollisionToRubberBand(info.Y, item.Length);
                        }

                        if (style.CanSelect != false)
                        {
                            //temporarySubSelected = CollisionToRubberBand( bx, info.Y, item.Length);
                            temporarySubSelected = CollisionToRubberBand(name, info.Y, item.Length);
                        }
                        break;
                }

                if (temporarySelected != false)
                {
                    if (_RubberBandToggle != false)
                    {
                        selected = !selected;
                    }
                    else
                    {
                        selected = true;
                    }
                }

                if (temporarySubSelected != false)
                {
                    if (_RubberBandToggle != false)
                    {
                        subSelected = !subSelected;
                    }
                    else
                    {
                        subSelected = true;
                    }
                }
            }

            //値の取得
            value = OwnerCtrl.GetConstValue(item, name);

            //描画属性の設定
            desc.Item = item;
            desc.Index = info.Index;
            desc.Name = name;
            desc.Bounds = subItemBounds;
            desc.Caret = caret;
            desc.Enabled = item.Enabled;
            desc.Selected = selected;
            desc.SubSelected = subSelected;
            desc.Parameter = value;
            desc.PartParameters = subItem.PartParameters;
            desc.DrawerArgument = OwnerCtrl.GetDrawerArgument(style, item);
            desc.ContentDrawer = OwnerCtrl.ContentDrawer;
            desc.ColumnStyle = style;

            // 選択状態からテキストブラシを設定します。
            if (selected != false || subSelected != false)
            {
                desc.TextBrush = _SelectedTextBrush;
            }
            else
            {
                desc.TextBrush = _TextBrush;
            }

            //背景の描画
            OwnerCtrl.BackgroundDrawer.Draw(desc);

            //パーツの描画
            if (value != null)
            {

                //
                OwnerCtrl.SetupDrawDescriptor(item, name, desc);

                //
                foreach (IListPart part in subItem.GetParts(desc))
                {

                    if (OwnerCtrl.PartDrawers.TryGetValue(part.DrawerName,
                                                           out partDrawer) == false)
                    {
                        continue;
                    }

                    if (part.Bounds == Rectangle.Empty)
                    {
                        bounds.X = bx;
                        bounds.Y = by;
                        bounds.Width = subItemBounds.Width;
                        bounds.Height = subItemBounds.Height;
                    }
                    else
                    {

                        bounds = part.Bounds;
                        bounds.X += bx;
                        bounds.Y += by;
                    }

                    //
                    desc.Bounds = bounds;
                    desc.Part = part;

                    partDrawer.Draw(desc);
                }
            }
            else
            {

                partDrawer = OwnerCtrl.PartDrawers[String.Empty];
                desc.Bounds = subItemBounds;
                desc.Part = null;
                partDrawer.Draw(desc);
            }

            //
            if (_DashLineDictionary.IsMark(item, name) != false)
            {
                bounds.X = bx;
                bounds.Y = by;
                bounds.Width = subItemBounds.Width;
                bounds.Height = subItemBounds.Height;
                DrawDashLine(desc, bounds);
            }

            //キャレットの描画
            if (caret != false && desc.Focused != false)
            {
                subItemBounds.X = HeaderCtrl.GetDisplayedPosition(name);
                subItemBounds.Width = HeaderCtrl.GetDisplayedLength(name);
                DrawCaret(desc, subItemBounds);
            }
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの描画
        /// アイテム = 一行
        /// </summary>
        protected void DrawItem(ListDrawDescriptor desc, IListItemInfo info, string[] names)
        {
            foreach (string name in names)
            {
                DrawSubItem(desc, info, name);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void DrawCaretItem()
        {
            DrawItem(_CaretItem);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void DrawItem(IListItem item)
        {
            if (item == null)
            {
                return;
            }

            Graphics gc = null;
            Graphics imageGc = null;
            ListDrawDescriptor desc = null;
            IListItemInfo info = null;
            Image image = null;
            Rectangle bounds = new Rectangle();
            int width = ClientRectangleWithoutScrollBar.Width;
            int height = ClientRectangleWithoutScrollBar.Height;
            string[] names = null;

            try
            {
                if (width <= 0 || height <= 0)
                {
                    return;
                }

                if ((info = GetItemInfo(item)) == null)
                {
                    return;
                }

                image = new Bitmap(width, height);
                imageGc = Graphics.FromImage(image);

                desc = CreateDrawDescriptor(imageGc);
                names = DrawColumnNames;

                DrawItem(desc, info, names);
                DrawBorderLine(desc, info);

                //
                gc = CreateGraphics();

                bounds.X = 0;
                bounds.Y = ToClientY(info.Y);
                bounds.Width = width;
                bounds.Height = info.Item.Length;

                gc.DrawImage(image, bounds, bounds, GraphicsUnit.Pixel);
            }

            finally
            {
                if (gc != null) { gc.Dispose(); }
                if (imageGc != null) { imageGc.Dispose(); }
            }
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected override void Draw(Graphics gc)
        {
#if DEBUG
            //Debug.WriteLine( "-------- Draw list ctrl" + _DrawCounter.ToString());
            _DrawCounter++;
#endif

            ListDrawDescriptor desc = null;
            IListItemInfo[] infos = GetDisplayedItemInfos();
            string[] names = null;

            desc = CreateDrawDescriptor(gc);
            names = DrawColumnNames;

            //
            foreach (IListItemInfo info in infos)
            {
                DrawItem(desc, info, names);
                DrawBorderLine(desc, info);
            }

            DrawDragMarker(desc.Gc);
        }

        ///--------------------------------
        /// <summary>
        /// 描画すべき名前を描画の順番で取得
        /// </summary>
        protected string[] DrawColumnNames
        {
            get
            {
                List<string> list = new List<string>();

                list.AddRange(HeaderCtrl.DisplayedFreeItemsName);
                list.AddRange(HeaderCtrl.DisplayedLeftFixedItemsName);
                list.AddRange(HeaderCtrl.DisplayedRightFixedItemsName);
                return list.ToArray();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListDrawDescriptor CreateDrawDescriptor(Graphics gc)
        {
            ListDrawDescriptor desc = null;

            desc = new ListDrawDescriptor(gc, Font, OwnerCtrl.Images);
            desc.Focused = Focused;
            desc.Counter = AnimationCounter;

            if (Focused != false)
            {
                desc.SelectedItemBackBrush = _ItemSelectedFtBrush;
                desc.SelectedSubItemBackBrush = _SubItemSelectedFtBrush;
            }
            else
            {
                desc.SelectedItemBackBrush = _ItemSelectedFfBrush;
                desc.SelectedSubItemBackBrush = _SubItemSelectedFfBrush;
            }

            desc.ItemBackBrush = _ItemBackBrush;
            desc.ItemBorderPen = _ItemBorderPen;

            return desc;
        }

        ///--------------------------------
        /// <summary>
        /// 区切り線の描画
        /// </summary>
        protected void DrawBorderLine(ListDrawDescriptor desc, IListItemInfo info)
        {
            IListItem item = info.Item;
            int bx = 0;
            int by = ToClientY(info.Y);
            int ex = _RightEdgeClientX;
            int ey = ToClientY(info.Y + info.Item.Length - 1);

            desc.Gc.DrawLine(desc.ItemBorderPen, bx, ey, ex, ey);
            desc.Gc.DrawLine(desc.ItemBorderPen, ex, by, ex, ey);
        }

        ///--------------------------------
        /// <summary>
        /// カレット線の描画
        /// </summary>
        protected void DrawCaret(ListDrawDescriptor desc, Rectangle bounds)
        {
            Pen pen = null;
            Color color = SystemColors.Window;
            int width = 2;

            color = Color.FromArgb(255, color.R ^ 255, color.G ^ 255, color.B ^ 255);

            pen = new Pen(color);
            pen.Width = width;
            //pen.DashStyle = DashStyle.Dot;

            pen.DashOffset = 0.0F;
            if ((desc.Counter & 1) == 0)
            {
                pen.DashOffset = 1.0F;
            }

            bounds.X += width / 2;
            bounds.Y += width / 2;
            bounds.Width -= width;
            bounds.Height -= width + 1;

            desc.Gc.DrawRectangle(pen, bounds);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void DrawDashLine(ListDrawDescriptor desc, Rectangle bounds)
        {
            Pen penBk = SystemPens.Window.Clone() as Pen;
            Pen penLine = Pens.Black.Clone() as Pen;

            penLine.DashStyle = DashStyle.Dash;
            penLine.DashPattern = new float[] { 2.0f, 2.0f };

            bounds.Width--;
            bounds.Height--;

            desc.Gc.DrawRectangle(penLine, bounds);

            bounds.Inflate(-1, -1);
            penLine.DashOffset = 2.0f;
            desc.Gc.DrawRectangle(penBk, bounds);
            desc.Gc.DrawRectangle(penLine, bounds);
        }

        ///--------------------------------
        /// <summary>
        /// ドラックマーカーの描画
        /// </summary>
        protected void DrawDragMarker(Graphics gc)
        {
            if (_EnabledDragMarker == false)
            {
                return;
            }

            ListItemInPosition inPosition = ListItemInPosition.None;
            IListItemInfo info = null;
            IListItem item = null;
            int x = 0;
            int y = 0;
            int width = ClientRectangleWithoutScrollBar.Width;
            int height = 4;

            if ((info = GetItemInfo(_DragItem)) == null)
            {
                return;
            }

            item = info.Item;
            inPosition = _DragItemInPosition;

            switch (inPosition)
            {
                case ListItemInPosition.Upper:
                    y = ToClientY(info.Y - height / 2);
                    break;

                case ListItemInPosition.Lower:
                    y = ToClientY(info.Y + item.Length - height / 2);
                    break;

                case ListItemInPosition.Center:
                    y = ToClientY(info.Y);
                    height = item.Length;
                    break;

                default:
                    return;
            }

            gc.FillRectangle(new SolidBrush(SystemColors.Highlight),
                              x, y, width, height);
        }

        ///--------------------------------
        /// <summary>
        /// ドラックの開始
        /// </summary>
        protected void TrackDragDrop(DragDropEffects effects)
        {
            DataObject data = null;

            if ((data = OwnerCtrl.TrackDragDrop()) == null)
            {
                data = new DataObject();
            }

            DoDragDrop(data, effects);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListItemInfo GetItemInfo(IListItem item)
        {
            IListItemInfo[] itemInfos = ItemInfos;

            if (itemInfos == null)
            {
                return null;
            }

            try
            {
                var infos = itemInfos
                    .Where(i => i.Item == item);
                return infos.FirstOrDefault();
            }
            catch { return null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected IListItemInfo GetItemInfo(int index)
        {
            IListItemInfo[] infos = ItemInfos;

            if (infos == null)
            {
                return null;
            }

            if (index < 0 || index >= infos.Length)
            {
                return null;
            }

            return infos[index];
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool EditReserved
        {
            get
            {
                return _ReserveBeginEditByKey != false ? true : false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// サブアイテムの編集矩形の取得
        /// (クライアント座標)
        /// </summary>
        protected Rectangle GetSubItemBounds(IListItem item, string name)
        {
            IListItemInfo info = null;
            ColumnInfo columnInfo = null;
            Rectangle bounds = new Rectangle();

            //カラム情報の取得
            if ((columnInfo = GetColumnInfo(name)) == null)
            {
                return Rectangle.Empty;
            }

            if ((info = GetItemInfo(item)) == null)
            {
                return Rectangle.Empty;
            }

            bounds.X = columnInfo.ClientPosition;
            bounds.Y = ToClientY(info.Y);
            bounds.Width = columnInfo.Length;
            bounds.Height = item.Length;

            return bounds;
        }

        ///--------------------------------
        /// <summary>
        /// キー入力での編集開始の予約のキャンセル
        /// </summary>
        protected void CancelReserveBeginEditByKey()
        {
            _ReserveBeginEditByKey = false;
            _ReserveBeginByEditBeginKey = false;
            _ReserveImmediateEdit = false;
            _ReserveImmediateEditKey = 0;
            _ReservePressedShiftKey = false;
            _ReserveKey.Remove(0, _ReserveKey.Length);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ReserveImmediateEditKey(Keys key, bool shift)
        {
            _ReserveImmediateEditKey = key;
            _ReservePressedShiftKey = shift;
        }

        ///--------------------------------
        /// <summary>
        /// キー入力での編集開始が可能なのか？
        /// </summary>
        protected bool CanEditByKey(string name)
        {
            IListSubItem subItem = null;

            if ((subItem = GetSubItemByStyleName(_CaretColumnName)) == null)
            {
                return false;
            }

            return subItem.EditByKey;
        }

        ///--------------------------------
        /// <summary>
        /// キー入力での編集開始の予約
        /// </summary>
        protected bool PreparationBeginEditByKey(char keyChar)
        {
#if false
            if( _CaretItem == null || _CaretColumnName == null ||
                _ReserveBeginEditByKey != false ) {
                return false;
            }
#else

            if (_CaretItem == null || _CaretColumnName == null)
            {
                return false;
            }

            if (_ReserveBeginEditByKey != false)
            {
                if (Char.IsControl(keyChar) == false)
                {
                    _ReserveKey.Append(keyChar);
                }
                return true;
            }
#endif

            //
            if (Char.IsControl(keyChar) != false)
            {
                return false;
            }

            //
            if (CanEditByKey(_CaretColumnName) == false)
            {
                return false;
            }

            // スペースキーだったら編集開始しない。
            if (keyChar == ' ')
            {
                return false;
            }

            //編集開始の予約
            _ReserveBeginEditByKey = true;
            _ReserveBeginByEditBeginKey = false;
            _ReserveImmediateEdit = true;
            _ReserveKey.Append(keyChar);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 入力開始キーでの編集開始の予約
        /// この予約によって編集を開始すると、TextBox内部をカーソルキーで移動できる
        /// ようになる
        /// </summary>
        protected bool PreparationBeginEditByKey(bool immediateEdit)
        {
            if (_CaretItem == null || _CaretColumnName == null ||
                _ReserveBeginEditByKey != false)
            {
                return false;
            }

            IConstParameterValue value = null;

            //
            if (CanEditByKey(_CaretColumnName) == false)
            {
                return false;
            }

            //
            if ((value = OwnerCtrl.GetConstValue(_CaretItem, _CaretColumnName)) == null)
            {
                return false;
            }

            //編集開始の予約
            _ReserveBeginEditByKey = true;
            _ReserveBeginByEditBeginKey = true;
            _ReserveImmediateEdit = immediateEdit;
            _ReserveKey.Append(value.ToString());
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 予約されたキー入力で編集を開始する
        /// </summary>
        protected bool BeginEditByKey()
        {
            string text = null;

            if (_ReserveBeginEditByKey == false)
            {
                return false;
            }

            text = _ReserveKey.ToString();
            if (BeginEdit(_CaretItem, _CaretColumnName, null,
                           text,
                           _ReserveBeginByEditBeginKey,
                           _ReserveImmediateEdit,
                           _ReserveImmediateEditKey,
                           OwnerCtrl.MoveNextLineCaretAfterEndEditByEnter) != false)
            {
                return true;
            }

            if (_ReserveImmediateEditKey == Keys.Enter)
            {
                if (MoveCaret(_ReservePressedShiftKey ? CaretMove.Up : CaretMove.Down) == true)
                {
                    Invalidate();
                }
            }

            CancelReserveBeginEditByKey();
            return true;
        }
#if false
        ///--------------------------------
        /// <summary>
        /// 指定した文字列で編集を開始する
        /// </summary>
        protected bool BeginEditByKey( string text)
        {
            return BeginEdit( _CaretItem, _CaretColumnName, null,
                              text,
                              false, false, 0);
        }
#endif
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool BeginEditByDoubleClick(string editorName)
        {
            if (BeginEdit(_CaretItem, _CaretColumnName, editorName,
                           null, false, false, 0,
                           OwnerCtrl.MoveNextLineCaretAfterEndEditByEnter) != false)
            {
                return true;
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 編集の開始
        /// </summary>
        protected bool BeginEdit(IListItem item, string name, string editorName)
        {
            //return BeginEdit( item, name, editorName, null, false, false, 0);  テスト
            return BeginEdit(item, name, editorName, null,
                              true,
                              false, 0,
                              OwnerCtrl.MoveNextLineCaretAfterEndEditByEnter);
        }

        ///--------------------------------
        /// <summary>
        /// 編集の開始
        /// </summary>
        protected bool BeginEdit(IListItem item, string name, string editorName, string text, bool beginByEditBeginKey, bool immediateEdit, Keys immediateEditKey, bool moveNextLineCaretAfterEndEditByEnter)
        {
            if (_ActiveEditor != null)
            {
                return false;
            }

            if (item == null)
            {
                return false;
            }

            if (item.CanEdit == false)
            {
                return false;
            }

            //
            ListColumnStyle style = null;
            IListSubItem subItem = null;
            InplaceEditor editor = null;
            Rectangle bounds;
            IParameterValue value = null;

            HeaderCtrl.FitDisplay(name);
            bounds = GetSubItemBounds(item, name);
            bounds.Width = HeaderCtrl.GetDisplayedLength(name);

            //スタイルの取得
            if ((style = GetColumnStyle(name)) == null)
            {
                return false;
            }

            //エディタが指定されていないのか？
            if (editorName == null)
            {
                //サブアイテムの取得
                if ((subItem = GetSubItem(style.SubItemName)) == null)
                {
                    return false;
                }
                editorName = subItem.RepresentativeEditorName(immediateEdit);
            }

            //エディタの取得
            if ((editor = GetInplaceEditor(editorName)) == null)
            {
                return false;
            }

            //エディタに引数を渡す
            if (style.Argument != null)
            {
                editor.Arguments = OwnerCtrl.GetEditorArgument(style, item);
            }

            // 現在の値を取得します。
            if (editor.RequireInitialValue != false)
            {
                if ((value = OwnerCtrl.GetValue(item, name)) != null)
                {
                    editor.InitialValue = value;
                }
                else
                {
                    return false;
                }

                editor.ParameterManipulator = value;
                if (value == null)
                {
                    return false;
                }
            }
            else
            {
            }

            //
            if (immediateEdit != false)
            {
                if (editor.CanImmediateEdit == false)
                {
                    return false;
                }
            }

            //
            editor.OwnerCtrl = this;
            editor.Owner = this;
            editor.BeginByEditBeginKey = beginByEditBeginKey;
            editor.ImmediateEditKey = immediateEditKey;
            editor.MoveNextLineCaretAfterEndEditByEnter = moveNextLineCaretAfterEndEditByEnter;
            editor.TargetName = name;
            editor.Item = item;

            Debug.WriteLine("BeginEdit");
            if (editor.BeginEdit(bounds, text, ReadOnly) == false)
            {
                return false;
            }

            _ActiveEditor = editor;
            _EditingItem = item;
            _EditingName = name;

            //
            if (editor.Permanent == false)
            {
                //editor.EndEdit( editor.Cancel);
                editor.EndEdit();
            }
            else
            {
                OnEditBegan();
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// 選択が変更された時に呼ばれる
        /// </summary>
        protected void OnSelectChanged()
        {
            OwnerCtrl.OnSelectChanged();
        }

        ///--------------------------------
        /// <summary>
        /// 編集が開始された時に呼ばれる
        /// </summary>
        protected void OnEditBegan()
        {
            OwnerCtrl.OnEditBegan();
        }

        ///--------------------------------
        /// <summary>
        /// 編集が終了した時に呼ばれる
        /// </summary>
        protected void OnEditEnded()
        {
            OwnerCtrl.OnEditEnded();
        }

        ///--------------------------------
        /// <summary>
        /// ダブルクリックされた時に呼ばれる
        /// </summary>
        protected void OnDoubleClicked(string name, IListItem item)
        {
            OwnerCtrl.OnDoubleClicked(name, item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnTrackContextMenu(MouseEventArgs e)
        {
            if (_ActiveEditor != null)
            {
                return;
            }

            ContextMenuStrip contextMenu = null;

            if ((contextMenu = OwnerCtrl.AttachedContextMenuStrip) != null)
            {
                contextMenu.Show(this, new Point(e.X, e.Y));
            }
        }

        ///--------------------------------
        /// <summary>
        /// マウスボタンの押下処理
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (IsEditing == true)
            {
                EndEdit();
            }
            else
            {
                base.OnMouseDown(e);
            }

            this.BeginInvoke(new MethodInvoker(delegate () { this.Focus(); }));
        }

        ///--------------------------------
        /// <summary>
        /// マウスの移動処理
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            // セルにツールチップを表示します。
            {
                ListCollisionResult result = this.Collision(e.X, e.Y);

                IListItemToolTipProvider lastProvider = this.FindToolTipProvider(_lastMouseListCollisionResult);
                IListItemToolTipProvider newProvider = this.FindToolTipProvider(result);

                if (_lastMouseListCollisionResult?.Name != result.Name ||
                    _lastMouseListCollisionResult?.Item != result.Item ||
                    lastProvider != newProvider)
                {
                    // ホーバーしているセルが移動した感じを出すために、
                    // 現在のツールチップを一旦、消します。
                    _toolTipError.SetToolTip(this, string.Empty);
                    _toolTipWarning.SetToolTip(this, string.Empty);
                    _toolTipNormal.SetToolTip(this, string.Empty);

                    if (newProvider != null)
                    {
                        try
                        {
                            var message = newProvider.GetToolTip(result.Name, result.Item);
                            if (string.IsNullOrEmpty(message?.Message) == false)
                            {
                                switch (message.Type)
                                {
                                    case ListItemToolTipType.Error:
                                        _toolTipError.SetToolTip(this, message.Message);
                                        break;
                                    case ListItemToolTipType.Warning:
                                        _toolTipWarning.SetToolTip(this, message.Message);
                                        break;
                                    case ListItemToolTipType.Disabled:
                                    case ListItemToolTipType.Normal:
                                        _toolTipNormal.SetToolTip(this, message.Message);
                                        break;
                                }
                            }
                        }
                        catch
                        {
                        }
                    }
                }

                _lastMouseListCollisionResult = result;
            }
        }

        private IListItemToolTipProvider FindToolTipProvider(ListCollisionResult result)
        {
            var drawerName = result?.Part?.DrawerName;
            if (drawerName != null)
            {
                var provider = OwnerCtrl.PartDrawers[drawerName] as IListItemToolTipProvider;
                if (provider != null)
                {
                    return provider;
                }
            }

            return result?.Item as IListItemToolTipProvider;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            UpdateColumnInfos();
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            base.OnGotFocus(e);

            this.Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnLeave(EventArgs e)
        {
            base.OnLeave(e);

            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnSortChanged()
        {
            OnHeaderSortChanged(this, new EventArgs());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnHeaderResized(object sender, EventArgs e)
        {
            UpdateColumnInfos();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnHeaderChanged(object sender, EventArgs e)
        {
            //補正、更新、補正とする必要があるので、UpdateScrollBar()を2回呼ぶ
            UpdateScrollBar();
            UpdateColumnInfos();
            UpdateScrollBar();
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        /// ヘッダのソート方法が変更された時に呼ばれる
        /// </summary>
        private void OnHeaderSortChanged(object sender, EventArgs e)
        {
            Sort();
        }

        ///--------------------------------
        /// <summary>
        /// ヘッダの内部位置が変更された時に呼ばれる
        /// </summary>
        private void OnHeaderPositionChanged(object sender, EventArgs e)
        {
            HorizontalScrollBar.Value = HeaderCtrl.Position;

            UpdateColumnInfos();
            UpdateScrollBar();
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        /// ヘッダのカラムの区切り線がダブルクリックされた時に呼ばれる
        /// </summary>
        private void OnHeaderAdjustColumnWidth(object sender,
                                                HeaderCtrl.VSplitDoubleClickEventArgs e)
        {
            string name = e.Name;
            IConstParameterValue maxValue = null;

            foreach (IListItem item in ItemsSource.Items)
            {
                IConstParameterValue value = item.GetConstValue(name);
                if (value != null)
                {
                    if (maxValue == null) maxValue = value;
                    else if (maxValue.ToString().Length < value.ToString().Length)
                    {
                        maxValue = value;
                    }
                }
            }
            if (maxValue != null)
            {
                foreach (IHeaderItem item in HeaderSource.Items)
                {
                    if (item.Name == name)
                    {
                        item.Length = GetPartsDrawWidth(name, maxValue, item.Length);
                        break;
                    }
                }
            }

            UpdateColumnInfos();
            UpdateScrollBar();
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnHeaderGotFocus(object sender, EventArgs e)
        {
            this.Select();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int GetPartsDrawWidth(string name, IConstParameterValue value, int currentWidth)
        {
            int newWidth = 0;
            Graphics Gc = null;

            try
            {
                IListSubItem subItem = GetSubItemByStyleName(name);
                Gc = CreateGraphics();
                ListDrawDescriptor desc = new ListDrawDescriptor(Gc, Font, OwnerCtrl.Images);
                desc.Selected = true;
                desc.Parameter = value;
                desc.Bounds = new Rectangle(0, 0, currentWidth, 0);
                IListPart[] parts = subItem.GetParts(desc);
                foreach (IListPart part in parts)
                {
                    IListPartDrawer drawer = OwnerCtrl.PartDrawers[part.DrawerName];
                    desc.Bounds = part.Bounds;
                    newWidth += drawer.DrawWidth(desc);
                }
            }
            finally
            {
                if (Gc != null) { Gc.Dispose(); }
            }

            return newWidth;
        }

        private void OnVerticalScrollBarValueChanged(object sender, EventArgs e)
        {
            if (VerticalScrollValueChanged != null)
            {
                VerticalScrollValueChanged(sender, e);
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// 基本のオペレータ
        /// </summary>
        private class ListCtrlCoreOperator : OperatableControlOperator
        {
            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public ListCtrlCoreOperator(ListCtrlCore ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected ListCtrlCore OwnerCtrl
            {
                get { return (ListCtrlCore)base.OperatableControl; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedControlKey
            {
                get { return (Control.ModifierKeys & Keys.Control) != 0; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedShiftKey
            {
                get { return (Control.ModifierKeys & Keys.Shift) != 0; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected int GetLogicalIndex(int cx)
            {
                string name = null;

                if ((name = OwnerCtrl.GetColumnNameByClientX(cx)) == null)
                {
                    return -1;
                }

                return OwnerCtrl.ToIndex(name);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected int GetLogicalX(int cx)
            {
                return OwnerCtrl.ToLogicalX(cx);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected int GetLogicalY(int cy)
            {
                return OwnerCtrl.ToLogicalY(cy);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected Point GetValidLogialPoint(int x, int y)
            {
                Size size = OwnerCtrl.ClientSize;

                if (x < 0) { x = 0; }
                if (y < 0) { y = 0; }
                if (x > size.Width) { x = size.Width - 1; }
                if (y > size.Height) { y = size.Height - 1; }

                return OwnerCtrl.ToLogicalPoint(new Point(x, y));
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected void Normalize(ref int minValue, ref int maxValue)
            {
                OwnerCtrl.Normalize(ref minValue, ref maxValue);
            }

            ///--------------------------------
            /// <summary>
            /// アイテム、サブアイテムの選択解除
            /// </summary>
            protected void UnselectItem()
            {
                OwnerCtrl.UnselectItem();
            }

            ///--------------------------------
            /// <summary>
            /// アイテムの選択
            /// </summary>
            protected void SelectItem(int logY)
            {
                OwnerCtrl.SelectItem(logY, true);
            }

            ///--------------------------------
            /// <summary>
            /// アイテムの選択
            /// </summary>
            protected IListItem[] SelectItem(int beginLogY, int endLogY)
            {
                return OwnerCtrl.SelectItem(beginLogY, endLogY, true);
            }

            ///--------------------------------
            /// <summary>
            /// アイテムの反転選択
            /// </summary>
            protected IListItem[] ToggleSelectItem(int beginLogY, int endLogY)
            {
                return OwnerCtrl.ToggleSelectItem(beginLogY, endLogY);
            }

            ///--------------------------------
            /// <summary>
            /// サブアイテムの選択
            /// </summary>
            protected IListItem[] SelectSubItem(int beginIndex, int endIndex, int beginLogY, int endLogY)
            {
                return OwnerCtrl.SelectSubItem(beginIndex, endIndex,
                                                beginLogY, endLogY, true);
            }

            ///--------------------------------
            /// <summary>
            /// サブアイテムの反転選択
            /// </summary>
            protected IListItem[] ToggleSelectSubItem(int beginIndex, int endIndex, int beginLogY, int endLogY)
            {
                return OwnerCtrl.ToggleSelectSubItem(beginIndex, endIndex,
                                                      beginLogY, endLogY);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool CanPutCaret(string name)
            {
                ListColumnStyle style = null;

                if ((style = OwnerCtrl.GetColumnStyle(name)) == null)
                {
                    return false;
                }

                return style.CanPutCaret;
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// 待機状態のオペレータ
        /// </summary>
        private class ExchangeOperator : ListCtrlCoreOperator
        {
            private Point _FirstMouseDownPoint = new Point(-1, -1);
            private bool _InterimDrag = false;         //ドラッグ処理の可能性
            private bool _InterimSelectItem = false;     //選択処理の可能性
            private bool _InterimSelectSubItem = false;  //選択処理の可能性
            private bool _InterimEditSubItem = false;

            private IListItem _CandidateEditItem = null;
            private string _CandidateEditName = null;
            private string _CandidateEditEditorName = null;

            private int _BeginY = -1;
            private int _BeginIndex = -1;
            private int _EndY = -1;
            private int _EndIndex = -1;
            private bool _OnSelected = false;

            //キャプション編集の可能性がある場合にtrueになる
            //Down、Upの両方がtrueにならないと編集モードにはならない
            private bool _InterimEditCaptionDown = false;
            private bool _InterimEditCaptionUp = false;

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public ExchangeOperator(ListCtrlCore ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _InterimDrag = false;
                _InterimSelectItem = false;
                _InterimSelectSubItem = false;
                _InterimEditSubItem = false;

                _CandidateEditItem = null;
                _CandidateEditName = null;
                _CandidateEditEditorName = null;

                _InterimEditCaptionDown = false;
                _InterimEditCaptionUp = false;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                switch (e.Button)
                {
                    case MouseButtons.Left:
                        return MouseButtonDownOperation(e, true);

                    case MouseButtons.Middle:
                        break;

                    case MouseButtons.Right:
                        return MouseButtonDownOperation(e, false);
                }
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                switch (e.Button)
                {
                    case MouseButtons.Left:
                        return MouseLeftButtonUpOperation(e);

                    case MouseButtons.Middle:
                        break;

                    case MouseButtons.Right:
                        MouseRightButtonUpOperation(e);
                        TrackContextMenu(e);
                        break;
                }
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDoubleClick(MouseEventArgs e)
            {
                switch (e.Button)
                {
                    case MouseButtons.Left:
                        MouseLeftButtonDoubleClickedOperation(e);
                        _InterimEditCaptionDown = false;
                        _InterimEditCaptionUp = false;
                        break;
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                Size dragSize = SystemInformation.DragSize;

                if (_InterimDrag != false ||
                    _InterimSelectItem != false || _InterimSelectSubItem != false)
                {

                    //
                    if (this._InterimEditSubItem != false)
                    {
                        return true;
                    }

                    //
                    if (Math.Abs(e.X - _FirstMouseDownPoint.X) > dragSize.Width ||
                        Math.Abs(e.Y - _FirstMouseDownPoint.Y) > dragSize.Height)
                    {

                        if (_InterimDrag != false)
                        {
                            OwnerCtrl.ChangeOperatorToMove(_FirstMouseDownPoint);
                        }

                        if (_InterimSelectItem != false)
                        {
                            OwnerCtrl.ChangeOperatorToSelectItem(_FirstMouseDownPoint);
                        }

                        if (_InterimSelectSubItem != false)
                        {
                            OwnerCtrl.ChangeOperatorToSelectSubItem(_FirstMouseDownPoint);
                        }

                        //
                        OwnerCtrl.CaretMarkerTo(false);

                        if (PressedShiftKey == false)
                        {
                            OwnerCtrl.SetCardinalItemByCaret();
                        }

                        OwnerCtrl.CaretMarker();
                        OwnerCtrl.RubberBand();
                    }
                }
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseWheel(MouseEventArgs e)
            {
                int relativeMove = 0;

                relativeMove = e.Delta * SystemInformation.MouseWheelScrollLines / 120;

                //
                if (PressedShiftKey != false)
                {
                    if (OwnerCtrl.ScrollHorizontalScrollBarRelative
                        (-relativeMove * 16) != false)
                    {
                        OwnerCtrl.Invalidate();
                    }
                    return true;
                }

                //
                if (OwnerCtrl.ScrollVerticalScrollBarRelative
                    (-relativeMove) != false)
                {
                    OwnerCtrl.UpdateScrollBar();
                    OwnerCtrl.Invalidate();
                }
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyDown(KeyEventArgs e)
            {
                // 既に編集開始の予約がされているのか？
                if (OwnerCtrl.EditReserved != false)
                {
                    return;
                }

                Keys key = e.KeyCode;
                //Rectangle       bounds;
                bool pressedControlKey = e.Control;
                bool pressedShiftKey = e.Shift;
                bool redraw = false;
                bool select = false;
                bool newLine = false;

                //
                switch (key)
                {
                    case Keys.Escape:
                        OwnerCtrl.UnselectItem();
                        OwnerCtrl.OnSelectChanged();
                        OwnerCtrl.Invalidate();
                        return;

                    case Keys.F12:
                        OwnerCtrl.OnSortChanged();
                        return;

                    case Keys.Tab:
                        if (e.Control != false)
                        {
                            return;
                        }

                        key = e.Shift != false ? Keys.Left : Keys.Right;
                        pressedShiftKey = false;
                        newLine = true;
                        break;
                }

                //
                switch (key)
                {
                    case Keys.Up:
                        redraw = OwnerCtrl.MoveCaret(pressedControlKey ?
                                                      CaretMove.MostUpInClient :
                                                      CaretMove.Up);
                        select = pressedShiftKey;
                        break;

                    case Keys.Down:
                        redraw = OwnerCtrl.MoveCaret(pressedControlKey ?
                                                      CaretMove.MostDownInClient :
                                                      CaretMove.Down);
                        select = pressedShiftKey;
                        break;

                    case Keys.Left:
                        redraw = OwnerCtrl.MoveCaret(pressedControlKey ?
                                                      CaretMove.MostLeftInClient :
                                                      (newLine != false ?
                                                        CaretMove.Left :
                                                        CaretMove.LeftWithoutNewLine));
                        select = pressedShiftKey;
                        break;

                    case Keys.Right:
                        redraw = OwnerCtrl.MoveCaret(pressedControlKey ?
                                                      CaretMove.MostRightInClient :
                                                      (newLine != false ?
                                                        CaretMove.Right :
                                                        CaretMove.RightWithoutNewLine));
                        select = pressedShiftKey;
                        break;

                    case Keys.PageUp:
                        redraw = OwnerCtrl.MoveCaret(CaretMove.PageUp);
                        select = pressedShiftKey;
                        break;

                    case Keys.PageDown:
                        redraw = OwnerCtrl.MoveCaret(CaretMove.PageDown);
                        select = pressedShiftKey;
                        break;

                    case Keys.Home:
                        redraw = OwnerCtrl.MoveCaret(pressedControlKey ?
                                                      CaretMove.MostUp :
                                                      CaretMove.MostLeft);
                        select = pressedShiftKey;
                        break;

                    case Keys.End:
                        redraw = OwnerCtrl.MoveCaret(pressedControlKey ?
                                                      CaretMove.MostDown :
                                                      CaretMove.MostRight);
                        select = pressedShiftKey;
                        break;

                    case Keys.Enter:
                        if (OwnerCtrl.MoveNextLineCaretAfterEndEditByEnter == true &&
                            OwnerCtrl.ContainsImmediateEditBeginKey(key) == false)
                        {
                            redraw = OwnerCtrl.MoveCaret(pressedShiftKey ?
                                                          CaretMove.Up :
                                                          CaretMove.Down);
                        }
                        break;
                }

                //
                if (pressedShiftKey == false && pressedControlKey == false)
                {

                    //編集開始キーなのか？
                    if (OwnerCtrl.ContainsEditBeginKey(key) != false)
                    {
                        OwnerCtrl.PreparationBeginEditByKey(false);
                    }
                }

                if (pressedControlKey == false)
                {
                    //直接編集開始キーなのか？
                    if (OwnerCtrl.ContainsImmediateEditBeginKey(key) != false)
                    {
                        OwnerCtrl.ReserveImmediateEditKey(key, pressedShiftKey);
                        OwnerCtrl.PreparationBeginEditByKey(true);
                    }
                }

                //基準アイテム〜カレットまでの範囲選択
                if (select != false)
                {
                    OwnerCtrl.SelectSubItemCardinalToCaret();
                    OwnerCtrl.OnSelectChanged();
                }
                else
                {

                    if (pressedShiftKey == false)
                    {
                        OwnerCtrl.SetCardinalItemByCaret();
                    }
                }

                //
                if (redraw != false)
                {
                    OwnerCtrl.Invalidate();
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyPress(KeyPressEventArgs e)
            {
                //編集開始キーなのか？
                if (OwnerCtrl.ContainsEditBeginKey((Keys)e.KeyChar) != false)
                {
                    return;
                }

                OwnerCtrl.PreparationBeginEditByKey(e.KeyChar);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyUp(KeyEventArgs e)
            {
                Keys key = e.KeyCode;
                Rectangle bounds;

                switch (key)
                {
                    case Keys.Apps:
                        bounds = OwnerCtrl.GetCaretItemClientBounds();
                        TrackContextMenu
                            (new MouseEventArgs(MouseButtons.None, 0, bounds.X, bounds.Y, 0));
                        return;
                }

                OwnerCtrl.BeginEditByKey();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnAnimation(EventArgs e)
            {
                //OwnerCtrl.DrawCaretItem();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnTimer(EventArgs e)
            {
                if (_InterimEditCaptionDown != false &&
                    _InterimEditCaptionUp != false)
                {

                    OwnerCtrl.StopTimer();
                    OwnerCtrl.BeginEdit(_CandidateEditItem,
                                         _CandidateEditName,
                                         _CandidateEditEditorName);

                    _InterimEditCaptionDown = false;
                    _InterimEditCaptionUp = false;
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragOver(DragEventArgs e)
            {
                Point point;
                ListCollisionResult result = null;
                int ly = 0;
                bool enableMarker = false;

                point = OwnerCtrl.PointToClient(new Point(e.X, e.Y));
                OwnerCtrl.DragScroll(point.X, point.Y);
                result = OwnerCtrl.CollisionForDrag(point.X, point.Y);

                if (e.Data.GetDataPresent(DataFormats.FileDrop) != false)
                {
                    enableMarker = true;
                }
                else
                {

                    OwnerCtrl.QueryDragOver(e, result.Item, ToDropPosition(result.InPosition));
                    if (e.Effect != DragDropEffects.None)
                    {
                        enableMarker = true;
                    }
                }

                //
                if (enableMarker != false)
                {
                    if (OwnerCtrl.DecideFileDropPosition != false)
                    {
                        ly = GetLogicalY(point.Y);
                        OwnerCtrl.RubberBand(ly, ly, false);
                    }
                    else
                    {
                        if (result.Item != null)
                        {
                            OwnerCtrl.DragMarker(result.Item, result.InPosition);
                        }
                        else
                        {
                            OwnerCtrl.DragMarker();
                        }
                    }

                    OwnerCtrl.Invalidate();
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragLeave(EventArgs e)
            {
                if (OwnerCtrl.DecideFileDropPosition != false)
                {
                    OwnerCtrl.RubberBand();
                }
                else
                {
                    OwnerCtrl.DragMarker();
                }
                OwnerCtrl.Invalidate();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragDrop(DragEventArgs e)
            {
                Point point;
                ListCollisionResult result = null;

                point = OwnerCtrl.PointToClient(new Point(e.X, e.Y));
                OwnerCtrl.DragScroll(point.X, point.Y);
                result = OwnerCtrl.CollisionForDrag(point.X, point.Y);

                if (e.Data.GetDataPresent(DataFormats.FileDrop) != false)
                {
                    string[] filePaths = (string[])e.Data.GetData(DataFormats.FileDrop);
                    Array.Sort(filePaths);

                    if (OwnerCtrl.DecideFileDropPosition != false)
                    {
                        if (OwnerCtrl.DropFiles(filePaths, result.Item) != false)
                        {
                        }
                    }
                    else
                    {
                        if (OwnerCtrl.DropFiles(filePaths, result.Item,
                                                 ToDropPosition(result.InPosition)) != false)
                        {
                        }
                    }
                }
                else
                {

                    OwnerCtrl.ExecuteDragDrop(result.Item, ToDropPosition(result.InPosition));
                }

                OwnerCtrl.RubberBand();
                OwnerCtrl.DragMarker();
                OwnerCtrl.Invalidate();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected DropPosition ToDropPosition(ListItemInPosition position)
            {
                switch (position)
                {
                    case ListItemInPosition.Upper: return DropPosition.Upper;
                    case ListItemInPosition.Lower: return DropPosition.Lower;
                    case ListItemInPosition.Center: return DropPosition.Center;
                }
                return DropPosition.None;
            }

            ///--------------------------------
            /// <summary>
            /// マウスのボタンを押した時の処理
            /// </summary>
            private bool MouseButtonDownOperation(MouseEventArgs e, bool canEdit)
            {
                ListCollisionResult result = OwnerCtrl.Collision(e.X, e.Y);

                //何にも衝突していないのか？
                if (result.Name == String.Empty)
                {
                    if (PressedShiftKey != false)
                    {
                        return true;
                    }

                    _InterimSelectItem = true;
                    _FirstMouseDownPoint.X = e.X;
                    _FirstMouseDownPoint.Y = e.Y;

                    if (OwnerCtrl.ClickedOutsideUnselectItem != false)
                    {
                        if (PressedControlKey == false)
                        {
                            UnselectItem();
                        }
                    }
                    return true;
                }

                //シフトキーが押されているのか？
                if (PressedShiftKey != false)
                {
                    if (PressedControlKey == false)
                    {
                        UnselectItem();
                    }

                    if (result.Part.ItemSelectable != false)
                    {
                        OwnerCtrl.SelectCardinalTo(result.Item);
                    }
                    else
                    {
                        OwnerCtrl.SelectCardinalTo(result.Item, result.Name);
                    }

                    OwnerCtrl.Caret(result.Item, result.Name, false);
                    OwnerCtrl.OnSelectChanged();
                    return true;
                }

                //
                if (canEdit != false)
                {
                    //コントロールキーが押されているのか？
                    if (PressedControlKey != false)
                    {
                    }
                    else
                    {

                        if (result.Selected == false)
                        {
                            UnselectItem();
                        }

                        //1クリックで編集可能なのか？
                        if (result.Part.EditByClick != false)
                        {
                            _InterimEditSubItem = true;
                            _CandidateEditItem = result.Item;
                            _CandidateEditName = result.Name;
                            _CandidateEditEditorName = result.Part.EditorName;
                        }

                        //カレットが存在する位置なのか？
                        if (result.Part.EditByClickOnCaret != false &&
                            OwnerCtrl.SameAsCaret(result.Item, result.Name) != false)
                        {
                            _InterimEditSubItem = true;
                            _CandidateEditItem = result.Item;
                            _CandidateEditName = result.Name;
                            _CandidateEditEditorName = result.Part.EditorName;
                        }
                    }
                }

                //アイテムの選択が可能なのか？
                if (result.Part.ItemSelectable != false)
                {
                    _InterimSelectItem = true;
                    _BeginY = GetLogicalY(e.Y);
                    _EndY = _BeginY;
                    _OnSelected = result.Selected;

                    OwnerCtrl.RubberBand(_BeginY, _EndY, PressedControlKey);

                    if (CanPutCaret(result.Name) != false)
                    {
                        OwnerCtrl.CaretMarker(result.Item, result.Name);
                    }
                    else
                    {
                        OwnerCtrl.CaretMarker(result.Item);
                    }
                }
                else
                {

                    _InterimSelectSubItem = true;
                    _BeginIndex = GetLogicalIndex(e.X);
                    _BeginY = GetLogicalY(e.Y);
                    _EndIndex = _BeginIndex;
                    _EndY = _BeginY;
                    _OnSelected = result.Selected;

                    OwnerCtrl.RubberBand(_BeginIndex, _EndIndex, _BeginY, _EndY,
                                          PressedControlKey);

                    if (CanPutCaret(result.Name) != false)
                    {
                        OwnerCtrl.CaretMarker(result.Item, result.Name);
                    }
                }

                //
                if (canEdit != false)
                {
                    //ドラッグが可能なのか？
                    if (OwnerCtrl.AllowDragItem != false &&
                        OwnerCtrl.ReadOnly == false)
                    {

                        if (result.Part.ItemDragable != false && result.Selected != false)
                        {
                            _InterimDrag = true;
                        }
                    }
                }

                //
                _FirstMouseDownPoint.X = e.X;
                _FirstMouseDownPoint.Y = e.Y;

                OwnerCtrl.DemandInvalidate();
                return true;
            }

            ///--------------------------------
            /// <summary>
            /// マウスの左ボタンを離した時の処理
            /// </summary>
            private bool MouseLeftButtonUpOperation(MouseEventArgs e)
            {
                ListCollisionResult result = null;
                bool selectChanged = false;

                if (_InterimEditCaptionDown != false)
                {
                    OwnerCtrl.StartTimer(SystemInformation.DoubleClickTime + 1);
                    _InterimEditCaptionUp = true;
                }

                //サブアイテムの編集開始の可能性があるのか？
                if (_InterimEditSubItem != false)
                {
                    result = OwnerCtrl.Collision(e.X, e.Y);
                    if (result.Name != String.Empty)
                    {

                        if (result.Item == _CandidateEditItem &&
                            result.Name == _CandidateEditName &&
                            result.Part.EditorName == _CandidateEditEditorName)
                        {
                            OwnerCtrl.BeginEdit(_CandidateEditItem,
                                                 _CandidateEditName,
                                                 _CandidateEditEditorName);
                        }
                    }
                }

                //アイテムの選択の可能性があるのか？
                if (_InterimSelectItem != false)
                {
                    _BeginY = GetLogicalY(e.Y);
                    _EndY = _BeginY;

                    PreserveSelectedState();

                    if (PressedShiftKey == false &&
                        PressedControlKey == false)
                    {
                        UnselectItem();
                    }

                    if (PressedControlKey != false)
                    {
                        ToggleSelectItem(_BeginY, _EndY);
                        OwnerCtrl.CaretMarkerTo(false);
                    }
                    else
                    {
                        SelectItem(_BeginY, _EndY);
                        OwnerCtrl.CaretMarkerTo(false);
                    }

                    if (PressedShiftKey == false)
                    {
                        OwnerCtrl.SetCardinalItemByCaret();
                    }

                    selectChanged = IsChangedSelectedState;
                }

                //サブアイテムの選択の可能性があるのか？
                if (_InterimSelectSubItem != false)
                {
                    _BeginY = GetLogicalY(e.Y);
                    _EndY = _BeginY;

                    PreserveSelectedState();

                    if (PressedShiftKey == false &&
                        PressedControlKey == false)
                    {
                        UnselectItem();
                    }

                    if (PressedControlKey != false)
                    {
                        ToggleSelectSubItem(_BeginIndex, _EndIndex, _BeginY, _EndY);
                        OwnerCtrl.CaretMarkerTo(false);
                    }
                    else
                    {
                        SelectSubItem(_BeginIndex, _EndIndex, _BeginY, _EndY);
                        OwnerCtrl.CaretMarkerTo(true);
                    }

                    if (PressedShiftKey == false)
                    {
                        OwnerCtrl.SetCardinalItemByCaret();
                    }

                    selectChanged = IsChangedSelectedState;
                }

                //
                _InterimDrag = false;
                _InterimSelectItem = false;
                _InterimSelectSubItem = false;
                _InterimEditSubItem = false;

                OwnerCtrl.CaretMarker();
                OwnerCtrl.RubberBand();

                if (selectChanged != false)
                {
                    OwnerCtrl.OnSelectChanged();
                    OwnerCtrl.DemandInvalidate();
                }
                else
                {
                    OwnerCtrl.Invalidate();
                }

                return true;
            }

            ///--------------------------------
            /// <summary>
            /// マウスの右ボタンを離した時の処理
            /// </summary>
            private bool MouseRightButtonUpOperation(MouseEventArgs e)
            {
                //ListCollisionResult     result = null;
                bool selectChanged = false;

                //アイテムの選択の可能性があるのか？
                if (_InterimSelectItem != false)
                {
                    if (_OnSelected == false)
                    {
                        _BeginY = GetLogicalY(e.Y);
                        _EndY = _BeginY;

                        if (PressedShiftKey == false &&
                            PressedControlKey == false)
                        {
                            UnselectItem();
                        }

                        if (PressedControlKey != false)
                        {
                            ToggleSelectItem(_BeginY, _EndY);
                            OwnerCtrl.CaretMarkerTo(false);
                        }
                        else
                        {
                            SelectItem(_BeginY, _EndY);
                            OwnerCtrl.CaretMarkerTo(false);
                        }

                        selectChanged = true;
                    }
                    else
                    {

                        OwnerCtrl.CaretMarkerTo(false);
                    }

                    if (PressedShiftKey == false)
                    {
                        OwnerCtrl.SetCardinalItemByCaret();
                    }
                }

                //サブアイテムの選択の可能性があるのか？
                if (_InterimSelectSubItem != false)
                {
                    if (_OnSelected == false)
                    {
                        _BeginY = GetLogicalY(e.Y);
                        _EndY = _BeginY;

                        if (PressedShiftKey == false &&
                            PressedControlKey == false)
                        {
                            UnselectItem();
                        }

                        if (PressedControlKey != false)
                        {
                            ToggleSelectSubItem(_BeginIndex, _EndIndex, _BeginY, _EndY);
                            OwnerCtrl.CaretMarkerTo(false);
                        }
                        else
                        {
                            SelectSubItem(_BeginIndex, _EndIndex, _BeginY, _EndY);
                            OwnerCtrl.CaretMarkerTo(true);
                        }

                        selectChanged = true;
                    }
                    else
                    {

                        OwnerCtrl.CaretMarkerTo(false);
                    }

                    if (PressedShiftKey == false)
                    {
                        OwnerCtrl.SetCardinalItemByCaret();
                    }
                }

                //
                _InterimDrag = false;
                _InterimSelectItem = false;
                _InterimSelectSubItem = false;
                _InterimEditSubItem = false;

                OwnerCtrl.CaretMarker();
                OwnerCtrl.RubberBand();

                if (selectChanged != false)
                {
                    OwnerCtrl.OnSelectChanged();
                    OwnerCtrl.DemandInvalidate();
                }
                else
                {
                    OwnerCtrl.Invalidate();
                }

                return true;
            }

            /// <summary>
            ///
            /// </summary>
            private void MouseLeftButtonDoubleClickedOperation(MouseEventArgs e)
            {
                ListCollisionResult result = OwnerCtrl.Collision(e.X, e.Y);
                string name = result.Name;

                if (name != String.Empty)
                {

                    //ダブルクリックが可能なのか？
                    if (result.Part.DoubleClickable != false)
                    {
                        OwnerCtrl.OnDoubleClicked(name, result.Item);
                    }

                    //ダブルクリックで編集できるのか？
                    if (result.Part.EditByDoubleClick != false)
                    {
                        OwnerCtrl.CaretMarker(result.Item, result.Name);
                        OwnerCtrl.CaretMarkerTo(true);
                        OwnerCtrl.BeginEditByDoubleClick(result.Part.EditorNameByDoubleClick);
                    }

                    _InterimEditSubItem = false;
                    _InterimSelectSubItem = false;

                    OwnerCtrl.CaretMarker();
                    OwnerCtrl.RubberBand();
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private ListItemSelectedDictionary _PreserveDictionary = null;

            private void PreserveSelectedState()
            {
                _PreserveDictionary = OwnerCtrl.GetItemSelecteds().Clone() as
                    ListItemSelectedDictionary;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private bool IsChangedSelectedState
            {
                get
                {
                    ListItemSelectedDictionary dictionary = null;
                    bool result = false;
                    dictionary = OwnerCtrl.GetItemSelecteds();
                    result = !_PreserveDictionary.Compare(dictionary);
                    return result;
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void TrackContextMenu(MouseEventArgs e)
            {
                _InterimEditCaptionDown = false;
                _InterimEditCaptionUp = false;

                OwnerCtrl.OnTrackContextMenu(e);
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// アイテム選択処理状態のオペレータ
        /// </summary>
        private class SelectItemOperator : ListCtrlCoreOperator
        {
            private int _BeginX = -1;
            private int _BeginY = -1;
            private int _EndY = -1;
            private bool _LastControlKey = false;
            private bool _LockToSelectMode; // 選択動作に限定（ドラッグに入らない）
            private bool _InterimDrag; //ドラッグ処理の可能性

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public SelectItemOperator(ListCtrlCore ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _BeginX = GetLogicalX(beginPoint.X);
                _BeginY = GetLogicalY(beginPoint.Y);

                _LockToSelectMode = false;
                _InterimDrag = false;

                ListCollisionResult result = OwnerCtrl.Collision(beginPoint.X, beginPoint.Y);
                if (result.Name != string.Empty && result.Part.ItemDragable)
                {
                    _InterimDrag = true;
                }

                OwnerCtrl.BeginEventLoop();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                _EndY = GetLogicalY(e.Y);

                OwnerCtrl.EndEventLoop();
                OwnerCtrl.RubberBand();

                //
                IListItem[] items = null;
                bool selectChanged = false;

                if (PressedControlKey == false && PressedShiftKey == false)
                {
                    UnselectItem();
                    selectChanged = true;
                }

                Normalize(ref _BeginY, ref _EndY);

                if (PressedControlKey != false)
                {
                    items = ToggleSelectItem(_BeginY, _EndY);
                }
                else
                {
                    items = SelectItem(_BeginY, _EndY);
                }

                if (items.Length > 0)
                {
                    selectChanged = true;
                }

                //
                if (selectChanged != false)
                {
                    OwnerCtrl.OnSelectChanged();
                    OwnerCtrl.DemandInvalidate();
                }
                else
                {

                    OwnerCtrl.Invalidate();
                }

                OwnerCtrl.ChangeOperatorToExchange(new Point(e.X, e.Y));
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                _EndY = GetLogicalY(e.Y);
                OwnerCtrl.RubberBand(_BeginY, _EndY, _LastControlKey);

                OwnerCtrl.Invalidate();

                if (_InterimDrag == true)
                {
                    if (_LockToSelectMode == false)
                    {
                        if (Math.Abs(_BeginY - _EndY) > SystemInformation.DragSize.Height)
                        {
                            _LockToSelectMode = true;
                        }
                    }

                    if (_LockToSelectMode == false)
                    {
                        var currentX = GetLogicalX(e.X);
                        if (Math.Abs(_BeginX - currentX) > SystemInformation.DragSize.Width)
                        {
                            OwnerCtrl.ChangeOperatorToMove(new Point(_BeginX, _BeginY));
                        }
                    }
                }

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnEventLoop(MouseEventArgs e)
            {
                OwnerCtrl.DragScroll(e.X, e.Y);

                //コントロールキーの状態が変化したら再描画を行う
                if (_LastControlKey != PressedControlKey)
                {
                    OwnerCtrl.Invalidate();
                }
                _LastControlKey = PressedControlKey;

                //
                _EndY = GetLogicalY(e.Y);
                OwnerCtrl.RubberBand(_BeginY, _EndY, _LastControlKey);
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// サブアイテム選択処理状態のオペレータ
        /// </summary>
        private class SelectSubItemOperator : ListCtrlCoreOperator
        {
            private int _BeginY = -1;
            private int _BeginIndex = -1;
            private int _EndY = -1;
            private int _EndIndex = -1;
            private bool _LastControlKey = false;

            ///--------------------------------
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public SelectSubItemOperator(ListCtrlCore ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _BeginIndex = GetLogicalIndex(beginPoint.X);
                _BeginY = GetLogicalY(beginPoint.Y);

                OwnerCtrl.BeginEventLoop();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                _EndIndex = GetLogicalIndex(e.X);
                _EndY = GetLogicalY(e.Y);

                if (_EndIndex < 0)
                {
                    _EndIndex = OwnerCtrl.ColumnCount - 1;
                }

                Normalize(ref _BeginIndex, ref _EndIndex);
                Normalize(ref _BeginY, ref _EndY);

                OwnerCtrl.EndEventLoop();
                OwnerCtrl.RubberBand();

                //
                IListItem[] items = null;
                bool selectChanged = false;

                if (PressedControlKey == false && PressedShiftKey == false)
                {
                    UnselectItem();
                    selectChanged = true;
                }

                if (PressedControlKey != false)
                {
                    items = ToggleSelectSubItem(_BeginIndex, _EndIndex, _BeginY, _EndY);
                }
                else
                {
                    items = SelectSubItem(_BeginIndex, _EndIndex, _BeginY, _EndY);
                }

                if (items.Length > 0)
                {
                    selectChanged = true;
                }

                //
                if (selectChanged != false)
                {
                    OwnerCtrl.OnSelectChanged();
                    OwnerCtrl.DemandInvalidate();
                }
                else
                {

                    OwnerCtrl.Invalidate();
                }

                OwnerCtrl.ChangeOperatorToExchange(new Point(e.X, e.Y));
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                _EndIndex = GetLogicalIndex(e.X);
                _EndY = GetLogicalY(e.Y);

                if (_EndIndex < 0)
                {
                    _EndIndex = OwnerCtrl.ColumnCount - 1;
                }

                OwnerCtrl.RubberBand(_BeginIndex, _EndIndex, _BeginY, _EndY, _LastControlKey);
                OwnerCtrl.Invalidate();
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnEventLoop(MouseEventArgs e)
            {
                OwnerCtrl.DragScroll(e.X, e.Y);

                //コントロールキーの状態が変化したら再描画を行う
                if (_LastControlKey != PressedControlKey)
                {
                    OwnerCtrl.Invalidate();
                }
                _LastControlKey = PressedControlKey;

                //
                _EndIndex = GetLogicalIndex(e.X);
                _EndY = GetLogicalY(e.Y);

                if (_EndIndex < 0)
                {
                    _EndIndex = OwnerCtrl.ColumnCount - 1;
                }

                OwnerCtrl.RubberBand(_BeginIndex, _EndIndex, _BeginY, _EndY, _LastControlKey);
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// 移動処理状態のオペレータ
        /// </summary>
        private class MoveOperator : ListCtrlCoreOperator
        {
            private bool _CanMove = false;
            private int _BeginY = -1;

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MoveOperator(ListCtrlCore ctrl) : base(ctrl)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                //開始時の座標で選択する
                //ドラックの開始しただけでは、選択にならない為
                //選択にしないと、ドラックを開始したアイテムが移動の対象にならない
                _BeginY = GetLogicalY(beginPoint.Y);

                //クリックの開始位置に選択されているアイテムが無く、コントロールキーが
                //押されていない場合に、全選択を解除する
                ListCollisionResult result = null;

                result = OwnerCtrl.Collision(beginPoint.X, beginPoint.Y);

                if (result.Name == String.Empty)
                {
                    OwnerCtrl.ChangeOperatorToExchange
                        (GetValidLogialPoint(beginPoint.X, beginPoint.Y));
                    return;
                }

                //
                if (result.Selected == false && PressedControlKey == false)
                {
                    UnselectItem();
                }

                SelectItem(_BeginY);

                _CanMove = OwnerCtrl.CanMoveSelectedItem();

                DragDropEffects effects = DragDropEffects.None;

                effects |= DragDropEffects.Copy;
                effects |= DragDropEffects.Link;
                if (_CanMove != false) { effects |= DragDropEffects.Move; }

                OwnerCtrl.CaretMarker();
                OwnerCtrl.RubberBand();

                OwnerCtrl.TrackDragDrop(effects);

                OwnerCtrl.DragMarker();
                OwnerCtrl.ChangeOperatorToExchange(beginPoint);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                OwnerCtrl.ChangeOperatorToExchange(MousePosition);
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragDrop(DragEventArgs e)
            {
                if (_CanMove == false) { return; }

                //
                Point point;
                ListCollisionResult result = null;

                point = OwnerCtrl.PointToClient(new Point(e.X, e.Y));
                result = OwnerCtrl.CollisionForDrag(point.X, point.Y);

                if (result.Item != null)
                {
                    MoveItem(result.Info, result.InPosition);
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnDragOver(DragEventArgs e)
            {
                if (_CanMove == false)
                {
                    e.Effect = DragDropEffects.None;
                    OwnerCtrl.DragMarker();
                    return;
                }

                Point point;
                ListCollisionResult result = null;

                point = OwnerCtrl.PointToClient(new Point(e.X, e.Y));

                OwnerCtrl.DragScroll(point.X, point.Y);
                result = OwnerCtrl.CollisionForDrag(point.X, point.Y);

                //
                if (result.Item != null)
                {
                    OwnerCtrl.DragMarker(result.Item, result.InPosition);
                }
                else
                {
                    OwnerCtrl.DragMarker();
                }

                OwnerCtrl.Invalidate();
                e.Effect = DragDropEffects.Copy;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnEventLoop(MouseEventArgs e)
            {
                OwnerCtrl.DragScroll(e.X, e.Y);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool MoveItem(IListItemInfo info, ListItemInPosition inPosition)
            {
                switch (inPosition)
                {
                    case ListItemInPosition.Upper:
                        return OwnerCtrl.MoveSelectedItemToPreviousSibling(info.Item);

                    case ListItemInPosition.Lower:
                        return OwnerCtrl.MoveSelectedItemToNextSibling(info.Item);

                    case ListItemInPosition.Center:
                        //TODO: 未実装
                        return false;
                }
                return false;
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public enum ListItemInPosition
    {
        None,
        Upper,
        Lower,
        Center,
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public enum DropPosition
    {
        None,
        Upper,
        Lower,
        Center,
    }
}
