﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Drawing.Imaging;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows;

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListCtrl : UserControl
    {
        private IListItemsSource _ItemsSource = null;
        private ListItemInfo[] _Infos = null;

        private ListImageDictionary _Images = null;
        private InplaceEditorDictionary _InplaceEditors = null;
        private ListPartDrawerDictionary _PartDrawers = null;
        private ListSubItemDictionary _SubItems = null;
        private ListColumnStyleDictionary _ColumnStyles = null;

        private ListItemSelectedDictionary _ItemSelecteds = null;
        private List<ListCtrlCore> _CoreCtrls = new List<ListCtrlCore>();

        private ContextMenuStrip _ContextMenuStrip = null;
        private int _HeaderHeight = 20;

        public event EventHandler SelectChanged;
        public event EventHandler EditBegan;
        public event EventHandler EditEnded;

        private KeyDictionary _EditBeginKeys = new KeyDictionary();
        private KeyDictionary _ImmediateEditBeginKeys = new KeyDictionary();

        public event ListItemDoubleClickedEventHandler ItemDoubleClicked;
        public delegate void ListItemDoubleClickedEventHandler
                             (object sender, ListItemDoubleClickedEventArgs e);

        public event FileDroppedEventHandler QueryFileDropped;
        public event FileDroppedEventHandler FileDropped;
        public delegate void FileDroppedEventHandler(object sender, FileDroppedEventArgs e);

        public event DroppedEventHandler QueryDropped;
        public event DroppedEventHandler Dropped;
        public delegate void DroppedEventHandler(object sender, DroppedEventArgs e);

        public event EventHandler VerticalScrollValueChanged;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListCtrl()
        {
            _Images = new ListImageDictionary();

            _InplaceEditors = new InplaceEditorDictionary();
            _InplaceEditors.Add("Text", new TextInplaceEditor());
            _InplaceEditors.Add("Integer", new IntegerInplaceEditor());
            _InplaceEditors.Add("ULong", new ULongInplaceEditor());
            _InplaceEditors.Add("Decimal", new DecimalInplaceEditor());
            _InplaceEditors.Add("Double", new DoubleInplaceEditor());
            _InplaceEditors.Add("Float", new FloatInplaceEditor());
            _InplaceEditors.Add("FilePath", new FilePathInplaceEditor());
            _InplaceEditors.Add("FilePathDialog", new OpenFilePathDialogInplaceEditor());
            _InplaceEditors.Add("CheckBox", new CheckBoxInplaceEditor());
            _InplaceEditors.Add("DropDown", new DropDownInplaceEditor());
            _InplaceEditors.Add("AssistDropDown", new AssistDropDownInplaceEditor());
            _InplaceEditors.Add("DropDownList", new DropDownListInplaceEditor());

            _PartDrawers = new ListPartDrawerDictionary();
            _PartDrawers.Add(new ListPartDrawer());
            _PartDrawers.Add(new ListPartBlankDrawer());
            _PartDrawers.Add(new ListPartTextDrawer());
            _PartDrawers.Add(new ListPartFilePathDrawer());
            _PartDrawers.Add(new ListPartIntegerDrawer());
            _PartDrawers.Add(new ListPartULongDrawer());
            _PartDrawers.Add(new ListPartDecimalDrawer());
            _PartDrawers.Add(new ListPartDoubleDrawer());
            _PartDrawers.Add(new ListPartFloatDrawer());
            _PartDrawers.Add(new ListPartCheckBoxDrawer());
            _PartDrawers.Add(new ListPartButtonDrawer());
            _PartDrawers.Add(new ListPartSizeDrawer());

            _SubItems = new ListSubItemDictionary();
            _SubItems.Add(new ListSubItem());
            _SubItems.Add(new ListSubItemRowHeader());
            _SubItems.Add(new ListSubItemText());
            _SubItems.Add(new ListSubItemInteger());
            _SubItems.Add(new ListSubItemULong());
            _SubItems.Add(new ListSubItemDecimal());
            _SubItems.Add(new ListSubItemDouble());
            _SubItems.Add(new ListSubItemFloat());
            _SubItems.Add(new ListSubItemFilePath());
            _SubItems.Add(new ListSubItemCheckBox());
            _SubItems.Add(new ListSubItemDropDown());
            _SubItems.Add(new ListSubItemAssistDropDown());
            _SubItems.Add(new ListSubItemDropDownList());
            _SubItems.Add(new ListSubItemSize());

            _ColumnStyles = new ListColumnStyleDictionary();
            _ItemSelecteds = new ListItemSelectedDictionary();
            BackgroundDrawer = new ListBackgroundDrawer();
            ContentDrawer = new ListContentDrawer();

            //
            AddCoreCtrl(this);

            //
            ImmediateEditBeginKeys.Add(Keys.F2);

            //
            AllowDragItem = true;
            FollowSelectedItemToCaret = true;
            InterlockSelectedItemAndSubItem = true;
            StopCaretLeftAndRight = false;
            MoveNewLineCaretLeftAndRight = true;
            DropOnDragItem = false;
            MoveNextLineCaretAfterEndEditByEnter = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override ContextMenuStrip ContextMenuStrip
        {
            get { return null; }
            set { }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ContextMenuStrip AttachedContextMenuStrip
        {
            get { return _ContextMenuStrip; }
            set { _ContextMenuStrip = value; }
        }

        ///--------------------------------
        /// <summary>
        /// 読み取り専用なのか調べる
        /// </summary>
        public bool ReadOnly { get; set; }

        ///--------------------------------
        /// <summary>
        /// アイテムのドラッグの有効/無効
        /// </summary>
        public bool AllowDragItem { get; set; }

        ///--------------------------------
        /// <summary>
        /// 範囲外をクリックした時に選択を解除するか
        /// </summary>
        public bool ClickedOutsideUnselectItem { get; set; }

        ///--------------------------------
        /// <summary>
        /// カレットに選択がついていくのか？
        /// </summary>
        public bool FollowSelectedItemToCaret { get; set; }

        ///--------------------------------
        /// <summary>
        /// サブアイテムを選択した時に、アイテムも選択されるのか？
        /// </summary>
        public bool InterlockSelectedItemAndSubItem { get; set; }

        ///--------------------------------
        /// <summary>
        /// カレットが左右で停止するのか？
        /// </summary>
        public bool StopCaretLeftAndRight { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool MoveNewLineCaretLeftAndRight { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool MoveNextLineCaretAfterEndEditByEnter { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DropOnDragItem { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DecideFileDropPosition { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool SuspendTransaction { get; set; }

        public VScrollBar VerticalScrollBar
        {
            get
            {
                ListCtrlCore coreCtrl = ActiveCoreCtrl;
                return coreCtrl.VerticalScrollBar;
            }
        }

        public HScrollBar HorizontalScrollBar
        {
            get
            {
                ListCtrlCore coreCtrl = ActiveCoreCtrl;
                return coreCtrl.HorizontalScrollBar;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 編集開始キーの取得
        /// </summary>
        public KeyDictionary EditBeginKeys
        {
            get { return _EditBeginKeys; }
        }

        ///--------------------------------
        /// <summary>
        /// 直接編集開始キーの取得
        /// </summary>
        public KeyDictionary ImmediateEditBeginKeys
        {
            get { return _ImmediateEditBeginKeys; }
        }

        ///--------------------------------
        /// <summary>
        /// 直接指定キーが編集開始キーなのか調べる
        /// </summary>
        public bool ContainsImmediateEditBeginKey(Keys key)
        {
            return ImmediateEditBeginKeys.Contains(key);
        }

        ///--------------------------------
        /// <summary>
        /// 指定キーが編集開始キーなのか調べる
        /// </summary>
        public bool ContainsEditBeginKey(Keys key)
        {
            return EditBeginKeys.Contains(key);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListImageDictionary Images
        {
            get { return _Images; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListBackgroundDrawer BackgroundDrawer { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListContentDrawer ContentDrawer { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListPartDrawerDictionary PartDrawers
        {
            get { return _PartDrawers; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public InplaceEditorDictionary InplaceEditors
        {
            get { return _InplaceEditors; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListColumnStyleDictionary ColumnStyles
        {
            get { return _ColumnStyles; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListItemsSource ItemsSource
        {
            get { return _ItemsSource; }
            set
            {
                if (_ItemsSource != null)
                {
                    _ItemsSource.Updated -= OnItemsSourceUpdated;
                }

                _ItemSelecteds.Clear();
                _ItemsSource = value;

                if (_ItemsSource != null)
                {
                    _ItemsSource.Updated += OnItemsSourceUpdated;
                }

                UpdateInfos();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private IHeaderSource _HeaderSource = null;

        public IHeaderSource HeaderSource
        {
            get { return _HeaderSource; }
            set
            {
                _HeaderSource = value;

                foreach (ListCtrlCore coreCtrl in _CoreCtrls)
                {
                    coreCtrl.HeaderSource = _HeaderSource;
                    coreCtrl.UpdateScrollBar();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int HeaderHeight
        {
            get { return _HeaderHeight; }
            set
            {
                _HeaderHeight = value;

                foreach (ListCtrlCore coreCtrl in _CoreCtrls)
                {
                    coreCtrl.ResizeHeaderHeight(_HeaderHeight);
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListItemInfo[] ItemInfos
        {
            get { return _Infos; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListSubItemDictionary SubItems
        {
            get { return _SubItems; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int ItemCount
        {
            get { return _Infos != null ? _Infos.Length : 0; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int ToIndex(IListItem item)
        {
            int index = -1;

            if (_Infos != null)
            {
                foreach (IListItemInfo info in _Infos)
                {
                    index++;
                    if (info.Item == item)
                    {
                        break;
                    }
                }
            }
            return index;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListItem ToItem(int index)
        {
            IListItem item = null;

            if (_Infos != null)
            {
                if (index >= 0 && index < _Infos.Length)
                {
                    item = _Infos[index].Item;
                }
            }
            return item;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ShowByItem(IListItem item)
        {
            ListCtrlCore coreCtrl = ActiveCoreCtrl;
            coreCtrl.ShowItem(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SelectAll()
        {
            if (ItemInfos == null)
            {
                return;
            }

            ListItemSelectedDictionary selectedItems = GetItemSelecteds(null);

            selectedItems.Clear();
            foreach (IListItemInfo info in ItemInfos)
            {
                selectedItems.Select(info.Item, true);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UnselectAll()
        {
            if (ItemInfos == null)
            {
                return;
            }

            ListItemSelectedDictionary selectedItems = GetItemSelecteds(null);
            selectedItems.Clear();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ClearDashLine()
        {
            ListCtrlCore coreCtrl = ActiveCoreCtrl;
            coreCtrl.ClearDashLine();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetDashLineBySelectedSubItem()
        {
            ListCtrlCore coreCtrl = ActiveCoreCtrl;
            coreCtrl.SetDashLineBySelectedSubItem();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListItemSelectedDictionary GetItemSelecteds(ListCtrlCore ctrl)
        {
            return _ItemSelecteds;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListItemSelectedDictionary GetItemSelecteds()
        {
            return GetItemSelecteds(null);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public HeaderCtrl HeaderCtrl
        {
            get
            {
                ListCtrlCore coreCtrl = ActiveCoreCtrl;
                return coreCtrl.HeaderCtrl;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string[] ItemsName
        {
            get
            {
                ListCtrlCore coreCtrl = ActiveCoreCtrl;
                return coreCtrl.HeaderCtrl.ItemsName;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public new void Invalidate()
        {
            DemandInvalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DemandInvalidate()
        {
            foreach (ListCtrlCore coreCtrl in _CoreCtrls)
            {
                coreCtrl.Invalidate();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void InvalidateHeader()
        {
            foreach (ListCtrlCore coreCtrl in _CoreCtrls)
            {
                coreCtrl.HeaderCtrl.Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SplitHorizontal()
        {
            Split(Orientation.Horizontal);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SplitVertical()
        {
            Split(Orientation.Vertical);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListCtrlCore FocusedCoreCtrl
        {
            get
            {
                foreach (ListCtrlCore ctrl in _CoreCtrls)
                {
                    if (ctrl.Focused != false)
                    {
                        return ctrl;
                    }
                }
                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void OnSelectChanged()
        {
            if (SelectChanged != null)
            {
                SelectChanged(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        /// 編集が開始された時に呼ばれる
        /// </summary>
        public void OnEditBegan()
        {
            if (EditBegan != null)
            {
                EditBegan(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        /// 編集が終了した時に呼ばれる
        /// </summary>
        public void OnEditEnded()
        {
            if (EditEnded != null)
            {
                EditEnded(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void OnDoubleClicked(string name, IListItem item)
        {
            if (ItemDoubleClicked != null)
            {
                ItemDoubleClicked(this, new ListItemDoubleClickedEventArgs(item, name));
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual DataObject TrackDragDrop()
        {
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void QueryDragOver(DragEventArgs e, IListItem item, DropPosition dropPosition)
        {
            DroppedEventArgs args = null;

            if (QueryDropped != null)
            {
                args = new DroppedEventArgs(item, dropPosition);
                QueryDropped(this, args);

                e.Effect = args.Cancel != false ?
                    DragDropEffects.None : DragDropEffects.Copy;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ExecuteDragDrop(IListItem item, DropPosition dropPosition)
        {
            DroppedEventArgs args = null;

            if (Dropped != null)
            {
                args = new DroppedEventArgs(item, dropPosition);
                Dropped(this, args);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool CanDropFiles(string[] filePaths, IListItem item)
        {
            FileDroppedEventArgs args = null;

            if (FileDropped != null && QueryFileDropped != null)
            {
                args = new FileDroppedEventArgs(filePaths, item);
                QueryFileDropped(this, args);

                if (args.Cancel == false)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DropFiles(string[] filePaths, IListItem item)
        {
            return DropFiles(filePaths, item, DropPosition.Center);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DropFiles(string[] filePaths, IListItem item, DropPosition dropPosition)
        {
            FileDroppedEventArgs args = null;

            if (FileDropped != null)
            {
                args = new FileDroppedEventArgs(filePaths, item, dropPosition);
                FileDropped(this, args);

                if (args.Cancel == false)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        /// 描画の引数の取得
        /// </summary>
        public virtual IListDrawerArgument GetDrawerArgument(ListColumnStyle style, IListItem item)
        {
            if (style.Argument == null) { return null; }
            return style.Argument.DrawerArgument;
        }

        ///--------------------------------
        /// <summary>
        /// エディタの引数の取得
        /// 動的にエディタに引数を渡したい場合には、これを overrideする
        /// </summary>
        public virtual IInplaceEditorArgument GetEditorArgument(ListColumnStyle style, IListItem item)
        {
            return style.Argument.EditorArgument;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual IParameterValue GetValue(IListItem item, string name)
        {
            if (name == "Blank")
            {
                return new TextParameterValue(String.Empty);
            }

            return item.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual IConstParameterValue GetConstValue(IListItem item, string name)
        {
            if (name == "Blank")
            {
                return new TextParameterValue(String.Empty);
            }

            return item.GetConstValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void SetupDrawDescriptor(IListItem item, string name, ListDrawDescriptor desc)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void InitializeCaret()
        {
            foreach (ListCtrlCore coreCtrl in _CoreCtrls)
            {
                coreCtrl.InitializeCaret();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void GetCaret(out IListItem item, out string columnName)
        {
            ListCtrlCore coreCtrl = ActiveCoreCtrl;
            coreCtrl.GetCaret(out item, out columnName);
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetCaret()
        {
            ListCtrlCore coreCtrl = ActiveCoreCtrl;
            coreCtrl.SetCaret();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetCaret(IListItem item)
        {
            ListCtrlCore coreCtrl = ActiveCoreCtrl;
            coreCtrl.SetCaret(item);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetCaretWithoutFollow(IListItem item, string columnName)
        {
            ListCtrlCore coreCtrl = ActiveCoreCtrl;
            coreCtrl.SetCaret(item, columnName, false);
        }

        /// <summary>
        /// 基準アイテムをカレットで設定
        /// </summary>
        public void SetCardinalItemByCaret()
        {
            ListCtrlCore coreCtrl = ActiveCoreCtrl;
            coreCtrl.SetCardinalItemByCaret();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Sort()
        {
            foreach (ListCtrlCore coreCtrl in _CoreCtrls)
            {
                coreCtrl.Sort();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateInfos()
        {
            //Debug.WriteLine( "-------- Update item info");

            if (ItemsSource == null)
            {
                _Infos = null;
                return;
            }

            _Infos = InfoCreator.Create(ItemsSource.Items.ToArray());

            foreach (ListCtrlCore coreCtrl in _CoreCtrls)
            {
                coreCtrl.UpdatedInfos();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateScrollBar()
        {
            foreach (ListCtrlCore coreCtrl in _CoreCtrls)
            {
                coreCtrl.UpdateScrollBar();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListItem[] SelectedItems
        {
            get
            {
                ListItemSelectedDictionary selectedItems = GetItemSelecteds(null);
                return selectedItems
                    .Where(p => ((ListItemSelectedState)p.Value).Selected != false)
                    .Select(p => p.Key).Cast<IListItem>().ToArray();
            }
        }

        /// <summary>
        /// 表示されているアイテムを取得します。
        /// </summary>
        public IEnumerable<IListItem> GetDisplayedItems()
        {
            return _CoreCtrls.SelectMany(c => c.GetDisplayedItems());
        }

        /// <summary>
        /// 編集を終了します。
        /// </summary>
        public void EndEdit()
        {
            ActiveCoreCtrl.EndEdit();
        }

        /// <summary>
        /// 編集をキャンセルします。
        /// </summary>
        public void CancelEdit()
        {
            ActiveCoreCtrl.CancelEdit();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            if (_CoreCtrls.Count > 0)
            {
                _CoreCtrls[0].Focus();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void UpdateSelectedItems()
        {
            if (_Infos == null)
            {
                _ItemSelecteds.Clear();
                return;
            }

            IListItem[] items = null;

            items = _ItemSelecteds.Keys.Cast<IListItem>().ToArray();

            foreach (IListItem item in items)
            {
                if (ContainsInfo(item) == false)
                {
                    _ItemSelecteds.Remove(item);
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool ContainsInfo(IListItem item)
        {
            if (_Infos != null)
            {
                foreach (ListItemInfo info in _Infos)
                {
                    if (info.Item == item)
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void Split(Orientation orientation)
        {
            SplitContainer container = null;
            Control parent = null;
            ListCtrlCore coreCtrl = null;
            ListCtrlCore newCoreCtrl = null;

            if ((coreCtrl = FocusedCoreCtrl) == null)
            {
                return;
            }

            parent = coreCtrl.Parent;

            container = new SplitContainer();
            container.Orientation = orientation;
            container.Parent = parent;
            container.Dock = DockStyle.Fill;
            container.Show();

            coreCtrl.Parent = container.Panel1;
            newCoreCtrl = AddCoreCtrl(container.Panel2);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListCtrlCore ActiveCoreCtrl
        {
            get
            {
                ListCtrlCore coreCtrl = FocusedCoreCtrl;
                return coreCtrl != null ? coreCtrl : _CoreCtrls[0];
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ListCtrlCore AddCoreCtrl(Control parent)
        {
            Panel panel = null;
            ListCtrlCore coreCtrl = null;
            HeaderCtrl headerCtrl = null;
            int width = 0;
            int height = 0;

            //
            panel = new Panel();
            panel.Parent = parent;
            panel.Dock = DockStyle.Fill;
            panel.Show();

            width = panel.ClientRectangle.Width;
            height = panel.ClientRectangle.Height;

            //
            headerCtrl = new ListHeaderCtrl(this);
            headerCtrl.Parent = panel;
            headerCtrl.Bounds = new Rectangle(0, 0, width, _HeaderHeight);
            headerCtrl.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
            headerCtrl.Show();

            //
            coreCtrl = new ListCtrlCore();
            coreCtrl.Parent = panel;
            coreCtrl.OwnerCtrl = this;
            coreCtrl.HeaderCtrl = headerCtrl;
            coreCtrl.Bounds = new Rectangle(0, _HeaderHeight, width, height - _HeaderHeight);
            coreCtrl.Anchor = (AnchorStyles.Top | AnchorStyles.Bottom |
                                AnchorStyles.Left | AnchorStyles.Right);
            coreCtrl.VerticalScrollValueChanged += OnVerticalScrollValueChanged;
            coreCtrl.Show();

            _CoreCtrls.Add(coreCtrl);

            return coreCtrl;
        }

        private void OnVerticalScrollValueChanged(object sender, EventArgs e)
        {
            this.VerticalScrollValueChanged?.Invoke(sender, e);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnItemsSourceUpdated(object sender, EventArgs e)
        {
            //Debug.WriteLine( "-------- ItemsSource updated");
            UpdateInfos();
            UpdateSelectedItems();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private class InfoCreator
        {
            private static List<ListItemInfo> _List = new List<ListItemInfo>();
            private static int _Index = 0;
            private static int _Y = 0;

            ///
            public static ListItemInfo[] Create(IListItem[] items)
            {
                _List.Clear();
                _Index = 0;
                _Y = 0;

                foreach (IListItem item in items)
                {
                    Create(item);
                }
                return _List.ToArray();
            }

            ///
            private static void Create(IListItem item)
            {
                ListItemInfo info = new ListItemInfo();

                info.Item = item;
                info.Index = _Index;
                info.Y = _Y;

                _List.Add(info);

                _Index++;
                _Y += item.Length;
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// リスト用ヘッダコントロール
    /// </summary>
    public class ListHeaderCtrl : HeaderCtrl
    {
        private ListCtrl _ListCtrl = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListHeaderCtrl(ListCtrl listCtrl)
        {
            _ListCtrl = listCtrl;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanDropFiles(string[] filePaths, object dropTarget)
        {
            return _ListCtrl.CanDropFiles(filePaths, (IListItem)dropTarget);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool DropFiles(string[] filePaths, object dropTarget)
        {
            return _ListCtrl.DropFiles(filePaths, (IListItem)dropTarget);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IListItemInfo
    {
        IListItem Item { get; }
        int Index { get; }
        int Y { get; }
    }

    ///-----------------------------------------------------------------------------
    ///
    public class ListItemInfo : IListItemInfo
    {
        public IListItem Item { get; set; }
        public int Index { get; set; }
        public int Y { get; set; }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListItemSelectedState : ICloneable
    {
        private HashSet<string> _SubSelected = new HashSet<string>();

        public bool Selected { get; set; }
        public HashSet<string> SubSelected
        {
            get { return _SubSelected; }
        }

        ///
        public object Clone()
        {
            ListItemSelectedState state = null;

            state = new ListItemSelectedState();
            state.Selected = this.Selected;
            foreach (string text in SubSelected)
            {
                state.SubSelected.Add(text);
            }
            return state;
        }

        ///
        public bool Compare(ListItemSelectedState state)
        {
            if (state.Selected != Selected ||
                state.SubSelected.Count != SubSelected.Count)
            {
                return false;
            }

            foreach (string text in SubSelected)
            {
                if (state.SubSelected.Contains(text) == false)
                {
                    return false;
                }
            }
            return true;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListSubItemDashLineState : HashSet<string>
    {
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListItemDashLineDictionary : Dictionary<IListItem, ListSubItemDashLineState>
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Mark(IListItem item, string name)
        {
            ListSubItemDashLineState state = GetState(item);
            state.Add(name);
        }
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool IsMark(IListItem item, string name)
        {
            ListSubItemDashLineState state = GetState(item);
            return state.Contains(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListSubItemDashLineState GetState(IListItem item)
        {
            ListSubItemDashLineState state = null;

            if (ContainsKey(item) == false)
            {
                state = new ListSubItemDashLineState();
                this[item] = state;
            }
            else
            {

                state = this[item];
            }
            return state;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListItemSelectedDictionary : Dictionary<IListItem, ListItemSelectedState>, ICloneable
    {
        ///
        public object Clone()
        {
            ListItemSelectedDictionary dictionary = null;
            ListItemSelectedState state = null;

            dictionary = new ListItemSelectedDictionary();

            foreach (IListItem item in Keys)
            {
                state = this[item];
                state = state.Clone() as ListItemSelectedState;
                dictionary.Add(item, state);
            }
            return dictionary;
        }

        ///
        public bool Compare(ListItemSelectedDictionary dictionary)
        {
            ListItemSelectedState state = null;

            if (dictionary.Count != Count)
            {
                return false;
            }

            foreach (IListItem item in Keys)
            {
                state = this[item];

                if (dictionary.ContainsKey(item) == false ||
                    state.Compare(dictionary[item]) == false)
                {
                    return false;
                }
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Select(IListItem item, bool b)
        {
            ListItemSelectedState state = GetState(item);
            state.Selected = b;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ToggleSelect(IListItem item)
        {
            ListItemSelectedState state = GetState(item);
            state.Selected = !state.Selected;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SubSelect(IListItem item, string name, bool b)
        {
            ListItemSelectedState state = GetState(item);

            if (b != false)
            {
                if (state.SubSelected.Contains(name) == false)
                {
                    state.SubSelected.Add(name);
                }
            }
            else
            {
                if (state.SubSelected.Contains(name) != false)
                {
                    state.SubSelected.Remove(name);
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ToggleSubSelect(IListItem item, string name)
        {
            ListItemSelectedState state = GetState(item);

            if (state.SubSelected.Contains(name) != false)
            {
                state.SubSelected.Remove(name);
            }
            else
            {
                state.SubSelected.Add(name);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Selected(IListItem item)
        {
            if (ContainsKey(item) != false)
            {
                return this[item].Selected;
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool SubSelected(IListItem item, string name)
        {
            if (ContainsKey(item) != false)
            {
                if (this[item].SubSelected.Contains(name) != false)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ListItemSelectedState GetState(IListItem item)
        {
            ListItemSelectedState state = null;

            if (ContainsKey(item) == false)
            {
                state = new ListItemSelectedState();
                this[item] = state;
            }
            else
            {

                state = this[item];
            }
            return state;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListColumnStyle
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListColumnStyle(string name) :
            this(name, String.Empty, typeof(string), null, true, true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListColumnStyle(string name, string subItemName, Type subItemType) :
            this(name, subItemName, subItemType, null, true, true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListColumnStyle(string name, string subItemName, Type subItemType, IListSubItemArgument argument) :
            this(name, subItemName, subItemType, argument, true, true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListColumnStyle(string name, string subItemName, Type subItemType, IListSubItemArgument argument, bool canPutCaret, bool canSelect)
        {
            Name = name;
            SubItemName = subItemName;
            Argument = argument;
            SubItemType = subItemType;
            CanPutCaret = canPutCaret;
            CanSelect = canSelect;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name { get; set; }
        public string SubItemName { get; set; }
        public IListSubItemArgument Argument { get; set; }

        public Type SubItemType { get; set; }

        public bool CanPutCaret { get; set; }
        public bool CanSelect { get; set; }
    }

    /// <summary>
    ///
    /// </summary>
    public class ListPartDrawerDictionary : Dictionary<string, IListPartDrawer>
    {
        /// <summary>
        /// ListPartDrawerを追加します。
        /// </summary>
        public void Add(IListPartDrawer drawer)
        {
            Add(drawer.Name, drawer);
        }

        /// <summary>
        /// ListPartDrawerを差し替えます。
        /// </summary>
        public void Replace(IListPartDrawer drawer)
        {
            if (ContainsKey(drawer.Name) == true)
            {
                Remove(drawer.Name);
            }
            Add(drawer);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class ListSubItemDictionary : Dictionary<string, IListSubItem>
    {
        /// <summary>
        /// ListSubItemを追加します。
        /// </summary>
        public void Add(IListSubItem subItem)
        {
            Add(subItem.Name, subItem);
        }

        /// <summary>
        /// ListSubItemを差し替えます。
        /// </summary>
        public void Replace(IListSubItem subItem)
        {
            if (ContainsKey(subItem.Name) == true)
            {
                Remove(subItem.Name);
            }
            Add(subItem);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListColumnStyleDictionary : Dictionary<string, ListColumnStyle>
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Add(ListColumnStyle style)
        {
            Add(style.Name, style);
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListImageDictionary : Dictionary<string, Image>
    {
    }

    ///------------------------------------------------------------------------
    /// <summary>
    /// 衝突判定の結果を保存クラス
    /// </summary>
    public class ListCollisionResult
    {
        private IListItemInfo _Info = null;
        private IListPart _Part = null;

        private string _Name = String.Empty;
        private ListItemInPosition _InPosition = ListItemInPosition.None;
        private bool _Selected = false;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListCollisionResult() : this(null, null, String.Empty)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListCollisionResult(IListItemInfo info, IListPart part, string name) :
            this(info, part, name, false)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListCollisionResult(IListItemInfo info, IListPart part, ListItemInPosition inPosition)
        {
            _Info = info;
            _Part = part;

            _InPosition = inPosition;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListCollisionResult(IListItemInfo info, IListPart part, string name, bool selected)
        {
            _Info = info;
            _Part = part;
            _Name = name;
            _Selected = selected;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListItemInfo Info
        {
            get { return _Info; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListItem Item
        {
            get { return _Info != null ? _Info.Item : null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListPart Part
        {
            get { return _Part; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name { get { return _Name; } }

        public ListItemInPosition InPosition { get { return _InPosition; } }
        public bool Selected { get { return _Selected; } }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// ダブルクリックした時のイベント引数
    /// </summary>
    public class ListItemDoubleClickedEventArgs : EventArgs
    {
        private IListItem _Item = null;
        private string _Name = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListItemDoubleClickedEventArgs(IListItem item, string name)
        {
            _Item = item;
            _Name = name;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListItem Item
        {
            get { return _Item; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return _Name; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    /// カレットが移動した時のイベント引数
    /// </summary>
    public class ListCaretMovedEventArgs
    {
        private IListItem _Item = null;
        private string _Name = String.Empty;
        private bool _Moved = false;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ListCaretMovedEventArgs(IListItem item, string name, bool moved)
        {
            _Item = item;
            _Name = name;
            _Moved = moved;
        }

        ///--------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ListCaretMovedEventArgs(IListItem item, string name) : this(item, name, true)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IListItem Item
        {
            get { return _Item; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return _Name; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool Moved
        {
            get { return _Moved; }
        }
    }
}
