﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Linq;
    using System.Text;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;

    public class ComponentSampleMapAdapter : ISampleMapItemsSource, IDisposable
    {
        private ComponentKeyRegionCollection _Items = null;
        public event EventHandler Updated;
        public event OperationExecutedEventHandler OperationExecuted;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentSampleMapAdapter()
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Dispose()
        {
            this.AdapterClear();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentList Children { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IKeyRegionCollection Items
        {
            get { return _Items; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public OperationHistory OperationHistory { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ExistBlankKeyRegion
        {
            get
            {
                bool returnValue = false;

                if (_Items == null)
                {
                    returnValue = true;
                }
                else
                {
                    int min = 0;
                    foreach (IKeyRegion keyRegion in _Items)
                    {
                        if (keyRegion.Minimum != min)
                        {
                            returnValue = true;
                            goto Return;
                        }
                        else
                        {
                            min = keyRegion.Maximum + 1;
                        }
                    }
                    if (min != 128)
                    {
                        returnValue = true;
                    }
                }

                Return:
                return returnValue;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetupSampleMap(ComponentList keyResionCollection)
        {
            if (Children == keyResionCollection)
            {
                return;
            }
            this.AdapterClear();
            Children = keyResionCollection;
            if (keyResionCollection != null)
            {
                _Items = new ComponentKeyRegionCollection();
                _Items.CollectionChanged += OnCollectionChanged;

                Children.CollectionChanged += OnComponentCollectionChanged;
                {
                    _Items.SuspendKeyRegionCollectionEvent();
                    try
                    {
                        Items.Clear();

                        foreach (KeyRegion keyRegion in keyResionCollection)
                        {
                            ComponentKeyRegion comKeyRegion = CreateKeyRegion(keyRegion);
                            Items.Add(comKeyRegion);
                        }
                    }
                    finally
                    {
                        _Items.ResumeKeyRegionCollectionEvent();
                    }
                }
            }
            else
            {
                _Items = null;
            }
            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public IKeyRegion CreateKeyRegion(int index, int minimum, int maximum, string filePath)
        {
            IKeyRegion kRegion = CreateKeyRegion(minimum, maximum);
            IVelocityRegion vRegion = CreateVelocityRegion(filePath);
            kRegion.Children.Add(vRegion);
            Items.Insert(index, kRegion);

            return kRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentVelocityRegion CreateVelocityRegion(VelocityRegion velRegion,
                                                             ComponentKeyRegion comKeyRegion)
        {
            ComponentVelocityRegion comVelRegion = new ComponentVelocityRegion(velRegion);
            comVelRegion.MinimumIntact = velRegion.VelocityMin;
            comVelRegion.MaximumIntact = velRegion.VelocityMax;
            comVelRegion.FilePath = velRegion.FilePath;
            comVelRegion.OriginalKey = velRegion.OriginalKey;
            comVelRegion.WaveEncoding = velRegion.Encoding;
            comVelRegion.Parent = comKeyRegion;

            return comVelRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DivideKeyRegion(IKeyRegion target)
        {
            int length = target.Maximum - target.Minimum;
            if (length > 0)
            {
                foreach (IKeyRegion kRegion in Items)
                {
                    if (target == kRegion)
                    {
                        divideKeyRegion(kRegion);
                        break;
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void DivideVelocityRegion(IVelocityRegion target)
        {
            int length = target.Maximum - target.Minimum;
            if (length > 0)
            {
                divideVelocityRegion(target.Parent, target);
            }
        }

        public IVelocityRegion GetPrevVelocityRegion(IVelocityRegion velRegion)
        {
            IVelocityRegion prevVelRegion = null;

            foreach (IKeyRegion kRegion in Items)
            {
                if (kRegion.Children.Contains(velRegion) == true)
                {
                    int index = kRegion.Children.IndexOf(velRegion);
                    --index;
                    if (0 <= index)
                    {
                        prevVelRegion = kRegion.Children[index];
                    }
                    break;
                }
            }

            return prevVelRegion;
        }

        public IVelocityRegion GetNextVelocityRegion(IVelocityRegion velRegion)
        {
            IVelocityRegion nextVelRegion = null;

            foreach (IKeyRegion kRegion in Items)
            {
                if (kRegion.Children.Contains(velRegion) == true)
                {
                    int index = kRegion.Children.IndexOf(velRegion);
                    ++index;
                    if (index < kRegion.Children.Count)
                    {
                        nextVelRegion = kRegion.Children[index];
                    }
                    break;
                }
            }

            return nextVelRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CopyKeyRegion(IKeyRegion iKeyRegion0, IKeyRegion iKeyRegion1)
        {
            KeyRegion keyRegion0 = (KeyRegion)(iKeyRegion0.Target);
            KeyRegion keyRegion1 = (KeyRegion)(iKeyRegion1.Target);
            KeyRegion newKeyRegion = DuplicateKeyRegion(keyRegion0);
            newKeyRegion.KeyMin = keyRegion1.KeyMin;
            newKeyRegion.KeyMax = keyRegion1.KeyMax;

            Instrument instument = (Instrument)(keyRegion0.Parent);
            int index = instument.Children.IndexOf(keyRegion1);

            try
            {
                BeginTransaction();
                RemoveComponent(keyRegion1);
                if (iKeyRegion1 is ComponentKeyRegion)
                {
                    ComponentKeyRegion componentKeyRegion1 = iKeyRegion1 as ComponentKeyRegion;
                    componentKeyRegion1.Dispose();
                }
                InsertComponent(instument, index, newKeyRegion);
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CopyKeyRegion(IKeyRegion iKeyRegion0, int keyMin, int keyMax)
        {
            KeyRegion keyRegion0 = (KeyRegion)(iKeyRegion0.Target);
            KeyRegion newKeyRegion = DuplicateKeyRegion(keyRegion0);
            newKeyRegion.KeyMin = keyMin;
            newKeyRegion.KeyMax = keyMax;

            Instrument instument = (Instrument)(keyRegion0.Parent);
            int index = GetIndexByMaximum(instument, keyMax);

            try
            {
                BeginTransaction();
                InsertComponent(instument, index, newKeyRegion);
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CopyVelocityRegion(IVelocityRegion iVelRegion0, IVelocityRegion iVelRegion1)
        {
            VelocityRegion velRegion0 = (VelocityRegion)(iVelRegion0.Target);
            VelocityRegion velRegion1 = (VelocityRegion)(iVelRegion1.Target);
            VelocityRegion newVelRegion = DuplicateVelocityRegion(velRegion0);
            newVelRegion.VelocityMin = velRegion1.VelocityMin;
            newVelRegion.VelocityMax = velRegion1.VelocityMax;

            KeyRegion keyRegion1 = (KeyRegion)(velRegion1.Parent);
            int index = keyRegion1.Children.IndexOf(velRegion1);

            try
            {
                BeginTransaction();
                RemoveComponent(velRegion1);
                if (iVelRegion1 is ComponentVelocityRegion)
                {
                    ComponentVelocityRegion componentVelocityRegion1 =
                        iVelRegion1 as ComponentVelocityRegion;
                    componentVelocityRegion1.Dispose();
                }
                InsertComponent(keyRegion1, index, newVelRegion);
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CopyVelocityRegion(IVelocityRegion iVelRegion0, int keyMin, int keyMax)
        {
            VelocityRegion velRegion0 = (VelocityRegion)(iVelRegion0.Target);
            VelocityRegion newVelRegion = DuplicateVelocityRegion(velRegion0);
            KeyRegion newKeyRegion = new KeyRegion();
            newKeyRegion.KeyMin = keyMin;
            newKeyRegion.KeyMax = keyMax;
            newVelRegion.VelocityMin = 0;
            newVelRegion.VelocityMax = 127;
            newKeyRegion.Children.Add(newVelRegion);

            Instrument instument = (Instrument)(velRegion0.Parent.Parent);
            int index = GetIndexByMaximum(instument, keyMax);

            try
            {
                BeginTransaction();
                InsertComponent(instument, index, newKeyRegion);
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SwapKeyRegion(IKeyRegion iKeyRegion0, IKeyRegion iKeyRegion1)
        {
            KeyRegion kRegion0 = (KeyRegion)(iKeyRegion0.Target);
            KeyRegion kRegion1 = (KeyRegion)(iKeyRegion1.Target);

            Instrument Parent = (Instrument)(kRegion0.Parent);
            ComponentList Children = Parent.Children;

            int min0 = kRegion0.KeyMin;
            int max0 = kRegion0.KeyMax;
            int min1 = kRegion1.KeyMin;
            int max1 = kRegion1.KeyMax;

            int index0 = Children.IndexOf(kRegion0);
            int index1 = Children.IndexOf(kRegion1);

            try
            {
                BeginTransaction();
                RemoveComponent(kRegion0);
                RemoveComponent(kRegion1);
                SetValue(kRegion0, ProjectParameterNames.KeyRegion.KeyMin, min1);
                SetValue(kRegion0, ProjectParameterNames.KeyRegion.KeyMax, max1);
                SetValue(kRegion1, ProjectParameterNames.KeyRegion.KeyMin, min0);
                SetValue(kRegion1, ProjectParameterNames.KeyRegion.KeyMax, max0);

                if (index0 < index1)
                {
                    InsertComponent(Parent, index0, kRegion1);
                    InsertComponent(Parent, index1, kRegion0);
                }
                else
                {
                    InsertComponent(Parent, index1, kRegion0);
                    InsertComponent(Parent, index0, kRegion1);
                }
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ChangeKeyRegionValue(IKeyRegion iKeyRegion, int min, int max)
        {
            KeyRegion kRegion = (KeyRegion)(iKeyRegion.Target);
            Instrument instrument = (Instrument)(kRegion.Parent);
            int keyRegionIndex = instrument.Children.IndexOf(kRegion);
            int index = GetIndexByMaximum(instrument, max);
            if (keyRegionIndex < index)
            {
                --index;
            }

            try
            {
                BeginTransaction();
                RemoveComponent(kRegion);
                SetValue(kRegion, ProjectParameterNames.KeyRegion.KeyMin, min);
                SetValue(kRegion, ProjectParameterNames.KeyRegion.KeyMax, max);
                InsertComponent(instrument, index, kRegion);
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SwapVelocityRegion(IVelocityRegion iVelRegion0, IVelocityRegion iVelRegion1)
        {
            VelocityRegion velRegion0 = (VelocityRegion)(iVelRegion0.Target);
            VelocityRegion velRegion1 = (VelocityRegion)(iVelRegion1.Target);
            KeyRegion keyRegion0 = (KeyRegion)(velRegion0.Parent);
            KeyRegion keyRegion1 = (KeyRegion)(velRegion1.Parent);
            int index0 = keyRegion0.Children.IndexOf(velRegion0);
            int index1 = keyRegion1.Children.IndexOf(velRegion1);
            int min0 = velRegion0.VelocityMin;
            int max0 = velRegion0.VelocityMax;
            int min1 = velRegion1.VelocityMin;
            int max1 = velRegion1.VelocityMax;

            try
            {
                BeginTransaction();
                RemoveComponent(velRegion0);
                RemoveComponent(velRegion1);
                SetValue(velRegion0, ProjectParameterNames.VelocityRegion.VelocityMin, min1);
                SetValue(velRegion0, ProjectParameterNames.VelocityRegion.VelocityMax, max1);
                SetValue(velRegion1, ProjectParameterNames.VelocityRegion.VelocityMin, min0);
                SetValue(velRegion1, ProjectParameterNames.VelocityRegion.VelocityMax, max0);

                if (index0 < index1)
                {
                    InsertComponent(keyRegion0, index0, velRegion1);
                    InsertComponent(keyRegion1, index1, velRegion0);
                }
                else
                {
                    InsertComponent(keyRegion1, index1, velRegion0);
                    InsertComponent(keyRegion0, index0, velRegion1);
                }
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void RemoveAndCreateVelocityRegion(IVelocityRegion iVelRegion,
                                                   int keyMin,
                                                   int keyMax)
        {
            VelocityRegion velRegion = (VelocityRegion)(iVelRegion.Target);
            Instrument instrument = (Instrument)(velRegion.Parent.Parent);
            KeyRegion keyRegion = null;

            try
            {
                BeginTransaction();
                ComponentList Children = velRegion.Parent.Children;
                if (Children.Count == 1)
                {
                    keyRegion = (KeyRegion)(velRegion.Parent);
                    RemoveComponent(keyRegion);
                    SetValue(keyRegion, ProjectParameterNames.KeyRegion.KeyMin, keyMin);
                    SetValue(keyRegion, ProjectParameterNames.KeyRegion.KeyMax, keyMax);
                }
                else
                {
                    keyRegion = new KeyRegion();
                    keyRegion.KeyMin = keyMin;
                    keyRegion.KeyMax = keyMax;

                    int index = Children.IndexOf(velRegion);
                    if (velRegion.VelocityMin == 0)
                    {
                        SetValue((VelocityRegion)(Children[index + 1]),
                                  ProjectParameterNames.VelocityRegion.VelocityMin,
                                  velRegion.VelocityMin);
                    }
                    else
                    {
                        SetValue((VelocityRegion)(Children[index - 1]),
                                  ProjectParameterNames.VelocityRegion.VelocityMax,
                                  velRegion.VelocityMax);
                    }
                    RemoveComponent(velRegion);
                    SetValue(velRegion, ProjectParameterNames.VelocityRegion.VelocityMin, 0);
                    SetValue(velRegion, ProjectParameterNames.VelocityRegion.VelocityMax, 127);
                    InsertComponent(keyRegion, 0, velRegion);
                }

                int i = GetIndexByMaximum(instrument, keyMax);
                InsertComponent(instrument, i, keyRegion);
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void BeginTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.BeginTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void EndTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.EndTransaction();
            }

            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CancelTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.CancelTransaction();
            }
        }

        public virtual void SetValue(Component component, string name, object value)
        {
            if (component.Parameters.ContainsKey(name) == false)
            {
                return;
            }

            SetParameterOperation operation =
                new SetParameterOperation(component.Parameters, name, value);
            operation.Execute();
            OnOperationExecuted(operation);
        }


        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Update()
        {
            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void InsertComponent(Component parent, int index, Component item)
        {
            InsertComponentOperation operation = null;
            operation = new InsertComponentOperation(parent, index, item);
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void RemoveComponent(Component item)
        {
            RemoveComponentOperation operation = null;

            operation = new RemoveComponentOperation(new[] { item });
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnOperationExecuted(Operation operation)
        {
            OnOperationExecuted(new OperationExecutedEventArgs(operation));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnOperationExecuted(OperationExecutedEventArgs e)
        {
            if (OperationHistory != null)
            {
                OperationHistory.AddOperation(e.Operation);
            }

            if (OperationExecuted != null)
            {
                OperationExecuted(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnUpdated()
        {
            OnUpdated(new EventArgs());
        }

        private void AdapterClear()
        {
            if (this.Children != null)
            {
                this.Children.CollectionChanged -= OnComponentCollectionChanged;
                this.Children = null;
            }
            if (_Items != null)
            {
                foreach (ComponentKeyRegion componentKeyRegion in _Items)
                {
                    componentKeyRegion.Dispose();
                }
                _Items = null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int GetIndexByMaximum(Instrument instrument, int max)
        {
            int index = 0;

            foreach (KeyRegion keyRegion in instrument.Children)
            {
                if (max < keyRegion.KeyMin) break;
                else ++index;
            }

            return index;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private KeyRegion DuplicateKeyRegion(KeyRegion keyRegion)
        {
            KeyRegion newKeyRegion = new KeyRegion();
            newKeyRegion.KeyMin = keyRegion.KeyMin;
            newKeyRegion.KeyMax = keyRegion.KeyMax;

            foreach (VelocityRegion velRegion in keyRegion.Children)
            {
                newKeyRegion.Children.Add(DuplicateVelocityRegion(velRegion));
            }

            return newKeyRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private VelocityRegion DuplicateVelocityRegion(VelocityRegion velRegion)
        {
            VelocityRegion newVelRegion = new VelocityRegion();

            newVelRegion.FilePath = velRegion.FilePath;
            newVelRegion.Encoding = velRegion.Encoding;
            newVelRegion.OriginalKey = velRegion.OriginalKey;
            newVelRegion.Envelope = velRegion.Envelope;
            newVelRegion.VelocityMin = velRegion.VelocityMin;
            newVelRegion.VelocityMax = velRegion.VelocityMax;
            newVelRegion.Volume = velRegion.Volume;
            newVelRegion.Pan = velRegion.Pan;
            newVelRegion.PitchSemitones = velRegion.PitchSemitones;
            newVelRegion.PitchCents = velRegion.PitchCents;
            newVelRegion.PercussionMode = velRegion.PercussionMode;
            newVelRegion.KeyGroup = velRegion.KeyGroup;
            newVelRegion.InterpolationType = velRegion.InterpolationType;
            newVelRegion.DataSize = velRegion.DataSize;

            return newVelRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private IKeyRegion CreateKeyRegion(int minimum, int maximum)
        {
            KeyRegion nwKeyRegion = new KeyRegion();
            nwKeyRegion.KeyMin = minimum;
            nwKeyRegion.KeyMax = maximum;
            ComponentKeyRegion kRegion = new ComponentKeyRegion(nwKeyRegion);
            kRegion.MinimumIntact = minimum;
            kRegion.MaximumIntact = maximum;
            kRegion.Parent = this;

            return kRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentKeyRegion CreateKeyRegion(KeyRegion keyRegion)
        {
            ComponentKeyRegion comKeyRegion = new ComponentKeyRegion(keyRegion);
            comKeyRegion.MinimumIntact = keyRegion.KeyMin;
            comKeyRegion.MaximumIntact = keyRegion.KeyMax;
            comKeyRegion.Parent = this;

            SetupVelocityRegion(comKeyRegion, keyRegion);

            return comKeyRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SetupVelocityRegion(ComponentKeyRegion comKeyRegion, KeyRegion keyRegion)
        {
            ((ComponentVelocityRegionCollection)(comKeyRegion.Children)).
                SuspendVelocityRegionCollectionEvent();
            try
            {
                foreach (VelocityRegion velRegion in keyRegion.Children)
                {
                    ComponentVelocityRegion comVelRegion =
                        CreateVelocityRegion(velRegion, comKeyRegion);
                    comKeyRegion.Children.Add(comVelRegion);
                }
            }
            finally
            {
                ((ComponentVelocityRegionCollection)(comKeyRegion.Children)).
                    ResumeVelocityRegionCollectionEvent();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void divideKeyRegion(IKeyRegion kRegion)
        {
            try
            {
                BeginTransaction();
                int index = Items.IndexOf(kRegion);
                int length = kRegion.Maximum - kRegion.Minimum;
                int diff = (length + 1) / 2;

                IKeyRegion newKRegion = CreateKeyRegion(kRegion.Minimum,
                                                         kRegion.Maximum - diff);
                foreach (IVelocityRegion vRegion in kRegion.Children)
                {
                    IVelocityRegion newVRegion = CloneVelocityRegion(vRegion);
                    newKRegion.Children.Add(newVRegion);
                }
                if (kRegion.Maximum - diff + 1 > kRegion.Maximum)
                {
                    kRegion.MinimumIntact = kRegion.Maximum;
                }
                else
                {
                    kRegion.MinimumIntact = kRegion.Maximum - diff + 1;
                }
                kRegion.MaximumIntact = kRegion.Maximum;
                Items.Insert(index, newKRegion);
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void divideVelocityRegion(IKeyRegion kRegion, IVelocityRegion vRegion)
        {
            try
            {
                BeginTransaction();
                int index = kRegion.Children.IndexOf(vRegion);
                int length = vRegion.Maximum - vRegion.Minimum;
                int diff = (length + 1) / 2;

                IVelocityRegion divideVel = CloneVelocityRegion(vRegion);
                kRegion.Children.Insert(index, divideVel);
                divideVel.MinimumIntact = vRegion.Minimum;
                divideVel.MaximumIntact = vRegion.Maximum - diff;
                if (vRegion.Maximum - diff + 1 > vRegion.Maximum)
                {
                    vRegion.MinimumIntact = vRegion.Maximum;
                }
                else
                {
                    vRegion.MinimumIntact = vRegion.Maximum - diff + 1;
                }
                vRegion.MaximumIntact = vRegion.Maximum;
                EndTransaction();
            }
            catch
            {
                CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private IVelocityRegion CreateVelocityRegion(string filePath)
        {
            VelocityRegion nwVelRegion = new VelocityRegion();
            nwVelRegion.VelocityMin = 0;
            nwVelRegion.VelocityMax = 127;
            nwVelRegion.FilePath = filePath;

            IVelocityRegion velRegion = new ComponentVelocityRegion(nwVelRegion);
            velRegion.MinimumIntact = nwVelRegion.VelocityMin;
            velRegion.MaximumIntact = nwVelRegion.VelocityMax;
            velRegion.FilePath = nwVelRegion.FilePath;

            return velRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private IVelocityRegion CloneVelocityRegion(IVelocityRegion vRegion)
        {
            VelocityRegion srcVelRegion = vRegion.Target as VelocityRegion;
            VelocityRegion newVelRegion = null;
            IVelocityRegion iVelRegion = null;

            newVelRegion = srcVelRegion.Clone();

            iVelRegion = new ComponentVelocityRegion(newVelRegion);
            iVelRegion.MinimumIntact = newVelRegion.VelocityMin;
            iVelRegion.MaximumIntact = newVelRegion.VelocityMax;
            iVelRegion.FilePath = newVelRegion.FilePath;
            iVelRegion.WaveEncoding = newVelRegion.Encoding;
            iVelRegion.OriginalKey = newVelRegion.OriginalKey;

            return iVelRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentKeyRegion GetKeyRegionByModel(KeyRegion keyRegion)
        {
            ComponentKeyRegion comKeyRegion = null;

            foreach (ComponentKeyRegion item in Items)
            {
                if (item.Target == keyRegion)
                {
                    comKeyRegion = item;
                    break;
                }
            }

            return comKeyRegion;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnUpdated(EventArgs e)
        {
            if (Updated != null)
            {
                Updated(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    if (e.NewItems.Count > 0 &&
                        e.NewItems[0] is ComponentKeyRegion)
                    {
                        ComponentKeyRegion keyRegion = (ComponentKeyRegion)e.NewItems[0];
                        keyRegion.Parent = this;
                        InsertComponent(Children.Owner, e.NewStartingIndex, keyRegion.Target);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    if (e.OldItems.Count > 0 &&
                         e.OldItems[0] is ComponentKeyRegion)
                    {
                        ComponentKeyRegion keyRegion = (ComponentKeyRegion)e.OldItems[0];
                        RemoveComponent(keyRegion.Target);
                    }
                    break;
            }

            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnComponentCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (_Items != null)
            {
                _Items.SuspendKeyRegionCollectionEvent();
                try
                {
                    switch (e.Action)
                    {
                        case NotifyCollectionChangedAction.Add:
                            if (e.NewItems.Count > 0 &&
                                e.NewItems[0] is KeyRegion)
                            {
                                KeyRegion keyRegion = (KeyRegion)e.NewItems[0];

                                ComponentKeyRegion region = GetKeyRegionByModel(keyRegion);
                                if (region == null)
                                {
                                    region = CreateKeyRegion(keyRegion);
                                    Items.Insert(e.NewStartingIndex, region);
                                }
                            }
                            break;

                        case NotifyCollectionChangedAction.Remove:
                            if (e.OldItems.Count > 0 &&
                                 e.OldItems[0] is KeyRegion)
                            {
                                KeyRegion keyRegion = (KeyRegion)e.OldItems[0];

                                ComponentKeyRegion region = GetKeyRegionByModel(keyRegion);
                                if (region != null)
                                {
                                    Items.Remove(region);
                                }
                            }
                            break;
                    }
                }
                finally
                {
                    _Items.ResumeKeyRegionCollectionEvent();
                }

                OnUpdated();
            }
        }
    }
}
