﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;

    public class ComponentListAdapter : IListItemsSource
    {
        private ComponentListItemCollection _Items = null;
        private ComponentList _ComponentsOwnerList = null;
        private ComponentListItem[] originalOrderItems = null;

        private bool _SuspendUpdatedEvent = false;
        private bool _UpdatedEventWasIgnored = false;

        private bool _SuspendCollectionChanged = false;
        private bool _SuspendComponentCollectionChanged = false;

        public event EventHandler Updated;
        public event OperationExecutedEventHandler OperationExecuted;

        /// <summary>
        ///
        /// </summary>
        public ComponentListAdapter()
        {
            InterlockComponent = true;

            this._Items = new ComponentListItemCollection();
            this._Items.CollectionChanged += OnCollectionChanged;
        }

        /// <summary>
        ///
        /// </summary>
        public Component ItemsOwner
        {
            get
            {
                return this._ComponentsOwnerList != null ? this._ComponentsOwnerList.Owner : null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public ComponentList OwnerItemList
        {
            get
            {
                return this._ComponentsOwnerList;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void AddItems(ComponentList list)
        {
            RemoveComponentCollectionChangedEvent(this._ComponentsOwnerList);

            try
            {
                InterlockComponent = false;

                Items.Clear();
                this.originalOrderItems = null;

                if (list != null)
                {
                    foreach (Component component in list)
                    {
                        Items.Add(CreateListItem(component));
                    }
                }
            }
            finally
            {
                InterlockComponent = true;
            }

            this._ComponentsOwnerList = list;

            AddComponentCollectionChangedEvent(this._ComponentsOwnerList);
        }

        /// <summary>
        ///
        /// </summary>
        public void AddItems(Component[] components)
        {
            RemoveComponentCollectionChangedEvent(this._ComponentsOwnerList);

            //
            foreach (Component component in components)
            {
                Items.Add(CreateListItem(component));
            }

            // 渡された Componentの順番を記録しておきます。
            // ソートにおいて渡された順番を復元するのに使われます。
            this.originalOrderItems = new ComponentListItem[components.Length];
            Items.CopyTo(this.originalOrderItems, 0);

            //
            this._ComponentsOwnerList = null;
        }

        /// <summary>
        ///
        /// </summary>
        public void RemoveAllItems()
        {
            AddItems((ComponentList)null);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ComponentListItemCollection Items
        {
            get { return _Items; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        IListItemCollection IListItemsSource.Items
        {
            get { return Items; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool InterlockComponent { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void BeginTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.BeginTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void EndTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.EndTransaction();
            }

            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void CancelTransaction()
        {
            if (OperationHistory != null)
            {
                OperationHistory.CancelTransaction();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public OperationHistory OperationHistory { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void Sort(string name, Type type, SortOrder sortOrder)
        {
            ComponentListItem[] items = Items.ToArray();

            if (items.Length <= 0)
            {
                return;
            }

            //
            switch (sortOrder)
            {
                case SortOrder.Ascending:
                    items = Sort(items, CreateComparer(name, type, true));
                    break;

                case SortOrder.Descending:
                    items = Sort(items, CreateComparer(name, type, false));
                    break;

                case SortOrder.None:
                    items = SortByOriginalOrder(items);
                    break;
            }

            try
            {
                InterlockComponent = false;

                Items.Clear();
                foreach (IListItem item in items)
                {
                    Items.Add(item);
                }
            }
            finally
            {
                InterlockComponent = true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void SetValue(Component targetComponent, string name, object value)
        {
            //Operation: ここで Operationにする

            if (targetComponent.Parameters.ContainsKey(name) == false)
            {
                return;
            }

            SetParameterOperation operation = null;
            operation = new SetParameterOperation(targetComponent.Parameters, name, value);
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SuspendUpdatedEvent()
        {
            _SuspendUpdatedEvent = true;
            _UpdatedEventWasIgnored = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ResumeUpdatedEvent()
        {
            _SuspendUpdatedEvent = false;

            if (_UpdatedEventWasIgnored != false)
            {
                OnUpdated();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual ComponentListItem CreateListItem(Component component)
        {
            return new ComponentListItem(component);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual ListComparer CreateComparer(string name, Type type, bool ascending)
        {
            return new ListComparer(name, type, ascending);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ComponentListItem GetNextSibling(ComponentListItem item)
        {
            int index = _Items.IndexOf(item) + 1;

            if (index >= _Items.Count)
            {
                return null;
            }
            return _Items[index] as ComponentListItem;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnOperationExecuted(Operation operation)
        {
            if (OperationHistory != null)
            {
                OperationHistory.AddOperation(operation);
            }

            OnOperationExecuted(new OperationExecutedEventArgs(operation));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected virtual void OnOperationExecuted(OperationExecutedEventArgs e)
        {
            if (OperationExecuted != null)
            {
                OperationExecuted(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnUpdated()
        {
            OnUpdated(new EventArgs());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnUpdated(EventArgs e)
        {
            if (_SuspendUpdatedEvent != false)
            {
                _UpdatedEventWasIgnored = true;
                return;
            }

            if (Updated != null)
            {
                Updated(this, e);
            }
        }

        /// ComponentListItemの追加によって Componentの追加が必要無いのならば
        /// AddComponent、RemoveComponentは必要無い。
        /// この場合、ListCtrlは Componentの完全な Viewとして動作する
        ///
        /// 駄目でした、AddComponent、RemoveComponentをしないとアイテムの移動ができません

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void AddComponent(ComponentListItem item)
        {
            if (InterlockComponent == false)
            {
                return;
            }

            //Operation: ここで Operationにする
            ComponentListItem nextSiblingItem = null;
            Component parent = null;
            Component nextSibling = null;

            nextSiblingItem = GetNextSibling(item);
            parent = _ComponentsOwnerList.Owner;

            nextSibling = nextSiblingItem != null ? nextSiblingItem.Target : null;

            InsertComponentOperation operation = null;
            operation = new InsertComponentOperation(parent, nextSibling,
                                                        new[] { item.Target });
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RemoveComponent(ComponentListItem item)
        {
            if (InterlockComponent == false)
            {
                return;
            }

            //Operation: ここで Operationにする
            RemoveComponentOperation operation = null;
            operation = new RemoveComponentOperation(new[] { item.Target });
            operation.Execute();
            OnOperationExecuted(operation);
        }

        /// <summary>
        /// Componentを移動します。
        /// </summary>
        private void MoveComponent(ComponentListItem item, int index)
        {
            if (InterlockComponent == false)
            {
                return;
            }

            MoveComponentOperation operation = new MoveComponentOperation(item.Target, index);
            operation.Execute();
            OnOperationExecuted(operation);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            _SuspendComponentCollectionChanged = true;

            try
            {
                switch (e.Action)
                {
                    case NotifyCollectionChangedAction.Add:
                        foreach (ComponentListItem newItem in e.NewItems)
                        {
                            newItem.Adapter = this;

                            if (_SuspendCollectionChanged == false)
                            {
                                AddComponent(newItem);
                            }
                        }
                        break;

                    case NotifyCollectionChangedAction.Remove:
                        foreach (ComponentListItem oldItem in e.OldItems)
                        {
                            if (_SuspendCollectionChanged == false)
                            {
                                RemoveComponent(oldItem);
                            }

                            oldItem.Adapter = null;
                        }
                        break;

                    case NotifyCollectionChangedAction.Move:
                        ComponentListItemCollection collection = sender as ComponentListItemCollection;
                        foreach (ComponentListItem newItem in e.NewItems)
                        {
                            if (_SuspendCollectionChanged == false)
                            {
                                int index = collection.IndexOf(newItem);
                                MoveComponent(newItem, index);
                            }
                        }

                        break;
                }
            }
            finally
            {
                _SuspendComponentCollectionChanged = false;
            }

            //
            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentListItem[] FindComponentListItems(Component component)
        {
            return _Items.Cast<ComponentListItem>()
                .Where(i => i.Target == component).ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void AddListItem(int index, Component component)
        {
            if (InterlockComponent == false)
            {
                return;
            }

            if (index < _ComponentsOwnerList.Count)
            {
                if (--index < 0)
                {
                    _Items.Insert(0, CreateListItem(component));
                }
                else
                {
                    // _Items はソートなどにより _ComponentOwnerList と並び順が違いますが
                    // ユーザーに見た目の違和感を与えないようにするため
                    // 直前のアイテムが同じになるようにします。
                    var prevComponent = _ComponentsOwnerList[index];
                    var items = FindComponentListItems(prevComponent);
                    Debug.Assert(items.Length == 1, "There are plural target component");
                    index = _Items.IndexOf(items[0]) + 1;
                    _Items.Insert(index, CreateListItem(component));
                }
            }
            else
            {
                _Items.Add(CreateListItem(component));
            }

        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RemoveListItem(Component component)
        {
            if (InterlockComponent == false)
            {
                return;
            }

            ComponentListItem[] items = FindComponentListItems(component);
            foreach (ComponentListItem item in items)
            {
                _Items.Remove(item);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void MoveListItem(Component component)
        {
            if (InterlockComponent == false)
            {
                return;
            }

            RemoveListItem(component);

            var index = this._ComponentsOwnerList.IndexOf(component);
            AddListItem(index, component);
        }

        /// <summary>
        ///
        /// </summary>
        private void AddComponentCollectionChangedEvent(ComponentList list)
        {
            if (list != null)
            {
                list.CollectionChanged += OnComponentCollectionChanged;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void RemoveComponentCollectionChangedEvent(ComponentList list)
        {
            if (list != null)
            {
                list.CollectionChanged -= OnComponentCollectionChanged;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnComponentCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (_SuspendComponentCollectionChanged != false)
            {
                return;
            }

            _SuspendCollectionChanged = true;
            _Items.BeginCompressEvent();

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    var index = e.NewStartingIndex;
                    foreach (Component newItem in e.NewItems)
                    {
                        AddListItem(index, newItem);
                        index++;
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (Component oldItem in e.OldItems)
                    {
                        RemoveListItem(oldItem);
                    }
                    break;

                case NotifyCollectionChangedAction.Move:
                    foreach (Component item in e.NewItems)
                    {
                        MoveListItem(item);
                    }
                    break;
            }

            _Items.EndCompressEvent();
            _SuspendCollectionChanged = false;

            //
            OnUpdated();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentListItem GetItemByComponent(
                                                     ComponentListItem[] items,
                                                     Component component
                                                     )
        {
            foreach (ComponentListItem item in items)
            {
                if (item.Target == component)
                {
                    return item;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentListItem[] SortByOriginalOrder(ComponentListItem[] items)
        {
            List<ComponentListItem> list = new List<ComponentListItem>();
            ComponentListItem item = null;

            if (_ComponentsOwnerList != null)
            {
                foreach (Component component in _ComponentsOwnerList)
                {
                    item = GetItemByComponent(items, component);
                    list.Add(item);
                }
            }
            else if (this.originalOrderItems != null)
            {
                return this.originalOrderItems;
            }

            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private ComponentListItem[] Sort(ComponentListItem[] items, IComparer<IListItem> comparer)
        {
            if (items.Length <= 0)
            {
                return new ComponentListItem[0];
            }

            List<ComponentListItem> list = new List<ComponentListItem>();

            list.AddRange(items);
            Sort(comparer, list);
            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void Sort(IComparer<IListItem> comparer, List<ComponentListItem> list)
        {
            List<ComponentListItem> listL = null;
            List<ComponentListItem> listR = null;
            ComponentListItem item = null;
            int center = 0;
            int index = 0;
            int indexL = 0;
            int indexR = 0;

            switch (list.Count)
            {
                case 0:
                    Debug.Assert(false, "list count is 0");
                    break;

                case 1:
                    break;

                case 2:
                    if (comparer.Compare(list[0], list[1]) > 0)
                    {
                        item = list[0];
                        list[0] = list[1];
                        list[1] = item;
                    }
                    break;

                default:
                    listL = new List<ComponentListItem>();
                    listR = new List<ComponentListItem>();

                    //
                    center = list.Count / 2;
                    while (index < center)
                    {
                        listL.Add(list[index]);
                        index++;
                    }

                    while (index < list.Count)
                    {
                        listR.Add(list[index]);
                        index++;
                    }

                    //
                    Sort(comparer, listL);
                    Sort(comparer, listR);

                    //
                    list.Clear();
                    while (indexL < listL.Count && indexR < listR.Count)
                    {
                        if (comparer.Compare(listL[indexL], listR[indexR]) <= 0)
                        {
                            list.Add(listL[indexL]);
                            indexL++;
                        }
                        else
                        {

                            list.Add(listR[indexR]);
                            indexR++;
                        }
                    }

                    while (indexL < listL.Count)
                    {
                        list.Add(listL[indexL]);
                        indexL++;
                    }

                    while (indexR < listR.Count)
                    {
                        list.Add(listR[indexR]);
                        indexR++;
                    }
                    break;
            }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class OperationExecutedEventArgs : EventArgs
    {

        private Operation _Operation = null;

        ///
        public OperationExecutedEventArgs(Operation operation)
        {
            _Operation = operation;
        }

        ///
        public Operation Operation
        {
            get { return _Operation; }
        }
    }

    ///
    public delegate void OperationExecutedEventHandler(
                                                       Object sender,
                                                       OperationExecutedEventArgs e
                                                       );

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ListComparer : IComparer<IListItem>
    {
        private string _Name = String.Empty;
        private Type _Type = null;
        private bool _Ascending = true;

        private Dictionary<Type, CompareHandler> _Dictionary = null;

        protected delegate int CompareHandler(string a, string b);

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ListComparer(string name, Type type, bool ascending)
        {
            _Name = name;
            _Type = type;
            _Ascending = ascending;

            _Dictionary = new Dictionary<Type, CompareHandler>();
            _Dictionary.Add(typeof(string), CompareByString);
            _Dictionary.Add(typeof(int), CompareByInt);
            _Dictionary.Add(typeof(ulong), CompareByULong);
            _Dictionary.Add(typeof(float), CompareBySingle);
            _Dictionary.Add(typeof(double), CompareByDouble);
            _Dictionary.Add(typeof(bool), CompareByBoolean);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Dictionary<Type, CompareHandler> CompareHandlers
        {
            get { return _Dictionary; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Compare(IListItem at, IListItem bt)
        {
            CompareHandler handler = null;
            IConstParameterValue valueA = null;
            IConstParameterValue valueB = null;
            string textA = null;
            string textB = null;
            int result = 0;

            valueA = at.GetConstValue(_Name);
            valueB = bt.GetConstValue(_Name);

            if (valueA != null)
            {
                textA = valueA.ToString();
            }
            if (valueB != null)
            {
                textB = valueB.ToString();
            }

            if (textA == null) { textA = String.Empty; }
            if (textB == null) { textB = String.Empty; }

            if ((handler = CompareHandlers[_Type]) != null)
            {
                result = handler(textA, textB);
            }

            //
            if (_Ascending == false)
            {
                result = result * -1;
            }

            return result;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int CompareByString(string a, string b)
        {
            return a.CompareTo(b);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int CompareByInt(string a, string b)
        {
            int av = 0;
            int bv = 0;

            try
            {
                av = int.Parse(a);
                bv = int.Parse(b);
            }

            catch { return a.CompareTo(b); }
            return av - bv;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int CompareByULong(string a, string b)
        {
            ulong av = 0;
            ulong bv = 0;

            try
            {
                av = ulong.Parse(a);
                bv = ulong.Parse(b);
            }

            catch { return a.CompareTo(b); }

            if (av < bv) { return -1; }
            if (av > bv) { return 1; }
            return 0;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int CompareBySingle(string a, string b)
        {
            float av = 0.0F;
            float bv = 0.0F;

            try
            {
                av = float.Parse(a);
                bv = float.Parse(b);
            }

            catch { return a.CompareTo(b); }

            if (av < bv) { return -1; }
            if (av > bv) { return 1; }
            return 0;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int CompareByDouble(string a, string b)
        {
            double av = 0.0;
            double bv = 0.0;

            try
            {
                av = double.Parse(a);
                bv = double.Parse(b);
            }

            catch { return a.CompareTo(b); }

            if (av < bv) { return -1; }
            if (av > bv) { return 1; }
            return 0;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private int CompareByBoolean(string a, string b)
        {
            bool av = false;
            bool bv = false;

            try
            {
                av = bool.Parse(a);
                bv = bool.Parse(b);
            }

            catch { return a.CompareTo(b); }

            if (av == true && bv == false) { return -1; }
            if (av == false && bv == true) { return 1; }
            return 0;
        }
    }
}
