﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TrackBarCtrl : OperatableController, ISupportInitialize
    {
        /// <summary>
        /// RangeMarkerの種類
        /// </summary>
        public enum RangeMarkerType
        {
            Unknown = 0,	// 不明
            Begin,			// 領域の開始位置
            End				// 領域の終了位置
        }

        private TrackBarInfo _TrackBarInfo = null;

        protected ReadOnlyMouseOperator ReadOnlyOperator = null;
        protected ExchangeMouseOperator ExchangeOperator = null;
        protected MoveKeyMouseOperator MovePositionOperator = null;
        protected MoveRangeBeginMarkerOperator MoveRangeBeginMarkerMover = null;
        protected MoveRangeEndMarkerOperator MoveRangeEndMarkerMover = null;

        private TrackBarDrawer _Drawer = new TrackBarDrawer();

        private int _Position = 0;
        private int _Minimum = 0;
        private int _Maximum = 100;
        private int _SmallChange = 1;
        private int _LargeChange = 5;

        private int _ThumbSize = 32;
        private int _ThumbHeight = 0;

        private TickStyle _TickStyle = TickStyle.BottomRight;
        private int _TickFrequency = 10;
        private int _TickHeight = 3;

        private bool _EnableDragMarker = false;
        private int _DragMarkerPosition = -1;

        private bool _EnableDefaultValue = false;
        private int _DefaultValue = 0;

        private bool _EnableDirectPointing = false;

        private bool _EnableMarker = false;
        private ITrackBarMarker[] _Markers = null;
        private int _MarkerWidth = 8;
        private int _MarkerHeight = 8;

        private ToolTip _ToolTip = new ToolTip();

        private bool _EnableRangeMarker = false;
        private bool _RangeMarkerVisible = true;
        private int _RangeBeginMarker = 10;
        private int _RangeEndMarker = 20;
        private int _RangeMarkerWidth = 8;
        private int _RangeMarkerHeight = 8;
        private int _RangeMarkerBarHeight = 8;

        private bool _ReadOnly = false;

        //
        public delegate void ValueChangingEventHandler(Object sender, TrackBarEventArgs e);
        public delegate void ValueChangedEventHandler(Object sender, TrackBarEventArgs e);
        public delegate void MarkerEventHandler(Object sender, TrackBarMarkerEventArgs e);

        public event ValueChangingEventHandler ValueChanging;
        public event ValueChangedEventHandler ValueChanged;
        public event MarkerEventHandler MarkerClicked;
        public event EventHandler RangeValueChanging;
        public event EventHandler RangeValueChanged;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TrackBarCtrl()
        {
            InitializeComponent();

            BackColor = SystemColors.Control;

            ExchangeOperator = new ExchangeMouseOperator(this);
            MovePositionOperator = new MoveKeyMouseOperator(this);
            MoveRangeBeginMarkerMover = new MoveRangeBeginMarkerOperator(this);
            MoveRangeEndMarkerMover = new MoveRangeEndMarkerOperator(this);
            ReadOnlyOperator = new ReadOnlyMouseOperator(this);
            CurrentOperator = ExchangeOperator;

            AllowDrop = true;

            HorizontalScrollBar.Visible = false;
            VerticalScrollBar.Visible = false;

            AddDrawer(DefaultDrawer);

            _ToolTip.Active = true;
            //			_ToolTip.IsBalloon = true;

            UpdateHeight();
            RefreshInnerData();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool ReadOnly
        {
            get
            {
                return _ReadOnly;
            }
            set
            {
                if (_ReadOnly == true && value == false)
                {
                    CurrentOperator = ExchangeOperator;
                }
                _ReadOnly = value;
                if (_ReadOnly == true)
                {
                    CurrentOperator = ReadOnlyOperator;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void BeginInit()
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void EndInit()
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateHeight()
        {
            int count = 3;
            int height = 0;

            if (_TickStyle != TickStyle.None) { count++; }
            if (_EnableMarker != false) { count++; }
            if (_EnableRangeMarker != false) { count += 3; }

            height = (ClientRectangleWithoutScrollBar.Height - 2) / count;

            _ThumbHeight = height * 3;
            _TickHeight = height;
            _MarkerHeight = height;
            _RangeMarkerBarHeight = height * 3;
            _RangeMarkerHeight = height;
            _RangeMarkerWidth = height;
        }

        ///--------------------------------
        /// <summary>
        /// デフォルトドロワーの取得
        /// </summary>
        public virtual OperatableControllerDrawer DefaultDrawer
        {
            get { return _Drawer; }
        }

        ///--------------------------------
        /// <summary>
        /// ツールチップの表示
        /// </summary>
        protected void ShowToolTip(string text)
        {
            //IsBalloonを trueにした時には、以下の処理を行わないとマウスが停止して
            //いるにも関わらず OnMoveイベントが発生してしまう(原因不明)
            if (_ToolTip.GetToolTip(this) != text)
            {
                _ToolTip.SetToolTip(this, text);
            }
        }

        ///--------------------------------
        /// <summary>
        /// ツールチップの非表示
        /// </summary>
        protected void HideToolTip()
        {
            _ToolTip.SetToolTip(this, null);
        }

        private void FocusMarker(ITrackBarMarker target)
        {
            _TrackBarInfo.FocusedMarker = target;

            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void RefreshInnerData()
        {
            TrackBarInfo info = null;
            Rectangle clientRect = ClientRectangleWithoutScrollBar;
            int bottom = clientRect.Height - 1;
            int trackbarTop = 1;
            int graduationTop = -1;
            int markerTop = -1;
            int loopMarkerTop = 1;

            //
            if (_EnableMarker != false)
            {

                if (_EnableRangeMarker != false)
                {

                    trackbarTop = loopMarkerTop + _RangeMarkerBarHeight;
                    markerTop = bottom - _MarkerHeight;
                    graduationTop = markerTop - _TickHeight;

                }
                else
                {

                    markerTop = bottom - _MarkerHeight;
                    graduationTop = markerTop - _TickHeight;
                }

            }
            else
            {

                if (_EnableRangeMarker != false)
                {

                    trackbarTop = loopMarkerTop + _RangeMarkerBarHeight;
                    graduationTop = bottom - _TickHeight;

                }
                else
                {

                    graduationTop = bottom - _TickHeight;
                }
            }

            //
            info = new TrackBarInfo();
            info.Position = _Position;
            info.EnableDragMarker = _EnableDragMarker;
            info.DragMarkerPosition = _DragMarkerPosition;

            //
            info.TrackBarY = trackbarTop;
            info.ThumbY = 0;
            info.ThumbHeight = _ThumbHeight;

            //目盛り情報の作成
            if (_TickStyle != TickStyle.None)
            {
                GraduationInfo[] graduations = null;
                int count = 0;

                count = (Maximum - Minimum) / _TickFrequency + 1;
                graduations = new GraduationInfo[count];

                for (int index = 0; index < count; index++)
                {
                    graduations[index] = new GraduationInfo();
                    graduations[index].Position = LogicalToClientPosition
                        (Minimum + (index * _TickFrequency));
                }

                info.Graduations = graduations;
                info.GraduationY = graduationTop;
                info.GraduationHeight = _TickHeight;
            }

            //マーカー情報の作成
            if (_EnableMarker != false && _Markers != null)
            {
                TrackBarMarker[] markers = null;
                int count = 0;

                if ((count = _Markers.Length) > 0)
                {
                    markers = new TrackBarMarker[count];

                    for (int index = 0; index < count; index++)
                    {
                        markers[index] = new TrackBarMarker();
                        markers[index].Marker = _Markers[index];
                        markers[index].Rectangle = GetMarkerRect(_Markers[index]);
                    }

                    info.Markers = markers;
                    info.MarkerY = markerTop;
                    info.MarkerHeight = _MarkerHeight;
                }
            }

            //ループマーカー情報の作成
            if (_EnableRangeMarker != false)
            {
                TrackBarRangeMarker loopMarker = null;

                loopMarker = new TrackBarRangeMarker();
                loopMarker.BeginRectangle = GetRangeMarkerRect(_RangeBeginMarker, RangeMarkerType.Begin);
                loopMarker.EndRectangle = GetRangeMarkerRect(_RangeEndMarker, RangeMarkerType.End);

                info.RangeMarker = loopMarker;
            }

            //
            _TrackBarInfo = info;
            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        /// 表示情報の取得
        /// </summary>
        protected override Object GetDrawData()
        {
            return _TrackBarInfo;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TickStyle TickStyle
        {
            get { return _TickStyle; }
            set { _TickStyle = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int TickFrequency
        {
            get { return _TickFrequency; }
            set
            {
                _TickFrequency = value;
                RefreshInnerData();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Minimum
        {
            get { return _Minimum; }
            set
            {
                _Minimum = value;
                RefreshInnerData();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int Maximum
        {
            get { return _Maximum; }
            set
            {
                _Maximum = value;
                RefreshInnerData();
            }
        }

        ///--------------------------------
        /// <summary>
        /// 値の設定
        /// 外側から値を設定する場合にはイベントは発生していけない
        /// </summary>
        public int Value
        {
            get { return _Position; }
            set
            {
                SetValue(value);
                RefreshInnerData();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetValue(int value)
        {
            if (value < Minimum) { value = Minimum; }
            if (value > Maximum) { value = Maximum; }

            _Position = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int ThumbSize
        {
            get { return _ThumbSize; }
            set
            {
                if (_ThumbSize == value) return;

                _ThumbSize = value;
                RefreshInnerData();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int SmallChange
        {
            get { return _SmallChange; }
            set { _SmallChange = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int LargeChange
        {
            get { return _LargeChange; }
            set { _LargeChange = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool EnableDefaultValue
        {
            get { return _EnableDefaultValue; }
            set { _EnableDefaultValue = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int DefaultValue
        {
            get { return _DefaultValue; }
            set { _DefaultValue = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool EnableDirectPointing
        {
            get { return _EnableDirectPointing; }
            set
            {
                if (_EnableDirectPointing == value)
                    return;

                _EnableDirectPointing = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool EnableMarker
        {
            get { return _EnableMarker; }
            set
            {
                if (_EnableMarker == value) return;

                _EnableMarker = value;
                UpdateHeight();
                RefreshInnerData();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public ITrackBarMarker[] Markers
        {
            get { return _Markers; }
            set { _Markers = value; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool EnableRangeMarker
        {
            get { return _EnableRangeMarker; }
            set
            {
                if (_EnableRangeMarker == value) return;

                _EnableRangeMarker = value;
                UpdateHeight();
                RefreshInnerData();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool RangeMarkerVisible
        {
            get { return _RangeMarkerVisible; }
            set
            {
                if (_RangeMarkerVisible == value) return;

                _RangeMarkerVisible = value;
                Invalidate();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int RangeBeginMarkerValue
        {
            get { return _RangeBeginMarker; }
            set
            {
                _RangeBeginMarker = value;
                RefreshInnerData();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int RangeEndMarkerValue
        {
            get { return _RangeEndMarker; }
            set
            {
                RefreshInnerData();
                _RangeEndMarker = value;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetRangeBeginMarkerValue(int value)
        {
            if (value >= _RangeEndMarker)
            {
                value = _RangeEndMarker - 1;
            }

            _RangeBeginMarker = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SetRangeEndMarkerValue(int value)
        {
            if (value <= _RangeBeginMarker)
            {
                value = _RangeBeginMarker + 1;
            }

            _RangeEndMarker = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle GetTrackBarRect()
        {
            return GetTrackBarRect(ClientRectangle);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle GetTrackBarRect(Rectangle rect)
        {
            int height = rect.Height - 2;

            switch (_TickStyle)
            {
                case TickStyle.None:
                    break;

                case TickStyle.BottomRight:
                    // Tick と Bar の間に 1px スペースを空ける
                    height -= _TickHeight + 1;
                    break;
            }

            if (_EnableMarker != false)
            {
                height -= _MarkerHeight;
            }

            if (_EnableRangeMarker != false)
            {
                height -= _RangeMarkerBarHeight;
            }

            int trackbarY = (_TrackBarInfo != null) ? _TrackBarInfo.TrackBarY : 0;

            if (height < 0)
            {
                height = 0;
            }

            return new Rectangle(ClientRectangle.X,
                                 ClientRectangle.Y + trackbarY,
                                 ClientRectangle.Width,
                                 height);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Rectangle ValidRect
        {
            get
            {
                Rectangle rect = GetTrackBarRect();
                int width = ThumbSize;

                return new Rectangle(rect.X + width / 2,
                                      rect.Y + 1,
                                      rect.Width - width,
                                      rect.Height - 2);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Rectangle GetPositionRect(int position)
        {
            return GetPositionRect(ClientRectangle, position);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Rectangle GetPositionRect(Rectangle rect, int position)
        {
            int x = LogicalToClientPosition(position);
            int width = ThumbSize;

            rect = GetTrackBarRect(rect);

            return new Rectangle(x - width / 2,
                                  rect.Y + 1,
                                  width,
                                  rect.Height - 1);
        }

        ///--------------------------------
        /// <summary>
        /// 論理座標からクライアント座標に変換する
        /// </summary>
        public int LogicalToClientPosition(int position)
        {
            Rectangle rect = ValidRect;

            return (int)(((double)rect.Width / (double)(Maximum - Minimum)) *
                         (double)(position - Minimum)) + rect.X;
        }

        ///--------------------------------
        /// <summary>
        /// クライアント座標から論理座標に変換する
        /// </summary>
        public int ClientToLogicalPosition(int x)
        {
            Rectangle rect = ValidRect;
            int position = -1;

            double range = 0.0;

            range = (double)rect.Width / (double)(Maximum - Minimum);
            position = (int)((double)((x - rect.X) + (range / 2.0)) / range);

            position += Minimum;

            if (position < Minimum) { position = Minimum; }
            if (position > Maximum) { position = Maximum; }

            return position;
        }

        ///--------------------------------
        /// <summary>
        /// 待機状態のオペレータを設定
        /// </summary>
        protected internal void ChangeOperatorToExchange(Point beginPoint)
        {
            CurrentOperator = ExchangeOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// 移動状態のオペレータを設定
        /// </summary>
        protected internal void ChangeOperatorToMovePosition(Point beginPoint)
        {
            CurrentOperator = MovePositionOperator;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// ループマーカー開始点の移動状態のオペレータを設定
        /// </summary>
        protected internal void ChangeOperatorToMoveRangeBeginMarker(Point beginPoint)
        {
            CurrentOperator = MoveRangeBeginMarkerMover;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// ループマーカー終了点の移動状態のオペレータを設定
        /// </summary>
        protected internal void ChangeOperatorToMoveRangeEndMarker(Point beginPoint)
        {
            CurrentOperator = MoveRangeEndMarkerMover;
            CurrentOperator.Initialize(beginPoint);
        }

        ///--------------------------------
        /// <summary>
        /// 衝突判定
        /// 座標はクライアント座標を指定する
        /// </summary>
        protected virtual CollisionResult Collision(int mx, int my)
        {
            CollisionResult result = new CollisionResult();
            Rectangle rect = ClientRectangle;
            int width = ValidRect.Width;
            int height = ValidRect.Height;
            int position = -1;
            Rectangle positionRect;

            //ここでキーと当り判定を行う
            position = _TrackBarInfo.Position;
            positionRect = GetPositionRect(position);

            //
            if (positionRect.Contains(mx, my) != false)
            {
                result.HitToThumb(position);
                return result;
            }

            //
            if (mx < positionRect.Left &&
                my >= positionRect.Top && my < positionRect.Bottom)
            {
                result.HitToNearSpace(position);
                return result;
            }

            //
            if (mx > positionRect.Right &&
                my >= positionRect.Top && my < positionRect.Bottom)
            {
                result.HitToFarSpace(position);
                return result;
            }

            //マーカーとの衝突判定
            if (_EnableMarker != false && _Markers != null)
            {
                if (CollisionToMarker(result, mx, my) != false)
                {
                    return result;
                }
            }

            //ループマーカーとの衝突判定
            if (_EnableRangeMarker != false)
            {
                if (CollisionToRangeMarker(result, mx, my) != false)
                {
                    return result;
                }
            }

            //
            return result;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle GetMarkerRect(ITrackBarMarker marker)
        {
            int bottom = ClientRectangleWithoutScrollBar.Height - 2;
            int cx = LogicalToClientPosition(marker.Value);

            return new Rectangle(cx - _MarkerWidth / 2, bottom - _MarkerHeight - 1,
                                  _MarkerWidth, _MarkerHeight);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle GetRangeMarkerRect(int position)
        {
            return GetRangeMarkerRect(position, RangeMarkerType.Unknown);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Rectangle GetRangeMarkerRect(int position, RangeMarkerType type)
        {
            Rectangle clientRect = ClientRectangleWithoutScrollBar;
            int bottom = clientRect.Height;
            int cx = LogicalToClientPosition(position);

            switch (type)
            {
                case RangeMarkerType.Begin:
                    return new Rectangle(cx, 0, _RangeMarkerWidth, _RangeMarkerHeight);

                case RangeMarkerType.End:
                    return new Rectangle(cx - _RangeMarkerWidth, 0, _RangeMarkerWidth, _RangeMarkerHeight);
            }

            return new Rectangle(cx - _RangeMarkerWidth / 2, 0, _RangeMarkerWidth, _RangeMarkerHeight);
        }

        ///--------------------------------
        /// <summary>
        /// マーカーとの衝突判定
        /// </summary>
        protected bool CollisionToMarker(CollisionResult result, int mx, int my)
        {
            Rectangle rect;

            for (int index = _Markers.Length - 1; index >= 0; index--)
            {

                ITrackBarMarker marker = _Markers[index];

                rect = GetMarkerRect(marker);
                if (rect.Contains(mx, my) != false)
                {
                    result.HitToMarker(marker);
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        /// ループマーカーとの衝突判定
        /// </summary>
        protected bool CollisionToRangeMarker(CollisionResult result, int mx, int my)
        {
            if (!EnableRangeMarker || !RangeMarkerVisible) { return false; }

            Rectangle rect;

            rect = GetRangeMarkerRect(_RangeBeginMarker, RangeMarkerType.Begin);

            if (rect.Contains(mx, my) != false)
            {
                result.HitToRangeBeginMarker();
                return true;
            }

            rect = GetRangeMarkerRect(_RangeEndMarker, RangeMarkerType.End);
            if (rect.Contains(mx, my) != false)
            {
                result.HitToRangeEndMarker();
                return true;
            }

            return false;
        }

        ///--------------------------------
        /// <summary>
        /// ドラックマーカーを非表示にする
        /// </summary>
        protected void DragMarkerNone()
        {
            _EnableDragMarker = false;
        }

        ///--------------------------------
        /// <summary>
        /// キーのドラックマーカーを表示する
        /// </summary>
        protected void DragMarker(int position)
        {
            _EnableDragMarker = true;
            _DragMarkerPosition = position;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventValueChanging()
        {
            if (ValueChanging != null)
            {
                ValueChanging(this, new TrackBarEventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventValueChanging(TrackBarEventArgs e)
        {
            if (ValueChanging != null)
            {
                ValueChanging(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventValueChanged()
        {
            if (ValueChanged != null)
            {
                ValueChanged(this, new TrackBarEventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventValueChanged(TrackBarEventArgs e)
        {
            if (ValueChanged != null)
            {
                ValueChanged(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventMarkerClicked(TrackBarMarkerEventArgs e)
        {
            if (MarkerClicked != null)
            {
                MarkerClicked(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventRangeMarkerValueChanging()
        {
            if (RangeValueChanging != null)
            {
                RangeValueChanging(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void ExecuteEventRangeMarkerValueChanged()
        {
            if (RangeValueChanged != null)
            {
                RangeValueChanged(this, new EventArgs());
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool SmallChangePosition(int direction)
        {
            SetValue(Value + direction * SmallChange);
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool LargeChangePosition(int direction)
        {
            SetValue(Value + direction * LargeChange);
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void MinimumPosition()
        {
            SetValue(Minimum);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void MaximumPosition()
        {
            SetValue(Maximum);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnLeave(EventArgs e)
        {
            base.OnLeave(e);

            Invalidate();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            UpdateHeight();
            RefreshInnerData();
        }

        ///--------------------------------
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        ///--------------------------------
        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                    components.Dispose();
            }
            base.Dispose(disposing);
        }

        ///--------------------------------
        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード］エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // HorizontalScrollBar
            //
            this.HorizontalScrollBar.Location = new System.Drawing.Point(0, 77);
            this.HorizontalScrollBar.Name = "HorizontalScrollBar";
            this.HorizontalScrollBar.Size = new System.Drawing.Size(134, 16);
            //
            // vScrollBar
            //
            this.VerticalScrollBar.Location = new System.Drawing.Point(134, 0);
            this.VerticalScrollBar.Name = "vScrollBar";
            this.VerticalScrollBar.Size = new System.Drawing.Size(16, 77);
            //
            // TrackBarCtrl
            //
            this.Name = "TrackBarCtrl";
            this.Size = new System.Drawing.Size(150, 93);
            this.ResumeLayout(false);

        }
        #endregion


        ///------------------------------------------------------------------------
        /// <summary>
        /// オペレータのベースクラス
        /// </summary>
        public class MouseOperator : OperatableControllerOperator
        {
            new protected TrackBarCtrl OwnerController
            {
                get { return (TrackBarCtrl)OperatableController; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MouseOperator(OperatableController controller) : base(controller)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedControl
            {
                get { return (Control.ModifierKeys & Keys.Control) != 0; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedShift
            {
                get { return (Control.ModifierKeys & Keys.Shift) != 0; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool PressedAlt
            {
                get { return (Control.ModifierKeys & Keys.Alt) != 0; }
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// 待機状態のオペレータ
        /// </summary>
        public class ReadOnlyMouseOperator : MouseOperator
        {
            public ReadOnlyMouseOperator(OperatableController controller) : base(controller)
            {
            }
        }

        ///------------------------------------------------------------------------
        /// <summary>
        /// 待機状態のオペレータ
        /// </summary>
        public class ExchangeMouseOperator : MouseOperator
        {
            private Point _MouseFirstDownPoint = new Point(-1, -1);
            private bool _InterimDragPositionMover = false;
            private bool _InterimDragRangeBeginMarkerMover = false;
            private bool _InterimDragRangeEndMarkerMover = false;
            private bool _InterimKey = false;

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public ExchangeMouseOperator(OperatableController controller) :
                base(controller)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                _InterimDragPositionMover = false;
                _InterimDragRangeBeginMarkerMover = false;
                _InterimDragRangeEndMarkerMover = false;
                _InterimKey = false;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void SetValue(int position)
            {
                OwnerController.SetValue(position);
                OwnerController.ExecuteEventValueChanged();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void SetValue(int position, bool pressedControl, bool pressedShift, bool pressedAlt)
            {
                TrackBarEventArgs args = null;

                args = new TrackBarEventArgs(pressedControl,
                                              pressedShift,
                                              pressedAlt);

                OwnerController.SetValue(position);
                OwnerController.ExecuteEventValueChanged(args);
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void SmallChangePosition(int direction)
            {
                OwnerController.SmallChangePosition(direction);
                OwnerController.ExecuteEventValueChanged();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void LargeChangePosition(int direction)
            {
                OwnerController.LargeChangePosition(direction);
                OwnerController.ExecuteEventValueChanged();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void MinimumPosition()
            {
                OwnerController.MinimumPosition();
                OwnerController.ExecuteEventValueChanged();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void MaximumPosition()
            {
                OwnerController.MaximumPosition();
                OwnerController.ExecuteEventValueChanged();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void HitToSpace(int x, int y, int direction)
            {
                int position = OwnerController.ClientToLogicalPosition(x);

                if (OwnerController.EnableDirectPointing)
                {
                    OwnerController.ChangeOperatorToMovePosition(new Point(x, y));
                }
                else
                {

                    if (PressedControl != false &&
                        OwnerController.EnableDefaultValue != false)
                    {
                        SetValue(OwnerController.DefaultValue,
                                  PressedControl, PressedShift, PressedAlt);
                        return;
                    }

                    if (PressedAlt != false)
                    {
                        SetValue(position, PressedControl, PressedShift, PressedAlt);
                        return;
                    }

                    LargeChangePosition(direction);

                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool MouseLeftButtonDownOperation(MouseEventArgs e)
            {
                CollisionResult result = OwnerController.Collision(e.X, e.Y);
                switch (result.Type)
                {

                    ///
                    case CollisionResult.HitType.None:
                        break;

                    ///
                    case CollisionResult.HitType.Thumb:
                        _InterimDragPositionMover = true;
                        _MouseFirstDownPoint.X = e.X;
                        _MouseFirstDownPoint.Y = e.Y;
                        break;

                    ///
                    case CollisionResult.HitType.RangeBeginMarker:
                        _InterimDragRangeBeginMarkerMover = true;
                        _MouseFirstDownPoint.X = e.X;
                        _MouseFirstDownPoint.Y = e.Y;
                        break;

                    ///
                    case CollisionResult.HitType.RangeEndMarker:
                        _InterimDragRangeEndMarkerMover = true;
                        _MouseFirstDownPoint.X = e.X;
                        _MouseFirstDownPoint.Y = e.Y;
                        break;

                    //
                    case CollisionResult.HitType.NearSpace:
                        HitToSpace(e.X, e.Y, -1);
                        break;

                    //
                    case CollisionResult.HitType.FarSpace:
                        HitToSpace(e.X, e.Y, 1);
                        break;

                    case CollisionResult.HitType.Marker:
                        OwnerController.ExecuteEventMarkerClicked(new TrackBarMarkerEventArgs(result.Marker));
                        break;
                }

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected bool MouseRightButtonDownOperation(MouseEventArgs e)
            {
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseDown(MouseEventArgs e)
            {
                switch (e.Button)
                {
                    case MouseButtons.Left:
                        return MouseLeftButtonDownOperation(e);

                    case MouseButtons.Right:
                        return MouseRightButtonDownOperation(e);

                    case MouseButtons.Middle:
                        break;
                }
                return true;
            }


            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                if (_InterimKey != false)
                {
                    int position = OwnerController.ClientToLogicalPosition(e.X);
                    OwnerController.SetValue(position);
                    OwnerController.ExecuteEventValueChanged();

                    //Cursor.Current = Cursors.Arrow;
                    OwnerController.DragMarkerNone();
                    OwnerController.Invalidate();

                    //OwnerController.RefreshInnerData();
                }

                _InterimDragPositionMover = false;
                _InterimDragRangeBeginMarkerMover = false;
                _InterimDragRangeEndMarkerMover = false;
                _InterimKey = false;
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private bool MouseMovedOverRange(int mx, int my, int range)
            {
                if (Math.Abs(mx - _MouseFirstDownPoint.X) > range ||
                    Math.Abs(my - _MouseFirstDownPoint.Y) > range)
                {
                    return true;
                }
                return false;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                Size dragSize = SystemInformation.DragSize;

                //
                if (_InterimDragPositionMover != false)
                {
                    if (MouseMovedOverRange(e.X, e.Y, 1) != false)
                    {
                        OwnerController.ChangeOperatorToMovePosition
                            (_MouseFirstDownPoint);
                        return true;
                    }
                }

                //
                if (_InterimDragRangeBeginMarkerMover != false)
                {
                    if (MouseMovedOverRange(e.X, e.Y, 1) != false)
                    {
                        OwnerController.ChangeOperatorToMoveRangeBeginMarker
                            (_MouseFirstDownPoint);
                        return true;
                    }
                }

                //
                if (_InterimDragRangeEndMarkerMover != false)
                {
                    if (MouseMovedOverRange(e.X, e.Y, 1) != false)
                    {
                        OwnerController.ChangeOperatorToMoveRangeEndMarker
                            (_MouseFirstDownPoint);
                        return true;
                    }
                }

                //
                CollisionResult result = OwnerController.Collision(e.X, e.Y);

                if (result.Type == CollisionResult.HitType.Marker)
                {

                    OwnerController.FocusMarker(result.Marker);
                    OwnerController.ShowToolTip(result.Marker.Text);

                }
                else
                {

                    if (result.Type == CollisionResult.HitType.RangeBeginMarker ||
                        result.Type == CollisionResult.HitType.RangeEndMarker)
                    {
                        OwnerController.Cursor = Cursors.SizeWE;
                    }
                    else
                    {
                        OwnerController.Cursor = Cursors.Arrow;
                    }

                    OwnerController.FocusMarker(null);
                    OwnerController.HideToolTip();

                }

                return true;
            }

            protected internal override bool OnMouseWheel(MouseEventArgs e)
            {
                int delta = (e.Delta < 0) ? 1 : -1;

                SetValue(OwnerController.Value + delta);

                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyDown(KeyEventArgs e)
            {
                switch (e.KeyCode)
                {

                    //
                    case Keys.Tab:
                        OwnerController.Parent.SelectNextControl
                            (OwnerController,
                              e.Shift == false ? true : false, true, true, true);
                        break;

                    //
                    case Keys.Up:
                    case Keys.Left:
                        SmallChangePosition(-1);
                        break;

                    //
                    case Keys.Right:
                    case Keys.Down:
                        SmallChangePosition(1);
                        break;

                    //
                    case Keys.PageUp:
                        LargeChangePosition(-1);
                        break;

                    //
                    case Keys.PageDown:
                        LargeChangePosition(1);
                        break;

                    //
                    case Keys.Home:
                        MinimumPosition();
                        break;

                    //
                    case Keys.End:
                        MaximumPosition();
                        break;
                }
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// 移動処理状態のオペレータ
        /// </summary>
        public class MoveKeyMouseOperator : MouseOperator
        {

            private int _Offset = -1;

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MoveKeyMouseOperator(OperatableController controller) : base(controller)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void ChangingValueWithOffset(int x)
            {
                int position = OwnerController.ClientToLogicalPosition(x + _Offset);

                OwnerController.SetValue(position);
                OwnerController.ExecuteEventValueChanging();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void ChangedValueWithOffset(int x)
            {
                int position = OwnerController.ClientToLogicalPosition(x + _Offset);

                OwnerController.SetValue(position);
                OwnerController.ExecuteEventValueChanged();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                CollisionResult result = OwnerController.Collision(beginPoint.X, beginPoint.Y);

                if (result.Type == CollisionResult.HitType.Thumb)
                {

                    int x = -1;

                    x = OwnerController.LogicalToClientPosition(OwnerController.Value);
                    _Offset = x - beginPoint.X;

                }

                ChangingValueWithOffset(beginPoint.X);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyDown(KeyEventArgs e)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                ChangedValueWithOffset(e.X);
                OwnerController.ChangeOperatorToExchange(new Point(e.X, e.Y));
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                ChangingValueWithOffset(e.X);
                return true;
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// ループマーカー移動処理状態のオペレータ
        /// </summary>
        public class MoveRangeMarkerOperator : MouseOperator
        {

            private int _Offset = -1;

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MoveRangeMarkerOperator(OperatableController controller) : base(controller)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected virtual int Value
            {
                get { Debug.Assert(false); return 0; }
                set { Debug.Assert(false); }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void ChangingValueWithOffset(int x)
            {
                int position = OwnerController.ClientToLogicalPosition(x + _Offset);

                Value = position;
                OwnerController.ExecuteEventRangeMarkerValueChanging();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private void ChangedValueWithOffset(int x)
            {
                int position = OwnerController.ClientToLogicalPosition(x + _Offset);

                Value = position;
                OwnerController.ExecuteEventRangeMarkerValueChanged();
                OwnerController.RefreshInnerData();
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void Initialize(Point beginPoint)
            {
                int x = -1;

                x = OwnerController.LogicalToClientPosition(Value);
                _Offset = x - beginPoint.X;

                ChangingValueWithOffset(beginPoint.X);
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override void OnKeyDown(KeyEventArgs e)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseUp(MouseEventArgs e)
            {
                ChangedValueWithOffset(e.X);
                OwnerController.ChangeOperatorToExchange(new Point(e.X, e.Y));
                return true;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected internal override bool OnMouseMove(MouseEventArgs e)
            {
                ChangingValueWithOffset(e.X);
                return true;
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// ループマーカー開始点の移動処理状態のオペレータ
        /// </summary>
        public class MoveRangeBeginMarkerOperator : MoveRangeMarkerOperator
        {

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MoveRangeBeginMarkerOperator(OperatableController controller) : base(controller)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected override int Value
            {
                get { return OwnerController.RangeBeginMarkerValue; }
                set { OwnerController.SetRangeBeginMarkerValue(value); }
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// ループマーカー終了点の移動処理状態のオペレータ
        /// </summary>
        public class MoveRangeEndMarkerOperator : MoveRangeMarkerOperator
        {

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public MoveRangeEndMarkerOperator(OperatableController controller) : base(controller)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected override int Value
            {
                get { return OwnerController.RangeEndMarkerValue; }
                set { OwnerController.SetRangeEndMarkerValue(value); }
            }
        }


        ///------------------------------------------------------------------------
        /// <summary>
        /// 衝突判定の結果を保持するクラス
        /// </summary>
        public class CollisionResult
        {
            public enum HitType
            {
                None,
                NearSpace,
                FarSpace,
                Thumb,
                Marker,
                RangeBeginMarker,
                RangeEndMarker,
            }

            private HitType _Type = HitType.None;
            private int _Position = -1;
            private ITrackBarMarker _Marker = null;

            ///--------------------------------
            /// <summary>
            /// 衝突無し
            /// </summary>
            public void HitToNone(int position)
            {
                _Type = HitType.None;
                _Position = position;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public void HitToNearSpace(int position)
            {
                _Type = HitType.NearSpace;
                _Position = position;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public void HitToFarSpace(int position)
            {
                _Type = HitType.FarSpace;
                _Position = position;
            }

            ///--------------------------------
            /// <summary>
            /// サムと衝突
            /// </summary>
            public void HitToThumb(int position)
            {
                _Type = HitType.Thumb;
                _Position = position;
            }

            ///--------------------------------
            /// <summary>
            /// マーカーと衝突
            /// </summary>
            public void HitToMarker(ITrackBarMarker marker)
            {
                _Type = HitType.Marker;
                _Marker = marker;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public void HitToRangeBeginMarker()
            {
                _Type = HitType.RangeBeginMarker;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public void HitToRangeEndMarker()
            {
                _Type = HitType.RangeEndMarker;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public HitType Type
            {
                get { return _Type; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public int Position
            {
                get { return _Position; }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public ITrackBarMarker Marker
            {
                get { return _Marker; }
            }
        }
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TrackBarDrawer : OperatableControllerDrawer
    {

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        new protected TrackBarCtrl OwnerController
        {
            get
            {
                return (TrackBarCtrl)base.OwnerController;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 描画
        /// </summary>
        protected internal override void Draw(Graphics gc, Rectangle clientRectangle, Object information, bool enabled)

        {
            TrackBarInfo info = information as TrackBarInfo;
            if (null == info) { return; }

            Pen pen = new Pen(SystemColors.ControlDarkDark);
            Brush brush = new SolidBrush(SystemColors.ControlDarkDark);

            Rectangle clientRect = clientRectangle;
            Rectangle positionRect = OwnerController.GetPositionRect(info.Position);


            gc.FillRectangle(new SolidBrush(OwnerController.BackColor), clientRect);

            //溝の描画
            Rectangle rect = OwnerController.ValidRect;
            rect.Y = positionRect.Y + (positionRect.Height / 2) - 2;
            rect.Height = 4;

            ControlPaint.DrawBorder(gc, rect, SystemColors.Control,
                                     ButtonBorderStyle.Inset);

            //ループマーカーの描画
            if (OwnerController.RangeMarkerVisible && info.RangeMarker != null)
            {
                Rectangle br = info.RangeMarker.BeginRectangle;
                Rectangle er = info.RangeMarker.EndRectangle;

                // Rangeの塗り
                Color color = Color.FromArgb(64, SystemColors.ControlDark.R,
                                              SystemColors.ControlDark.G,
                                              SystemColors.ControlDark.B);
                gc.FillRectangle(new SolidBrush(color), br.Left, clientRect.Top,
                                  er.Right - br.Left, clientRect.Height);

                // RangeMarkerの描画（▼マーク）
                SolidBrush loopMarkerBrush = new SolidBrush(SystemColors.ControlDarkDark);
                {
                    Point[] points = new Point[3];
                    points[0].X = br.Left;
                    points[0].Y = br.Y;
                    points[1].X = br.Left;
                    points[1].Y = br.Y + br.Height;
                    points[2].X = br.Right;
                    points[2].Y = br.Y;

                    gc.FillPolygon(loopMarkerBrush, points);
                }
                {
                    Point[] points = new Point[3];
                    points[0].X = er.Right;
                    points[0].Y = er.Y;
                    points[1].X = er.Right;
                    points[1].Y = er.Y + br.Height;
                    points[2].X = er.Left;
                    points[2].Y = er.Y;

                    gc.FillPolygon(loopMarkerBrush, points);
                }

            }

            if (enabled != false)
            {

                //サムの描画
                positionRect = OwnerController.GetPositionRect(info.Position);
                ControlPaint.DrawButton(gc, positionRect, ButtonState.Normal);
            }

            //目盛りの描画
            if (info.Graduations != null)
            {
                //int			y = clientRect.Bottom - info.GraduationHeight;
                int y = info.GraduationY;
                int height = info.GraduationHeight;

                foreach (GraduationInfo graduationInfo in info.Graduations)
                {
                    // マーカーとの間に2pxスペースを空けるために、height - 3
                    gc.DrawLine(pen,
                                 graduationInfo.Position, y,
                                 graduationInfo.Position, y + height - 3);
                }
            }

            //マーカーの描画
            if (info.Markers != null)
            {

                int y = info.MarkerY;
                int height = info.MarkerHeight;

                foreach (TrackBarMarker marker in info.Markers)
                {

                    bool focused = (null != info.FocusedMarker)
                        && (marker.Marker == info.FocusedMarker);
                    bool playable = !OwnerController.RangeMarkerVisible
                        || (OwnerController.RangeBeginMarkerValue <= marker.Marker.Value
                             && marker.Marker.Value <= OwnerController.RangeEndMarkerValue);

                    marker.Marker.Draw(gc, marker.Rectangle, new TrackBarMakerDrawInfo(focused, playable));

                }

            }

            //
            if (info.EnableDragMarker != false)
            {
                positionRect = OwnerController.GetPositionRect
                    (info.DragMarkerPosition);
                ControlPaint.DrawButton(gc, positionRect, ButtonState.Pushed);
            }

            //フォーカス線の描画
            if (OwnerController.Focused != false)
            {
                ControlPaint.DrawFocusRectangle(gc, clientRect);
            }
        }
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class TrackBarEventArgs : EventArgs
    {
        private bool _PressedControl = false;
        private bool _PressedShift = false;
        private bool _PressedAlt = false;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public TrackBarEventArgs()
        {
        }

        ///
        public TrackBarEventArgs(bool pressedControl, bool pressedShift, bool pressedAlt)
        {
            _PressedControl = pressedControl;
            _PressedShift = pressedShift;
            _PressedAlt = pressedAlt;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool PressedControl
        {
            get { return _PressedControl; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool PressedShift
        {
            get { return _PressedShift; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool PressedAlt
        {
            get { return _PressedAlt; }
        }
    }

    public class TrackBarMarkerEventArgs : EventArgs
    {
        private ITrackBarMarker _marker;

        public TrackBarMarkerEventArgs(ITrackBarMarker marker)
        {
            Debug.Assert(null != marker);
            _marker = marker;
        }

        #region ** プロパティ

        public ITrackBarMarker Marker
        {
            get { return _marker; }
        }

        #endregion
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    class TrackBarInfo
    {
        public int Position = -1;

        public bool EnableDragMarker = false;
        public int DragMarkerPosition = -1;

        public int TrackBarY = 0;

        public int ThumbY = -1;
        public int ThumbHeight = -1;

        public GraduationInfo[] Graduations = null;
        public int GraduationY = -1;
        public int GraduationHeight = -1;

        public TrackBarMarker[] Markers = null;
        public int MarkerY = -1;
        public int MarkerHeight = -1;
        public ITrackBarMarker FocusedMarker = null;

        public TrackBarRangeMarker RangeMarker = null;
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    class GraduationInfo
    {
        public int Position = 0;
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    class TrackBarMarker
    {
        public Rectangle Rectangle;
        public ITrackBarMarker Marker = null;
    }


    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    class TrackBarRangeMarker
    {
        public Rectangle BeginRectangle;
        public Rectangle EndRectangle;
    }


    public class TrackBarMakerDrawInfo
    {
        #region ** パラメータ

        private bool _focused;
        private bool _playable;

        #endregion

        public TrackBarMakerDrawInfo(bool focused, bool playable)
        {
            _focused = focused;
            _playable = playable;
        }

        #region ** プロパティ

        public bool Focused
        {
            get { return _focused; }
        }

        public bool Playable
        {
            get { return _playable; }
        }

        #endregion
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface ITrackBarMarker
    {

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        int Value
        {
            get;
            set;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        string Text
        {
            get;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        void Draw(Graphics gc, Rectangle rect, TrackBarMakerDrawInfo drawInfo);
    }
}
