﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using NintendoWare.SoundFoundation.Core.Win32;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    [ToolboxBitmap(typeof(NumericUpDown))]
    public class NNumericUpDown : NumericUpDown, IValueSynchronizable
    {
        // 読み取り専用時の値を保持
        private decimal _keepValue;
        private bool _forceSet;

        // コンポーネント
        private ValueSynchronizable _valueSync;	// 値の同期用コンポーネント

        public NNumericUpDown()
        {
            // コンポーネントの生成
            _valueSync = new ValueSynchronizable(this);
            Debug.Assert(null != _valueSync);

            // プロパティを初期化する
            base.TextAlign = HorizontalAlignment.Right;		// 右寄せ
        }

        #region ** プロパティ

        /// <summary>
        /// 値の同期用コンポーネント
        /// </summary>
        [Browsable(false)]
        public ValueSynchronizable ValueSynchronized
        {
            get { return _valueSync; }
            set { _valueSync = value; }
        }

        #endregion

        #region ** プロパティの再定義

        /// <summary>
        /// スピン ボックス (アップダウン コントロール) でのテキストの配置を取得または設定します。
        /// </summary>
        [DefaultValue(HorizontalAlignment.Right)]
        public new HorizontalAlignment TextAlign
        {
            get { return base.TextAlign; }
            set { base.TextAlign = value; }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// テキストの取得または設定します。
        /// </summary>
        public void SetText(string text)
        {
            try
            {
                _forceSet = true;
                this.Text = text;
            }
            finally
            {
                _forceSet = false;
            }
        }

        #endregion

        #region ** イベント

        public new event ValueChangedEventHandler ValueChanged
        {
            add { _valueSync.ValueChanged += value; }
            remove { _valueSync.ValueChanged -= value; }
        }

        public event ValueChangedEventHandler Synchronized
        {
            add { _valueSync.Synchronized += value; }
            remove { _valueSync.Synchronized -= value; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnValueChanged(EventArgs e)
        {
            if (this.ReadOnly == true && _forceSet == false)
            {
                if (this.Value != _keepValue)
                {
                    this.Value = _keepValue;    // 読み取り専用だったら元の値に戻して何もしない
                }
                return;
            }

            // 値の精度を Increment 単位に合わせる
            // ただし、指定値 = 上下限値の場合は丸めない
            decimal remainder = base.Value % Increment;

            if (base.Minimum < base.Value && base.Value < base.Maximum && 0 != remainder)
            {

                decimal value = Math.Round(Value / Increment) * Increment;

                if (value < base.Minimum)
                {
                    base.Value = base.Minimum;
                }
                else if (value > base.Maximum)
                {
                    base.Value = base.Maximum;
                }
                else
                {
                    base.Value = value;
                }
            }
            else
            {
                _valueSync.DispatchValueChangedEvent();
            }

            _keepValue = base.Value;
        }

        #endregion

        #region ** プライベートメソッド

        private void IncrementValue(int count, decimal delta)
        {
            decimal newValue = base.Value + delta * count;

            if (newValue > base.Maximum)
            {
                newValue = base.Maximum;
            }
            else if (newValue < base.Minimum)
            {
                newValue = base.Minimum;
            }

            base.Value = newValue;
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        protected override void OnMouseWheel(MouseEventArgs e)
        {
            IncrementValue((e.Delta > 0) ? 1 : -1, Increment);
        }

        /// <summary>
        /// Window プロシージャのオーバーライド
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // マウスホイール
                case WM.WM_MOUSEWHEEL:
                    {
                        // デフォルトの処理を行うとスクロール量が調節できないので、処理を差し替える
                        OnMouseWheel(new MouseEventArgs(MouseButtons, 0, Utility.LOWORD(m.LParam),
                                                          Utility.HIWORD(m.LParam),
                                                          Utility.SignedHIWORD(m.WParam)));
                    }
                    return;
            }

            base.WndProc(ref m);
        }

        #endregion

        #region ** オペレータ

        public static implicit operator ValueSynchronizable(NNumericUpDown src)
        {
            return src.ValueSynchronized;
        }

        #endregion
    }
}
