﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms
{
    public partial class FloatingTrackBar : Form
    {
        private NTrackBar _trackBar = new NTrackBar();
        private bool _thousandsSeparator = false;

        public FloatingTrackBar()
        {
            Debug.Assert(null != _trackBar);

            InitializeComponent();


            _panelTrackBar.Controls.Add(_trackBar);
            _trackBar.Dock = DockStyle.Fill;
        }

        #region ** プロパティ

        protected NTrackBar TrackBar
        {
            get { return _trackBar; }
        }

        public virtual decimal Value
        {
            get { return TrackBar.Value; }
            set { TrackBar.Value = value; }
        }

        public virtual decimal Maximum
        {
            get { return TrackBar.Maximum; }
            set
            {
                TrackBar.Maximum = value;
                _labelMaximum.Text = TrackBar.Maximum.ToString(TextFormat);
            }
        }

        public virtual decimal Minimum
        {
            get { return TrackBar.Minimum; }
            set
            {
                TrackBar.Minimum = value;
                _labelMinimum.Text = TrackBar.Minimum.ToString(TextFormat);
            }
        }

        public virtual decimal SmallChange
        {
            get { return TrackBar.SmallChange; }
            set { TrackBar.SmallChange = value; }
        }

        public virtual decimal LargeChange
        {
            get { return TrackBar.LargeChange; }
            set { TrackBar.LargeChange = value; }
        }

        public virtual decimal TickFrequency
        {
            get { return TrackBar.TickFrequency; }
            set { TrackBar.TickFrequency = value; }
        }

        public virtual int DecimalPlaces
        {
            get { return TrackBar.DecimalPlaces; }
            set { TrackBar.DecimalPlaces = value; }
        }

        public virtual bool ThousandsSeparator
        {
            get { return _thousandsSeparator; }
            set { _thousandsSeparator = value; }
        }


        public virtual decimal Increment
        {
            get { return TrackBar.Increment; }
            set { TrackBar.Increment = value; }
        }

        /// <summary>
        /// トラック バー上に目盛りを表示する方法を示す値を取得または設定します。
        /// </summary>
        public TickStyle TickStyle
        {
            get { return TrackBar.TickStyle; }
            set { TrackBar.TickStyle = value; }
        }

        /// <summary>
        /// 値の同期用コンポーネント
        /// </summary>
        [Browsable(false)]
        public ValueSynchronizable ValueSynchronized
        {
            get { return _trackBar.ValueSynchronized; }
            set { _trackBar.ValueSynchronized = value; }
        }

        /// <summary>
        /// 数値の書式を取得します。
        /// </summary>
        private string TextFormat
        {
            get
            {
                if (0 == TrackBar.DecimalPlaces) { return ""; }

                string format = ".";

                for (int i = 0; i < TrackBar.DecimalPlaces; i++)
                {
                    format += "0";
                }

                return format;
            }
        }

        #endregion

        #region ** イベント

        public event ValueChangedEventHandler ValueChanged
        {
            add { _trackBar.ValueChanged += value; }
            remove { _trackBar.ValueChanged -= value; }
        }

        #endregion

        #region ** イベントハンドラのオーバーライド

        /// <summary>
        /// OnDeactivate のオーバーライド
        /// </summary>
        protected override void OnDeactivate(EventArgs e)
        {
            // PostMessage経由で閉じる
            BeginInvoke(new MethodInvoker(this.Close));

            base.OnDeactivate(e);
        }

        /// <summary>
        /// OnPaint のオーバーライド
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            Rectangle rectBorder = ClientRectangle;
            rectBorder.Width -= 1;
            rectBorder.Height -= 1;
            e.Graphics.DrawRectangle(SystemPens.WindowFrame, rectBorder);
        }

        protected override void OnMove(EventArgs e)
        {
            Point newLocation = Location;

            if (Left < Screen.PrimaryScreen.WorkingArea.Left)
            {
                newLocation.X = Screen.PrimaryScreen.WorkingArea.Left;
            }
            else if (Right > Screen.PrimaryScreen.WorkingArea.Right)
            {
                newLocation.X = Screen.PrimaryScreen.WorkingArea.Right - Width;
            }

            if (Top < Screen.PrimaryScreen.WorkingArea.Top)
            {
                newLocation.Y = Screen.PrimaryScreen.WorkingArea.Top;
            }
            else if (Bottom > Screen.PrimaryScreen.WorkingArea.Bottom)
            {
                newLocation.Y = Screen.PrimaryScreen.WorkingArea.Bottom - Height;
            }

            if (Location != newLocation)
            {
                Location = newLocation;
            }

            base.OnMove(e);
        }

        #endregion

        #region ** オペレータ

        public static implicit operator ValueSynchronizable(FloatingTrackBar src)
        {
            return src._trackBar.ValueSynchronized;
        }

        #endregion
    }
}
