﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundFoundation.Windows.Forms.Windowless
{
    public partial class NWTabControl
    {
        #region ** レイアウト情報

        public class TabLayoutData
        {
            #region ** パラメータ

            private Size _tabItemSize = new Size(59, 20);
            private int _tabItemMaxWidth = 100;

            #endregion

            #region ** プロパティ

            /// <summary>
            /// コントロールのタブのサイズを取得または設定します。
            /// </summary>
            public Size ItemSize
            {
                get { return _tabItemSize; }
                set
                {
                    if (value.Width < 0 || value.Height < 0) { throw new ArgumentOutOfRangeException("value"); }
                    _tabItemSize = value;
                }
            }

            /// <summary>
            /// コントロールのタブの最大幅を取得または設定します。
            /// </summary>
            public int ItemMaxWidth
            {
                get { return _tabItemMaxWidth; }
                set
                {
                    if (value < 0) { throw new ArgumentOutOfRangeException("value"); }
                    _tabItemMaxWidth = value;
                }
            }

            #endregion
        }

        #endregion

        #region ** テンプレート

        public interface INWTabPanelTemplate
        {
            #region ** メソッド

            NWControl CreateTabInstance(Tab tab);

            #endregion
        }

        public class NWTabControlHostTemplate : NWControlHostTemplate
        {
            public NWTabControlHostTemplate(NWControlHost host) : base(host) { }

            public static INWComponentTemplate CreateInstance(NWControlHost host)
            {
                return new NWTabControlHostTemplate(host);
            }

            #region ** メソッド

            protected override NWControlRoot CreateRootControl(NWControlHost host)
            {
                return new NWTabControl(host);
            }

            #endregion
        }

        public class NWTabControlTemplate : NWControlTemplate
        {
            protected NWTabControlTemplate(NWControl control) : base(control) { }

            #region ** プロパティ

            protected NWTabPanel TabPanel
            {
                get { return Control.Controls["TabPanel"] as NWTabPanel; }
            }

            protected NWTabPagePanel TabPagePanel
            {
                get { return Control.Controls["TabPagePanel"] as NWTabPagePanel; }
            }

            #endregion

            #region ** メソッド

            public override void InitializeComponent()
            {
                base.InitializeComponent();

                Control.Controls.Add(new NWTabPanel(Host));
                Control.Controls.Add(new NWTabPagePanel(Host));
            }

            #endregion
        }

        public abstract class NWTabTemplate : NWControlTemplate
        {
            #region ** 固定値

            protected const int CloseButtonMaxSize = NWTabDrawInfo.CloseButtonMaxSize;
            protected const int TabSpacing = NWTabDrawInfo.TabSpacing;

            #endregion

            #region ** フィールド

            private bool _isNormalSize = true;    // lastSize が通常サイズかどうか

            #endregion

            protected NWTabTemplate(NWControl control, INWControlDrawer drawer)
                : base(control, drawer)
            {
                Control.StateChanged += OnControlStateChanged;
                Control.Resize += OnControlResize;
            }

            #region ** プロパティ

            protected virtual Padding TabPadding
            {
                get { return Padding.Empty; }
            }

            protected new NWTab Control
            {
                get { return base.Control as NWTab; }
            }

            protected NWTabControl RootControl
            {
                get { return Control.Root as NWTabControl; }
            }

            #endregion

            #region ** イベントハンドラ

            private void OnControlStateChanged(object sender, NWControlStateEventArgs e)
            {
                if ((e.OldState & NWControlState.Selected) == (e.NewState & NWControlState.Selected)) { return; }
                Control.PerformLayout();
            }

            private void OnControlResize(object sender, EventArgs e)
            {
                _isNormalSize = true;
            }

            #endregion

            #region ** メソッド

            #region ** 初期化

            public override void InitializeComponent()
            {
                base.InitializeComponent();

                Control.Controls.Add(new NWTabButton(Host, NWTab.CloseButtonName));
            }

            #endregion

            #region ** レイアウト

            protected sealed override void PerformLayoutInternal(object layoutData)
            {
                UpdateBounds();

                Rectangle innerRect = GetInnerRect(new Rectangle(0, 0, Control.Bounds.Width, Control.Bounds.Height));
                int buttonSize = RootControl.ShowTabCloseButton ? Math.Min(CloseButtonMaxSize, innerRect.Height) : 0;

                PerformLayoutInternal(innerRect, new Size(buttonSize, buttonSize));
            }

            protected abstract void PerformLayoutInternal(Rectangle innerRect, Size buttonSize);

            protected virtual Rectangle ExpandBounds(Rectangle bounds)
            {
                return bounds;
            }

            protected virtual Rectangle RestoreBounds(Rectangle bounds)
            {
                return bounds;
            }

            private void UpdateBounds()
            {
                if (Control.Selected)
                {

                    if (!_isNormalSize) { return; }

                    Control.Bounds = ExpandBounds(Control.Bounds);
                    _isNormalSize = false;

                    Control.Invalidate();

                }
                else
                {

                    if (_isNormalSize) { return; }

                    // 領域が縮まるので、先に Invalidate を実行しておく
                    Control.Invalidate();

                    Control.Bounds = RestoreBounds(Control.Bounds);
                    _isNormalSize = true;

                }
            }

            private Rectangle GetInnerRect(Rectangle tabInnerRect)
            {
                Rectangle work = tabInnerRect;
                work.X += TabPadding.Left;
                work.Y += TabPadding.Top;
                work.Width -= TabPadding.Horizontal;
                work.Height -= TabPadding.Vertical;

                return work;
            }

            #endregion

            #endregion
        }

        public class NWTabButtonTemplate : NWControlTemplate
        {
            public NWTabButtonTemplate(NWControl control) : base(control, new NWTabButtonDrawerVisualStyle()) { }
        }

        public class NWTabScrollButtonTemplate : NWControlTemplate
        {
            public NWTabScrollButtonTemplate(NWControl control) : base(control, new NWTabScrollButtonDrawerVisualStyle()) { }
        }

        public class NWDropDownButtonTemplate : NWControlTemplate
        {
            public NWDropDownButtonTemplate(NWControl control) : base(control, new NWDropDownButtonDrawerVisualStyle()) { }
        }

        #region ** Horizontal

        public class NWTabPanelTemplateHorizontal : NWControlTemplate, INWTabPanelTemplate
        {
            #region ** 固定値

            public static readonly Padding Padding = new Padding(2, 2, 0, 2);

            protected const int ControlSpacing = 2;
            protected const int ScrollButtonWidth = 14;
            protected const int DropDownButtonSize = 14;

            #endregion

            protected NWTabPanelTemplateHorizontal(NWControl control, INWControlDrawer drawer)
                : base(control, drawer) { }

            #region ** プロパティ

            protected NWTabControl RootControl
            {
                get { return base.Control.Root as NWTabControl; }
            }

            protected new NWTabPanel Control
            {
                get { return base.Control as NWTabPanel; }
            }

            #endregion

            #region ** メソッド

            #region ** 初期化

            public override void InitializeComponent()
            {
                base.InitializeComponent();

                Control.NcControls.Add(new ScrollLeftButton(Host));
                Control.NcControls.Add(new ScrollRightButton(Host));
                Control.NcControls.Add(new DropDownButton(Host));
            }

            #endregion

            #region ** タブの作成

            public NWControl CreateTabInstance(Tab tab)
            {
                Debug.Assert(null != tab, "unexpected error.");
                return new NWTab(Host, tab);
            }

            #endregion

            #region ** レイアウト

            protected override void PerformLayoutInternal(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { throw new Exception("layout data is null."); }

                int current = Padding.Left;

                // タブを再配置する
                foreach (NWTab tab in Control.Controls)
                {

                    Debug.Assert(null != tab.Template, "unexpected error.");
                    Size tabSize = tab.Template.Measure(layoutData);

                    tab.Bounds = new Rectangle(current, Padding.Top, tabSize.Width, tabSize.Height);

                    current += tabSize.Width;

                }
            }

            public override void PerformNcLayout(Rectangle newBounds, ref Rectangle clientRect)
            {
                if (1 >= Control.Controls.Count) { return; }

                NWTab lastTab = Control.Controls[Control.Controls.Count - 1] as NWTab;
                Debug.Assert(null != lastTab, "unexpected error.");

                int tabAreaWidth = newBounds.Width - ControlSpacing + DropDownButtonSize + ControlSpacing;
                int scrollButtonWidth = (tabAreaWidth > 0) ? Math.Min(ScrollButtonWidth, tabAreaWidth) : 0;

                if (newBounds.Right < lastTab.Right)
                {

                    if (0 < scrollButtonWidth)
                    {
                        clientRect.X = scrollButtonWidth + ControlSpacing;
                        clientRect.Width = tabAreaWidth - (scrollButtonWidth + ControlSpacing) * 2;
                    }
                    else
                    {
                        clientRect.Width = tabAreaWidth;
                    }

                }

                PerformLayoutScrollButtons(clientRect);
                PerformLayoutDropDownButton(newBounds);
            }

            private void PerformLayoutScrollButtons(Rectangle newClientRect)
            {
                // 表示する必要がない場合は Empty
                if (0 == newClientRect.Left)
                {
                    Control.ScrollUpButton.Bounds = Rectangle.Empty;
                    Control.ScrollDownButton.Bounds = Rectangle.Empty;
                    return;
                }

                // ScrollUp ボタンを再配置する
                Rectangle scrollUpButtonRect = Rectangle.Empty;
                scrollUpButtonRect.X = Padding.Left;
                scrollUpButtonRect.Y = Padding.Top;
                scrollUpButtonRect.Width = ScrollButtonWidth;
                scrollUpButtonRect.Height = newClientRect.Height - Padding.Vertical;

                Control.ScrollUpButton.Bounds = scrollUpButtonRect;

                // ScrollDown ボタンを再配置する
                Rectangle scrollDownButtonRect = Rectangle.Empty;
                scrollDownButtonRect.X = Math.Max(newClientRect.Right + ControlSpacing,
                                                        scrollUpButtonRect.Right);
                scrollDownButtonRect.Y = Padding.Top;
                scrollDownButtonRect.Width = ScrollButtonWidth;
                scrollDownButtonRect.Height = newClientRect.Height - Padding.Vertical;

                Control.ScrollDownButton.Bounds = scrollDownButtonRect;
            }

            private void PerformLayoutDropDownButton(Rectangle newBounds)
            {
                // 表示する必要がない場合は Empty
                if (!RootControl.ShowDropDownButton || 1 >= Control.Controls.Count)
                {
                    Control.DropDownButton.Bounds = Rectangle.Empty;
                    return;
                }

                Rectangle bounds = Rectangle.Empty;
                bounds.X = newBounds.Width - DropDownButtonSize - ControlSpacing;
                bounds.Y = (newBounds.Height - DropDownButtonSize) / 2;
                bounds.Width = DropDownButtonSize;
                bounds.Height = DropDownButtonSize;

                Control.DropDownButton.Bounds = bounds;
            }

            #endregion

            #endregion
        }

        public class NWTabTemplateHorizontal : NWTabTemplate
        {
            protected NWTabTemplateHorizontal(NWControl control, INWControlDrawer drawer)
                : base(control, drawer) { }

            #region ** メソッド

            #region ** レイアウト

            public override Size Measure(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { return Size.Empty; }

                NWTabDrawInfo drawInfo = CreateControlDrawInfo() as NWTabDrawInfo;
                Debug.Assert(null != drawInfo, "unexpected error.");

                Graphics g = Graphics.FromHwnd(Control.Host.Owner.Handle);

                SizeF size = g.MeasureString(Control.Model.Text,
                                             new Font(Control.Host.Owner.Font, FontStyle.Bold),
                                             0, drawInfo.TextFormat);

                // パディング
                size.Width += TabPadding.Horizontal;
                size.Height += TabPadding.Vertical;

                // 閉じるボタン + 閉じるボタンとのスペース
                if (RootControl.ShowTabCloseButton)
                {
                    size.Width += CloseButtonMaxSize + TabSpacing;
                }

                // イメージ + イメージとのスペース
                if (null != Control.Image)
                {
                    size.Width += Control.Image.Size.Width + TabSpacing;
                }

                // サイズ上下限チェック
                if (size.Width < data.ItemSize.Width)
                {
                    size.Width = data.ItemSize.Width;
                }
                if (size.Width > data.ItemMaxWidth)
                {
                    size.Width = data.ItemMaxWidth;
                }
                if (size.Height < data.ItemSize.Height)
                {
                    size.Height = data.ItemSize.Height;
                }

                return new Size((int)Math.Ceiling(size.Width), (int)Math.Ceiling(size.Height));
            }

            protected override void PerformLayoutInternal(Rectangle innerRect, Size buttonSize)
            {
                Rectangle closeButtonBounds = Rectangle.Empty;

                if (RootControl.ShowTabCloseButton)
                {
                    closeButtonBounds.X = innerRect.Right - buttonSize.Width;
                    closeButtonBounds.Y = innerRect.Top + (innerRect.Height - buttonSize.Height) / 2;
                    closeButtonBounds.Width = buttonSize.Width;
                    closeButtonBounds.Height = buttonSize.Height;
                }

                Control.CloseButton.Bounds = closeButtonBounds;
            }

            #endregion

            #endregion
        }

        #region ** Top

        public class NWTabControlTemplateTop : NWTabControlTemplate
        {
            public NWTabControlTemplateTop(NWControl control) : base(control) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabControlTemplateTop(control);
            }

            #region ** メソッド

            protected override void PerformLayoutInternal(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { return; }

                NWTabControl tabControl = Control as NWTabControl;
                if (null == tabControl) { return; }


                Rectangle validRect = tabControl.ClientRectangle;
                validRect.X += tabControl.Owner.Padding.Left;
                validRect.Y += tabControl.Owner.Padding.Top;
                validRect.Width -= tabControl.Owner.Padding.Horizontal;
                validRect.Height -= tabControl.Owner.Padding.Vertical;

                Rectangle tabPanelBounds = Rectangle.Empty;
                tabPanelBounds.X = validRect.Left;
                tabPanelBounds.Y = validRect.Top;
                tabPanelBounds.Width = validRect.Width;
                tabPanelBounds.Height = data.ItemSize.Height
                                             + NWTabPanelTemplateHorizontal.Padding.Vertical;

                Rectangle tabPageBounds = Rectangle.Empty;
                tabPageBounds.X = validRect.Left;
                tabPageBounds.Y = tabPanelBounds.Bottom + 1;
                tabPageBounds.Width = validRect.Width;
                tabPageBounds.Height = validRect.Bottom - tabPageBounds.Top;

                TabPanel.Bounds = tabPanelBounds;
                TabPagePanel.Bounds = tabPageBounds;
            }

            #endregion
        }

        public class NWTabPanelTemplateTop : NWTabPanelTemplateHorizontal
        {
            public NWTabPanelTemplateTop(NWControl control)
                : base(control, new NWTabPanelDrawerTop()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabPanelTemplateTop(control);
            }

            #region ** メソッド

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                Rectangle workRect = Control.AbsoluteBounds;
                workRect.Y = workRect.Bottom - Padding.Bottom;
                workRect.Height = Padding.Bottom;

                NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                drawInfo.BorderRect = workRect;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        public class NWTabTemplateTop : NWTabTemplateHorizontal
        {
            public NWTabTemplateTop(NWControl control) : base(control, new NWTabDrawerTop()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabTemplateTop(control);
            }

            #region ** プロパティ

            protected override Padding TabPadding
            {
                get { return NWTabTopDrawInfo.TabPadding; }
            }

            #endregion

            #region ** メソッド

            #region ** レイアウト

            protected override Rectangle ExpandBounds(Rectangle bounds)
            {
                Rectangle newBounds = bounds;
                newBounds.X -= 2;
                newBounds.Y -= 2;
                newBounds.Width += 4;
                newBounds.Height += 3;

                return newBounds;
            }

            protected override Rectangle RestoreBounds(Rectangle bounds)
            {
                Rectangle newBounds = bounds;
                newBounds.X += 2;
                newBounds.Y += 2;
                newBounds.Width -= 4;
                newBounds.Height -= 3;

                return newBounds;
            }

            #endregion

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                NWTabDrawInfo drawInfo = new NWTabTopDrawInfo(Control);

                drawInfo.TextFormat.LineAlignment = StringAlignment.Center;
                drawInfo.TextFormat.FormatFlags = StringFormatFlags.NoWrap;
                drawInfo.TextFormat.Trimming = StringTrimming.EllipsisCharacter;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** Bottom

        public class NWTabControlTemplateBottom : NWTabControlTemplate
        {
            public NWTabControlTemplateBottom(NWControl control) : base(control) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabControlTemplateBottom(control);
            }

            #region ** メソッド

            protected override void PerformLayoutInternal(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { return; }

                NWTabControl tabControl = Control as NWTabControl;
                if (null == tabControl) { return; }


                Rectangle validRect = tabControl.ClientRectangle;
                validRect.X += tabControl.Owner.Padding.Left;
                validRect.Y += tabControl.Owner.Padding.Top;
                validRect.Width -= tabControl.Owner.Padding.Horizontal;
                validRect.Height -= tabControl.Owner.Padding.Vertical;

                Rectangle tabPageBounds = Rectangle.Empty;
                tabPageBounds.X = validRect.Left;
                tabPageBounds.Y = validRect.Top;
                tabPageBounds.Width = validRect.Width;
                tabPageBounds.Height = validRect.Height - data.ItemSize.Height
                                            - NWTabPanelTemplateBottom.Padding.Vertical;

                Rectangle tabPanelBounds = Rectangle.Empty;
                tabPanelBounds.X = validRect.Left;
                tabPanelBounds.Y = tabPageBounds.Bottom + 1;
                tabPanelBounds.Width = validRect.Width;
                tabPanelBounds.Height = data.ItemSize.Height + NWTabPanelTemplateBottom.Padding.Vertical;

                TabPanel.Bounds = tabPanelBounds;
                TabPagePanel.Bounds = tabPageBounds;
            }

            #endregion
        }

        public class NWTabPanelTemplateBottom : NWTabPanelTemplateHorizontal
        {
            public NWTabPanelTemplateBottom(NWControl control)
                : base(control, new NWTabPanelDrawerBottom()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabPanelTemplateBottom(control);
            }

            #region ** メソッド

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                Rectangle workRect = Control.AbsoluteBounds;
                workRect.Height = Padding.Top;

                NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                drawInfo.BorderRect = workRect;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        public class NWTabTemplateBottom : NWTabTemplateHorizontal
        {
            public NWTabTemplateBottom(NWControl control) : base(control, new NWTabDrawerBottom()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabTemplateBottom(control);
            }

            #region ** プロパティ

            protected override Padding TabPadding
            {
                get { return NWTabBottomDrawInfo.TabPadding; }
            }

            #endregion

            #region ** メソッド

            #region ** レイアウト

            protected override Rectangle ExpandBounds(Rectangle bounds)
            {
                Rectangle newBounds = bounds;
                newBounds.X -= 2;
                newBounds.Y -= 1;
                newBounds.Width += 4;
                newBounds.Height += 3;

                return newBounds;
            }

            protected override Rectangle RestoreBounds(Rectangle bounds)
            {
                Rectangle newBounds = bounds;
                newBounds.X += 2;
                newBounds.Y += 1;
                newBounds.Width -= 4;
                newBounds.Height -= 3;

                return newBounds;
            }

            #endregion

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                NWTabDrawInfo drawInfo = new NWTabBottomDrawInfo(Control);

                drawInfo.TextFormat.LineAlignment = StringAlignment.Center;
                drawInfo.TextFormat.FormatFlags = StringFormatFlags.NoWrap;
                drawInfo.TextFormat.Trimming = StringTrimming.EllipsisCharacter;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** Minimized

        public class NWTabControlTemplateHorizontalMinimized : NWTabControlTemplate
        {
            public NWTabControlTemplateHorizontalMinimized(NWControl control) : base(control) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabControlTemplateHorizontalMinimized(control);
            }

            #region ** メソッド

            protected override void PerformLayoutInternal(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { return; }

                NWTabControl tabControl = Control as NWTabControl;
                if (null == tabControl) { return; }


                Rectangle validRect = tabControl.ClientRectangle;
                validRect.X += tabControl.Owner.Padding.Left;
                validRect.Y += tabControl.Owner.Padding.Top;
                validRect.Width -= tabControl.Owner.Padding.Horizontal;
                validRect.Height -= tabControl.Owner.Padding.Vertical;

                Rectangle tabPanelBounds = Rectangle.Empty;
                tabPanelBounds.X = validRect.Left;
                tabPanelBounds.Y = validRect.Top;
                tabPanelBounds.Width = validRect.Width;
                tabPanelBounds.Height = data.ItemSize.Height
                                             + NWTabPanelTemplateHorizontal.Padding.Vertical;

                TabPanel.Bounds = tabPanelBounds;
                TabPagePanel.Bounds = Rectangle.Empty;

                Control.MinimumSize = new Size(0, tabPanelBounds.Height);
            }

            #endregion
        }

        public class NWTabPanelTemplateHorizontalMinimized : NWTabPanelTemplateHorizontal
        {
            public NWTabPanelTemplateHorizontalMinimized(NWControl control) : base(control, new NWControlDrawer()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabPanelTemplateHorizontalMinimized(control);
            }
        }

        #endregion

        #endregion

        #region ** Vertical

        public class NWTabPanelTemplateVertical : NWControlTemplate, INWTabPanelTemplate
        {
            #region ** 固定値

            public static readonly Padding Padding = new Padding(2, 2, 2, 0);

            protected const int ControlSpacing = 2;
            protected const int ScrollButtonHeight = 14;
            protected const int DropDownButtonSize = 14;

            #endregion

            protected NWTabPanelTemplateVertical(NWControl control, INWControlDrawer drawer)
                : base(control, drawer) { }

            #region ** プロパティ

            protected NWTabControl RootControl
            {
                get { return base.Control.Root as NWTabControl; }
            }

            protected new NWTabPanel Control
            {
                get { return base.Control as NWTabPanel; }
            }

            #endregion

            #region ** メソッド

            #region ** 初期化

            public override void InitializeComponent()
            {
                base.InitializeComponent();

                Control.NcControls.Add(new ScrollUpButton(Host));
                Control.NcControls.Add(new ScrollDownButton(Host));
                Control.NcControls.Add(new DropDownButton(Host));
            }

            #endregion

            #region ** タブの作成

            public NWControl CreateTabInstance(Tab tab)
            {
                Debug.Assert(null != tab, "unexpected error.");
                return new NWTab(Host, tab);
            }

            #endregion

            #region ** レイアウト

            protected override void PerformLayoutInternal(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { throw new Exception("layout data is null."); }

                int current = Padding.Top;

                // タブを再配置する
                foreach (NWTab tab in Control.Controls)
                {

                    Debug.Assert(null != tab.Template, "unexpected error.");
                    Size tabSize = tab.Template.Measure(layoutData);

                    tab.Bounds = new Rectangle(Padding.Left, current, tabSize.Width, tabSize.Height);

                    current += tabSize.Height;

                }
            }

            public override void PerformNcLayout(Rectangle newBounds, ref Rectangle clientRect)
            {
                if (1 >= Control.Controls.Count) { return; }

                NWTab lastTab = Control.Controls[Control.Controls.Count - 1] as NWTab;
                Debug.Assert(null != lastTab, "unexpected error.");

                if (newBounds.Bottom < lastTab.Bottom)
                {
                    clientRect.Y = ScrollButtonHeight + ControlSpacing;
                    clientRect.Height = newBounds.Height - (ScrollButtonHeight + ControlSpacing) * 2;
                }

                clientRect.Height -= ControlSpacing + DropDownButtonSize + ControlSpacing;

                PerformLayoutScrollButtons(clientRect);
                PerformLayoutDropDownButton(newBounds);
            }

            private void PerformLayoutScrollButtons(Rectangle newClientRect)
            {
                // 表示する必要がない場合は Empty
                if (0 == newClientRect.Top)
                {
                    Control.ScrollUpButton.Bounds = Rectangle.Empty;
                    Control.ScrollDownButton.Bounds = Rectangle.Empty;
                    return;
                }

                // ScrollUp ボタンを再配置する
                Rectangle scrollUpButtonRect = Rectangle.Empty;
                scrollUpButtonRect.X = Padding.Left;
                scrollUpButtonRect.Y = Padding.Top;
                scrollUpButtonRect.Width = newClientRect.Width - Padding.Horizontal;
                scrollUpButtonRect.Height = ScrollButtonHeight;

                Control.ScrollUpButton.Bounds = scrollUpButtonRect;

                // ScrollDown ボタンを再配置する
                Rectangle scrollDownButtonRect = Rectangle.Empty;
                scrollDownButtonRect.X = Padding.Left;
                scrollDownButtonRect.Y = Math.Max(newClientRect.Bottom + ControlSpacing,
                                                        scrollUpButtonRect.Bottom);
                scrollDownButtonRect.Width = newClientRect.Width - Padding.Horizontal;
                scrollDownButtonRect.Height = ScrollButtonHeight;

                Control.ScrollDownButton.Bounds = scrollDownButtonRect;
            }

            private void PerformLayoutDropDownButton(Rectangle newBounds)
            {
                // 表示する必要がない場合は Empty
                if (!RootControl.ShowDropDownButton || 1 >= Control.Controls.Count)
                {
                    Control.DropDownButton.Bounds = Rectangle.Empty;
                    return;
                }

                Rectangle bounds = Rectangle.Empty;
                bounds.X = (newBounds.Width - DropDownButtonSize) / 2;
                bounds.Y = newBounds.Height - DropDownButtonSize - ControlSpacing;
                bounds.Width = DropDownButtonSize;
                bounds.Height = DropDownButtonSize;

                Control.DropDownButton.Bounds = bounds;
            }

            #endregion

            #endregion
        }

        public class NWTabTemplateVertical : NWTabTemplate
        {
            protected NWTabTemplateVertical(NWControl control, INWControlDrawer drawer)
                : base(control, drawer) { }

            #region ** メソッド

            #region ** レイアウト

            public override Size Measure(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { return Size.Empty; }

                NWTabDrawInfo drawInfo = CreateControlDrawInfo() as NWTabDrawInfo;
                Debug.Assert(null != drawInfo, "unexpected error.");

                Graphics g = Graphics.FromHwnd(Control.Host.Owner.Handle);

                SizeF size = g.MeasureString(Control.Model.Text,
                                              new Font(Control.Host.Owner.Font, FontStyle.Bold),
                                              0, drawInfo.TextFormat);

                // パディング
                size.Width += TabPadding.Horizontal;
                size.Height += TabPadding.Vertical;

                // 閉じるボタン + 閉じるボタンとのスペース
                if (RootControl.ShowTabCloseButton)
                {
                    size.Height += CloseButtonMaxSize + TabSpacing;
                }

                // イメージ + イメージとのスペース
                if (null != Control.Image)
                {
                    size.Height += Control.Image.Size.Height + TabSpacing;
                }

                // サイズ上下限チェック
                if (size.Width < data.ItemSize.Height)
                {
                    size.Width = data.ItemSize.Height;
                }
                if (size.Height < data.ItemSize.Width)
                {
                    size.Height = data.ItemSize.Width;
                }
                if (size.Height > data.ItemMaxWidth)
                {
                    size.Height = data.ItemMaxWidth;
                }

                return new Size((int)Math.Ceiling(size.Width), (int)Math.Ceiling(size.Height));
            }

            protected override void PerformLayoutInternal(Rectangle innerRect, Size buttonSize)
            {
                Rectangle closeButtonBounds = Rectangle.Empty;

                if (RootControl.ShowTabCloseButton)
                {
                    closeButtonBounds.X = innerRect.Left + (innerRect.Width - buttonSize.Width) / 2;
                    closeButtonBounds.Y = innerRect.Bottom - buttonSize.Height;
                    closeButtonBounds.Width = buttonSize.Width;
                    closeButtonBounds.Height = buttonSize.Height;
                }

                Control.CloseButton.Bounds = closeButtonBounds;
            }

            #endregion

            #endregion
        }

        #region ** Left

        public class NWTabControlTemplateLeft : NWTabControlTemplate
        {
            public NWTabControlTemplateLeft(NWControl control) : base(control) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabControlTemplateLeft(control);
            }

            #region ** メソッド

            protected override void PerformLayoutInternal(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { return; }

                NWTabControl tabControl = Control as NWTabControl;
                if (null == tabControl) { return; }


                Rectangle validRect = tabControl.ClientRectangle;
                validRect.X += tabControl.Owner.Padding.Left;
                validRect.Y += tabControl.Owner.Padding.Top;
                validRect.Width -= tabControl.Owner.Padding.Horizontal;
                validRect.Height -= tabControl.Owner.Padding.Vertical;

                Rectangle tabPanelBounds = Rectangle.Empty;
                tabPanelBounds.X = validRect.Left;
                tabPanelBounds.Y = validRect.Top;
                tabPanelBounds.Width = data.ItemSize.Height + NWTabPanelTemplateVertical.Padding.Horizontal;
                tabPanelBounds.Height = validRect.Height;

                Rectangle tabPageBounds = Rectangle.Empty;
                tabPageBounds.X = tabPanelBounds.Right + 1;
                tabPageBounds.Y = validRect.Top;
                tabPageBounds.Width = validRect.Right - tabPageBounds.Left;
                tabPageBounds.Height = validRect.Height;

                TabPanel.Bounds = tabPanelBounds;
                TabPagePanel.Bounds = tabPageBounds;
            }

            #endregion
        }

        public class NWTabPanelTemplateLeft : NWTabPanelTemplateVertical
        {
            public NWTabPanelTemplateLeft(NWControl control)
                : base(control, new NWTabPanelDrawerLeft()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabPanelTemplateLeft(control);
            }

            #region ** メソッド

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                Rectangle workRect = Control.AbsoluteBounds;
                workRect.X = workRect.Right - Padding.Right;
                workRect.Width = Padding.Right;

                NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                drawInfo.BorderRect = workRect;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        public class NWTabTemplateLeft : NWTabTemplateVertical
        {
            public NWTabTemplateLeft(NWControl control) : base(control, new NWTabDrawerLeft()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabTemplateLeft(control);
            }

            #region ** プロパティ

            protected override Padding TabPadding
            {
                get { return NWTabLeftDrawInfo.TabPadding; }
            }

            #endregion

            #region ** メソッド

            #region ** レイアウト

            protected override Rectangle ExpandBounds(Rectangle bounds)
            {
                Rectangle newBounds = bounds;
                newBounds.X -= 2;
                newBounds.Y -= 2;
                newBounds.Width += 3;
                newBounds.Height += 4;

                return newBounds;
            }

            protected override Rectangle RestoreBounds(Rectangle bounds)
            {
                Rectangle newBounds = bounds;
                newBounds.X += 2;
                newBounds.Y += 2;
                newBounds.Width -= 3;
                newBounds.Height -= 4;

                return newBounds;
            }

            #endregion

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                NWTabDrawInfo drawInfo = new NWTabLeftDrawInfo(Control);

                drawInfo.TextFormat.LineAlignment = StringAlignment.Center;
                drawInfo.TextFormat.FormatFlags = StringFormatFlags.NoWrap | StringFormatFlags.DirectionVertical;
                drawInfo.TextFormat.Trimming = StringTrimming.EllipsisCharacter;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** Right

        public class NWTabControlTemplateRight : NWTabControlTemplate
        {
            public NWTabControlTemplateRight(NWControl control) : base(control) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabControlTemplateRight(control);
            }

            #region ** メソッド

            protected override void PerformLayoutInternal(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { return; }

                NWTabControl tabControl = Control as NWTabControl;
                if (null == tabControl) { return; }


                Rectangle validRect = tabControl.ClientRectangle;
                validRect.X += tabControl.Owner.Padding.Left;
                validRect.Y += tabControl.Owner.Padding.Top;
                validRect.Width -= tabControl.Owner.Padding.Horizontal;
                validRect.Height -= tabControl.Owner.Padding.Vertical;

                Rectangle tabPageBounds = Rectangle.Empty;
                tabPageBounds.X = validRect.Left;
                tabPageBounds.Y = validRect.Top;
                tabPageBounds.Width = validRect.Width - data.ItemSize.Height
                                            - NWTabPanelTemplateRight.Padding.Horizontal;
                tabPageBounds.Height = validRect.Height;

                Rectangle tabPanelBounds = Rectangle.Empty;
                tabPanelBounds.X = tabPageBounds.Right + 1;
                tabPanelBounds.Y = validRect.Top;
                tabPanelBounds.Width = data.ItemSize.Height + NWTabPanelTemplateRight.Padding.Horizontal;
                tabPanelBounds.Height = validRect.Height;

                TabPanel.Bounds = tabPanelBounds;
                TabPagePanel.Bounds = tabPageBounds;
            }

            #endregion
        }

        public class NWTabPanelTemplateRight : NWTabPanelTemplateVertical
        {
            public NWTabPanelTemplateRight(NWControl control)
                : base(control, new NWTabPanelDrawerRight()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabPanelTemplateRight(control);
            }

            #region ** メソッド

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                Rectangle workRect = Control.AbsoluteBounds;
                workRect.Width = Padding.Left;

                NWTabPanelDrawInfo drawInfo = new NWTabPanelDrawInfo(Control);
                drawInfo.BorderRect = workRect;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        public class NWTabTemplateRight : NWTabTemplateVertical
        {
            public NWTabTemplateRight(NWControl control) : base(control, new NWTabDrawerRight()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabTemplateRight(control);
            }

            #region ** プロパティ

            protected override Padding TabPadding
            {
                get { return NWTabRightDrawInfo.TabPadding; }
            }

            #endregion

            #region ** メソッド

            #region ** レイアウト

            protected override Rectangle ExpandBounds(Rectangle bounds)
            {
                Rectangle newBounds = bounds;
                newBounds.X -= 1;
                newBounds.Y -= 2;
                newBounds.Width += 3;
                newBounds.Height += 4;

                return newBounds;
            }

            protected override Rectangle RestoreBounds(Rectangle bounds)
            {
                Rectangle newBounds = bounds;
                newBounds.X += 1;
                newBounds.Y += 2;
                newBounds.Width -= 3;
                newBounds.Height -= 4;

                return newBounds;
            }

            #endregion

            #region ** 描画情報

            protected override NWControlDrawInfo CreateControlDrawInfo()
            {
                NWTabDrawInfo drawInfo = new NWTabRightDrawInfo(Control);

                drawInfo.TextFormat.LineAlignment = StringAlignment.Center;
                drawInfo.TextFormat.FormatFlags = StringFormatFlags.NoWrap | StringFormatFlags.DirectionVertical;
                drawInfo.TextFormat.Trimming = StringTrimming.EllipsisCharacter;

                return drawInfo;
            }

            #endregion

            #endregion
        }

        #endregion

        #region ** Minimized

        public class NWTabControlTemplateVerticalMinimized : NWTabControlTemplate
        {
            public NWTabControlTemplateVerticalMinimized(NWControl control) : base(control) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabControlTemplateVerticalMinimized(control);
            }

            #region ** メソッド

            protected override void PerformLayoutInternal(object layoutData)
            {
                TabLayoutData data = layoutData as TabLayoutData;
                if (null == data) { return; }

                NWTabControl tabControl = Control as NWTabControl;
                if (null == tabControl) { return; }


                Rectangle validRect = tabControl.ClientRectangle;
                validRect.X += tabControl.Owner.Padding.Left;
                validRect.Y += tabControl.Owner.Padding.Top;
                validRect.Width -= tabControl.Owner.Padding.Horizontal;
                validRect.Height -= tabControl.Owner.Padding.Vertical;

                Rectangle tabPanelBounds = Rectangle.Empty;
                tabPanelBounds.X = validRect.Left;
                tabPanelBounds.Y = validRect.Top;
                tabPanelBounds.Width = data.ItemSize.Height + NWTabPanelTemplateVertical.Padding.Horizontal;
                tabPanelBounds.Height = validRect.Height;

                TabPanel.Bounds = tabPanelBounds;
                TabPagePanel.Bounds = Rectangle.Empty;

                Control.MinimumSize = new Size(tabPanelBounds.Width, 0);
            }

            #endregion
        }

        public class NWTabPanelTemplateVerticalMinimized : NWTabPanelTemplateVertical
        {
            public NWTabPanelTemplateVerticalMinimized(NWControl control) : base(control, new NWControlDrawer()) { }

            public static INWControlTemplate CreateInstance(NWControl control)
            {
                return new NWTabPanelTemplateVerticalMinimized(control);
            }
        }

        #endregion

        #endregion

        #endregion

        #region ** 描画情報

        /// <summary>
        /// タブパネル描画情報
        /// </summary>
        public class NWTabPanelDrawInfo : NWControlDrawInfo
        {
            #region ** フィールド

            private Rectangle _borderRect = Rectangle.Empty;

            #endregion

            public NWTabPanelDrawInfo(NWControl tabPanel) : base(tabPanel) { }

            #region ** プロパティ

            #region ** 座標

            public Rectangle BorderRect
            {
                get { return _borderRect; }
                set { _borderRect = value; }
            }

            #endregion

            #endregion
        }

        /// <summary>
        /// タブ描画情報
        /// </summary>
        public abstract class NWTabDrawInfo : NWControlDrawInfo
        {
            #region ** 固定値

            public const int CloseButtonMaxSize = 13;   // 閉じるボタンの最大サイズ
            public const int TabSpacing = 4;    // コンポーネント間のスペース
            protected const int TabPaddingBackForth = 6;    // 余白 前後
            protected const int TabPaddingOver = 4;    // 余白 上
            protected const int TabPaddingUnder = 1;    // 余白 下

            #endregion

            #region ** パラメータ

            private StringFormat _textFormat = new StringFormat();

            #endregion

            public NWTabDrawInfo(NWTab tab) : base(tab) { }

            #region ** プロパティ

            public new NWTab Control
            {
                get { return base.Control as NWTab; }
            }

            public Image Image
            {
                get { return Control.Image; }
            }

            public StringFormat TextFormat
            {
                get { return _textFormat; }
                set
                {
                    if (null == value) { throw new ArgumentNullException(); }
                    _textFormat = value;
                }
            }

            #region ** 状態

            public bool Visible
            {
                get { return Control.Visible; }
            }

            public bool CloseButtonVisible
            {
                get { return (Control.Selected); }
            }

            #endregion

            #region ** 座標

            public Rectangle BackgroundRect
            {
                get { return Control.AbsoluteBounds; }
            }

            public abstract Rectangle TextRect { get; }

            public abstract Rectangle ImageRect { get; }

            protected virtual Padding Padding
            {
                get { return Padding.Empty; }
            }

            protected Rectangle InnerRect
            {
                get
                {
                    Rectangle work = BackgroundRect;
                    work.X += Padding.Left;
                    work.Y += Padding.Top;
                    work.Width -= Padding.Horizontal;
                    work.Height -= Padding.Vertical;

                    return work;
                }
            }

            protected Size ImageSize
            {
                get
                {
                    if (null == Control.Image) { return Size.Empty; }
                    return Control.Image.Size;
                }
            }

            #endregion

            #endregion
        }

        /// <summary>
        /// タブ描画情報
        /// </summary>
        public class NWTabHorizontalDrawInfo : NWTabDrawInfo
        {
            public NWTabHorizontalDrawInfo(NWTab tab) : base(tab) { }

            #region ** プロパティ

            #region ** 座標

            public override Rectangle TextRect
            {
                get
                {
                    Rectangle innerRect = InnerRect;
                    Rectangle imageRect = ImageRect;

                    Rectangle work = innerRect;

                    if (imageRect.Width > 0)
                    {
                        work.X = imageRect.Right + TabSpacing;
                    }

                    if (Control.CloseButton.Visible)
                    {
                        work.Width = Control.CloseButton.AbsoluteBounds.Left - work.X;
                    }
                    else
                    {
                        work.Width = innerRect.Right - work.X;
                    }

                    // テキストは少し上に描画されるので、ここで微調整。
                    work.Y += 1;

                    return work;
                }
            }

            public override Rectangle ImageRect
            {
                get
                {
                    if (null == Control.Image) { return Rectangle.Empty; }

                    Rectangle innerRect = InnerRect;
                    Size imageSize = ImageSize;
                    Rectangle work = new Rectangle();

                    work.X = innerRect.Left;
                    work.Y = innerRect.Top + (innerRect.Height - imageSize.Height) / 2;
                    work.Width = imageSize.Width;
                    work.Height = imageSize.Height;

                    return work;
                }
            }

            #endregion

            #endregion
        }

        /// <summary>
        /// タブ描画情報
        /// </summary>
        public class NWTabTopDrawInfo : NWTabHorizontalDrawInfo
        {
            #region ** 固定値

            public static readonly Padding TabPadding = new Padding(TabPaddingBackForth, TabPaddingOver,
                                                                     TabPaddingBackForth, TabPaddingUnder);

            #endregion

            public NWTabTopDrawInfo(NWTab tab) : base(tab) { }

            #region ** プロパティ

            #region ** 座標

            protected override Padding Padding
            {
                get { return TabPadding; }
            }

            #endregion

            #endregion
        }

        /// <summary>
        /// タブ描画情報
        /// </summary>
        public class NWTabBottomDrawInfo : NWTabHorizontalDrawInfo
        {
            #region ** 固定値

            public static readonly Padding TabPadding = new Padding(TabPaddingBackForth, TabPaddingUnder,
                                                                     TabPaddingBackForth, TabPaddingOver);

            #endregion

            public NWTabBottomDrawInfo(NWTab tab) : base(tab) { }

            #region ** プロパティ

            #region ** 座標

            protected override Padding Padding
            {
                get { return TabPadding; }
            }

            #endregion

            #endregion
        }

        /// <summary>
        /// タブ描画情報
        /// </summary>
        public class NWTabVerticalDrawInfo : NWTabDrawInfo
        {
            public NWTabVerticalDrawInfo(NWTab tab) : base(tab) { }

            #region ** プロパティ

            #region ** 座標

            public override Rectangle TextRect
            {
                get
                {
                    Rectangle innerRect = InnerRect;
                    Rectangle imageRect = ImageRect;

                    Rectangle work = innerRect;

                    if (imageRect.Height > 0)
                    {
                        work.Y = imageRect.Bottom + TabSpacing;
                    }

                    if (Control.CloseButton.Visible)
                    {
                        work.Height = Control.CloseButton.AbsoluteBounds.Top - work.Y;
                    }
                    else
                    {
                        work.Height = innerRect.Bottom - work.Y;
                    }

                    // テキストは少し右に描画されるので、ここで微調整。
                    work.X -= 1;

                    return work;
                }
            }

            public override Rectangle ImageRect
            {
                get
                {
                    if (null == Control.Image) { return Rectangle.Empty; }

                    Rectangle innerRect = InnerRect;
                    Size imageSize = ImageSize;
                    Rectangle work = new Rectangle();

                    work.X = innerRect.Left + (innerRect.Width - imageSize.Width) / 2;
                    work.Y = innerRect.Top;
                    work.Width = imageSize.Width;
                    work.Height = imageSize.Height;

                    return work;
                }
            }

            #endregion

            #endregion
        }

        /// <summary>
        /// タブ描画情報
        /// </summary>
        public class NWTabLeftDrawInfo : NWTabVerticalDrawInfo
        {
            #region ** 固定値

            public static readonly Padding TabPadding = new Padding(TabPaddingOver, TabPaddingBackForth,
                                                                     TabPaddingUnder, TabPaddingBackForth);

            #endregion

            public NWTabLeftDrawInfo(NWTab tab) : base(tab) { }

            #region ** プロパティ

            #region ** 座標

            protected override Padding Padding
            {
                get { return TabPadding; }
            }

            #endregion

            #endregion
        }

        /// <summary>
        /// タブ描画情報
        /// </summary>
        public class NWTabRightDrawInfo : NWTabVerticalDrawInfo
        {
            #region ** 固定値

            public static readonly Padding TabPadding = new Padding(TabPaddingUnder, TabPaddingBackForth,
                                                                     TabPaddingOver, TabPaddingBackForth);

            #endregion

            public NWTabRightDrawInfo(NWTab tab) : base(tab) { }

            #region ** プロパティ

            #region ** 座標

            protected override Padding Padding
            {
                get { return TabPadding; }
            }

            #endregion

            #endregion
        }

        #endregion
    }
}
