﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Text;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;


namespace NintendoWare.SoundFoundation.Windows.Forms.Windowless
{
    public partial class NWTabControl
    {
        #region ** コントロール描画クラス

        public class NWTabPanelDrawer : NWControlDrawer
        {
            private Brush _backBrush = null;
            private Pen _edgePen = null;

            protected Pen EdgePen
            {
                get { return _edgePen; }
            }

            protected override void OnDraw(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!(drawInfo is NWTabPanelDrawInfo))
                {
                    Debug.Fail("target is not TabPanel.");
                    return;
                }
                if (!drawInfo.Control.Visible) { return; }

                Draw(g, drawInfo as NWTabPanelDrawInfo);
            }

            protected override void OnUpdateDrawComponent(NWControl control)
            {
                base.OnUpdateDrawComponent(control);

                UpdateDrawComponentInternal();
            }

            protected virtual void DrawEdge(Graphics g, Rectangle borderRect)
            {
            }

            private void UpdateDrawComponentInternal()
            {
                if (null != _backBrush)
                {
                    _backBrush.Dispose();
                }

                if (null != _edgePen)
                {
                    _edgePen.Dispose();
                }

                _backBrush = new SolidBrush(
                    NWColor.Blend(NWColor.Blend(NWColor.ThemeBorder, Color.White, 64), Color.White, 96));

                _edgePen = new Pen(NWColor.ThemeBorder);
            }

            private void Draw(Graphics g, NWTabPanelDrawInfo drawInfo)
            {
                Debug.Assert(null != g, "unexpected error.");
                Debug.Assert(null != drawInfo, "unexpected error.");

                if (0 == drawInfo.Control.Controls.Count) { return; }

                if (null == _backBrush || null == _edgePen)
                {
                    UpdateDrawComponentInternal();
                }

                g.FillRectangle(_backBrush, drawInfo.BorderRect);

                DrawEdge(g, drawInfo.BorderRect);
            }
        }

        public abstract class NWTabDrawer : NWControlDrawer
        {
            #region ** 固定値

            protected const int CornerDefaultSize = 4;

            #endregion

            #region ** フィールド

            private Rectangle _lastBgRectangle = Rectangle.Empty;    // 最近描画した背景領域
            private GraphicsPath _bgPath = null;               // 背景パス
            private Brush _bgBrush = null;               // 背景ブラシ
            private Brush _bgSelectedBrush = null;               // 背景ブラシ（選択）
            private Brush _textBrush = null;               // テキストブラシ
            private Font _selectedFont = null;               // フォント（選択）

            #endregion

            #region ** イベントハンドラ

            protected override void OnDraw(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!(drawInfo is NWTabDrawInfo))
                {
                    Debug.Fail("target is not Tab.");
                    return;
                }
                if (!drawInfo.Control.Visible) { return; }

                Draw(g, drawInfo as NWTabDrawInfo);
            }

            protected override void OnUpdateDrawComponent(NWControl control)
            {
                if (null == control) { return; }

                if (null != _bgBrush)
                {
                    _bgBrush.Dispose();
                }
                if (null != _bgSelectedBrush)
                {
                    _bgSelectedBrush.Dispose();
                }
                if (null != _textBrush)
                {
                    _textBrush.Dispose();
                }
                if (null != _selectedFont)
                {
                    _selectedFont.Dispose();
                }

                Rectangle rect = control.AbsoluteBounds;

                _bgBrush = CreateBrush(rect, SystemColors.Control);
                _bgSelectedBrush = CreateSelectedBrush(rect,
                                        NWColor.Blend(NWColor.ThemeBorder, Color.White, 96));

                _textBrush = new SolidBrush(control.Host.Owner.ForeColor);
                _selectedFont = new Font(control.Host.Owner.Font, FontStyle.Bold);
            }

            #endregion

            #region ** メソッド

            protected abstract GraphicsPath CreateBackgroundPath(NWTabDrawInfo drawInfo);

            private void Draw(Graphics g, NWTabDrawInfo drawInfo)
            {
                Debug.Assert(null != g, "unexpected error.");
                Debug.Assert(null != drawInfo, "unexpected error.");

                if (null == _bgPath || _lastBgRectangle != drawInfo.BackgroundRect)
                {

                    if (null != _bgPath)
                    {
                        _bgPath.Dispose();
                    }

                    _bgPath = CreateBackgroundPath(drawInfo);

                    UpdateDrawComponent(drawInfo.Control);

                }
                else if (null == _bgBrush || null == _bgSelectedBrush ||
                    null == _selectedFont || null == _textBrush)
                {
                    UpdateDrawComponent(drawInfo.Control);
                }


                Pen pen = null;
                Brush brush = null;

                if (!drawInfo.Control.Enabled)
                {
                    pen = SystemPens.ControlDark;
                    brush = new SolidBrush(SystemColors.Control);
                }
                else if (drawInfo.Control.Selected || drawInfo.Control.Hot)
                {
                    pen = new Pen(NWColor.ThemeBorder);
                    brush = _bgSelectedBrush;
                }
                else
                {
                    pen = SystemPens.ControlDark;
                    brush = _bgBrush;
                }

                Font font = (drawInfo.Control.Selected &&
                              (drawInfo.Control.Root as NWTabControl).SelectionExpression) ?
                                                _selectedFont : drawInfo.Control.Host.Owner.Font;


                // クリッピング
                Region oldClip = g.Clip;
                g.SetClip(drawInfo.ClipRect);

                // 背景を描画する
                g.FillPath(brush, _bgPath);

                SmoothingMode oldSmoothingMode = g.SmoothingMode;
                g.SmoothingMode = SmoothingMode.AntiAlias;

                g.DrawPath(pen, _bgPath);

                g.SmoothingMode = oldSmoothingMode;

                // テキストを描画する
                g.DrawString(drawInfo.Control.Model.Text, font, _textBrush,
                                drawInfo.TextRect, drawInfo.TextFormat);

                // イメージを描画する
                if (null != drawInfo.Image)
                {
                    g.DrawImage(drawInfo.Image, drawInfo.ImageRect);
                }

                g.Clip = oldClip;

                _lastBgRectangle = drawInfo.BackgroundRect;
            }

            protected virtual Brush CreateBrush(Rectangle rect, Color color)
            {
                return new SolidBrush(NWColor.Blend(color, ControlPaint.Dark(color), 224));
            }

            protected abstract Brush CreateSelectedBrush(Rectangle rect, Color color);

            #endregion
        }

        #region ** Horizontal Drawer

        public abstract class NWTabHorizontalDrawer : NWTabDrawer
        {
            #region ** メソッド

            protected override Brush CreateSelectedBrush(Rectangle rect, Color color)
            {
                Color lightColor = NWColor.Blend(color, Color.White, 64);

                ColorBlend workBlend = new ColorBlend();
                workBlend.Colors = new Color[] { lightColor, lightColor, color, color, lightColor };
                workBlend.Positions = new float[] { 0.0f, 0.4f, 0.5f, 0.7f, 1.0f };

                LinearGradientBrush workBrush = new LinearGradientBrush(rect, color, color,
                                                                         LinearGradientMode.Vertical);
                workBrush.InterpolationColors = workBlend;

                return workBrush;
            }

            #endregion
        }

        #region ** Top Drawer

        public class NWTabPanelDrawerTop : NWTabPanelDrawer
        {
            protected override void DrawEdge(Graphics g, Rectangle borderRect)
            {
                g.DrawLines(EdgePen,
                    new Point[]
                    {
                        new Point( borderRect.Left, borderRect.Bottom ),
                        new Point( borderRect.Left, borderRect.Top ),
                        new Point( borderRect.Right - 1, borderRect.Top ),
                        new Point( borderRect.Right - 1, borderRect.Bottom ),
                    });
            }
        }

        public class NWTabDrawerTop : NWTabHorizontalDrawer
        {
            protected override GraphicsPath CreateBackgroundPath(NWTabDrawInfo drawInfo)
            {
                Rectangle drawRect = drawInfo.BackgroundRect;
                drawRect.Y += 2;
                drawRect.Width -= (drawInfo.Control.Selected) ? 1 : 2;
                drawRect.Height -= 2;

                GraphicsPath newPath = new GraphicsPath();

                int cornerSize = (CornerDefaultSize < drawInfo.BackgroundRect.Width && CornerDefaultSize < drawInfo.BackgroundRect.Height)
                                    ? CornerDefaultSize : Math.Min(drawRect.Width, drawRect.Height);
                int arcSize = cornerSize * 2;

                newPath.AddLine(drawRect.Left, drawRect.Bottom, drawRect.Left, drawRect.Top + cornerSize);
                newPath.AddArc(drawRect.Left, drawRect.Top, arcSize, arcSize, 180, 90);
                newPath.AddLine(drawRect.Left + cornerSize, drawRect.Top, drawRect.Right - cornerSize, drawRect.Top);
                newPath.AddArc(drawRect.Right - arcSize, drawRect.Top, arcSize, arcSize, 270, 90);
                newPath.AddLine(drawRect.Right, drawRect.Top + cornerSize, drawRect.Right, drawRect.Bottom);

                return newPath;
            }
        }

        #endregion

        #region ** Bottom Drawer

        public class NWTabPanelDrawerBottom : NWTabPanelDrawer
        {
            protected override void DrawEdge(Graphics g, Rectangle borderRect)
            {
                g.DrawLines(EdgePen,
                    new Point[]
                    {
                        new Point( borderRect.Left, borderRect.Top ),
                        new Point( borderRect.Left, borderRect.Bottom ),
                        new Point( borderRect.Right - 1, borderRect.Bottom ),
                        new Point( borderRect.Right - 1, borderRect.Top ),
                    });
            }
        }

        public class NWTabDrawerBottom : NWTabHorizontalDrawer
        {
            protected override GraphicsPath CreateBackgroundPath(NWTabDrawInfo drawInfo)
            {
                Rectangle drawRect = drawInfo.BackgroundRect;
                drawRect.Width -= (drawInfo.Control.Selected) ? 1 : 2;
                drawRect.Height -= 2;

                GraphicsPath newPath = new GraphicsPath();

                int corner = (CornerDefaultSize < drawInfo.BackgroundRect.Width && CornerDefaultSize < drawInfo.BackgroundRect.Height)
                                ? CornerDefaultSize * 2 : 0;

                newPath.AddLine(drawRect.Left, drawRect.Top + 1, drawRect.Left, drawRect.Bottom - corner);
                newPath.AddArc(drawRect.Left, drawRect.Bottom - corner, corner, corner, -180, -90);
                newPath.AddLine(drawRect.Left + corner, drawRect.Bottom, drawRect.Right - corner, drawRect.Bottom);
                newPath.AddArc(drawRect.Right - corner, drawRect.Bottom - corner, corner, corner, -270, -90);
                newPath.AddLine(drawRect.Right, drawRect.Bottom - corner, drawRect.Right, drawRect.Top + 1);

                return newPath;
            }
        }

        #endregion

        #endregion

        #region ** Vertical Drawer

        public abstract class NWTabVerticalDrawer : NWTabDrawer
        {
            protected override Brush CreateSelectedBrush(Rectangle rect, Color color)
            {
                Color lightColor = NWColor.Blend(color, Color.White, 64);

                ColorBlend workBlend = new ColorBlend();
                workBlend.Colors = new Color[] { lightColor, color, color, lightColor, lightColor };
                workBlend.Positions = new float[] { 0.0f, 0.3f, 0.5f, 0.6f, 1.0f };

                LinearGradientBrush workBrush = new LinearGradientBrush(rect, color, color,
                                                                         LinearGradientMode.Horizontal);
                workBrush.InterpolationColors = workBlend;

                return workBrush;
            }
        }

        #region ** Left Drawer

        public class NWTabPanelDrawerLeft : NWTabPanelDrawer
        {
            protected override void DrawEdge(Graphics g, Rectangle borderRect)
            {
                g.DrawLines(EdgePen,
                    new Point[]
                    {
                        new Point( borderRect.Right, borderRect.Top ),
                        new Point( borderRect.Left,  borderRect.Top ),
                        new Point( borderRect.Left,  borderRect.Bottom - 1 ),
                        new Point( borderRect.Right, borderRect.Bottom - 1 ),
                    });
            }
        }

        public class NWTabDrawerLeft : NWTabVerticalDrawer
        {
            protected override GraphicsPath CreateBackgroundPath(NWTabDrawInfo drawInfo)
            {
                Rectangle drawRect = drawInfo.BackgroundRect;
                drawRect.X += 2;
                drawRect.Width -= 2;
                drawRect.Height -= (drawInfo.Control.Selected) ? 1 : 2;

                GraphicsPath newPath = new GraphicsPath();

                int cornerSize = (CornerDefaultSize < drawInfo.BackgroundRect.Width && CornerDefaultSize < drawInfo.BackgroundRect.Height)
                                    ? CornerDefaultSize : Math.Min(drawRect.Width, drawRect.Height);
                int arcSize = cornerSize * 2;

                newPath.AddLine(drawRect.Right, drawRect.Bottom, drawRect.Left + cornerSize, drawRect.Bottom);
                newPath.AddArc(drawRect.Left, drawRect.Bottom - arcSize, arcSize, arcSize, 90, 90);
                newPath.AddLine(drawRect.Left, drawRect.Bottom - cornerSize, drawRect.Left, drawRect.Top + cornerSize);
                newPath.AddArc(drawRect.Left, drawRect.Top, arcSize, arcSize, 180, 90);
                newPath.AddLine(drawRect.Left + cornerSize, drawRect.Top, drawRect.Right, drawRect.Top);

                return newPath;
            }
        }

        #endregion

        #region ** Right Drawer

        public class NWTabPanelDrawerRight : NWTabPanelDrawer
        {
            protected override void DrawEdge(Graphics g, Rectangle borderRect)
            {
                g.DrawLines(EdgePen,
                    new Point[]
                    {
                        new Point( borderRect.Left,  borderRect.Top ),
                        new Point( borderRect.Right, borderRect.Top ),
                        new Point( borderRect.Right, borderRect.Bottom - 1 ),
                        new Point( borderRect.Left,  borderRect.Bottom - 1 ),
                    });
            }
        }

        public class NWTabDrawerRight : NWTabVerticalDrawer
        {
            protected override GraphicsPath CreateBackgroundPath(NWTabDrawInfo drawInfo)
            {
                Rectangle drawRect = drawInfo.BackgroundRect;
                drawRect.Width -= 2;
                drawRect.Height -= (drawInfo.Control.Selected) ? 1 : 2;

                GraphicsPath newPath = new GraphicsPath();

                int cornerSize = (CornerDefaultSize < drawInfo.BackgroundRect.Width && CornerDefaultSize < drawInfo.BackgroundRect.Height)
                                    ? CornerDefaultSize : Math.Min(drawRect.Width, drawRect.Height);
                int arcSize = cornerSize * 2;

                newPath.AddLine(drawRect.Left, drawRect.Top, drawRect.Right - cornerSize, drawRect.Top);
                newPath.AddArc(drawRect.Right - arcSize, drawRect.Top, arcSize, arcSize, 270, 90);
                newPath.AddLine(drawRect.Right, drawRect.Top + cornerSize, drawRect.Right, drawRect.Bottom - cornerSize);
                newPath.AddArc(drawRect.Right - arcSize, drawRect.Bottom - arcSize, arcSize, arcSize, 0, 90);
                newPath.AddLine(drawRect.Right - cornerSize, drawRect.Bottom, drawRect.Left, drawRect.Bottom);

                return newPath;
            }
        }

        #endregion

        #endregion

        #region ** VisualStyle Drawer

        public class NWTabDrawerVisualStyle : NWControlDrawer
        {
            #region ** イベントハンドラ

            protected override void OnDraw(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!(drawInfo is NWTabDrawInfo))
                {
                    Debug.Fail("target is not Tab.");
                    return;
                }
                if (!drawInfo.Control.Visible) { return; }

                if (VisualStyleRenderer.IsSupported)
                {
                    DrawForVisualStyle(g, drawInfo as NWTabDrawInfo);
                }
                else
                {
                    DrawForClassicStyle(g, drawInfo as NWTabDrawInfo);
                }
            }

            #endregion

            #region ** メソッド

            private void DrawForVisualStyle(Graphics g, NWTabDrawInfo drawInfo)
            {
                VisualStyleRenderer renderer = GetRenderer(drawInfo.Control);

                // 背景を描画する
                renderer.DrawBackground(g, drawInfo.BackgroundRect, drawInfo.ClipRect);

                // テキストを描画する
                Rectangle textRect = Rectangle.Intersect(drawInfo.TextRect, drawInfo.ClipRect);
                if (!textRect.IsEmpty)
                {
                    g.DrawString(drawInfo.Control.Model.Text, drawInfo.Control.Host.Owner.Font,
                                    new SolidBrush(drawInfo.Control.Host.Owner.ForeColor),
                                    textRect, drawInfo.TextFormat);
                }

                // イメージを描画する
                if (null != drawInfo.Image)
                {
                    g.DrawImage(drawInfo.Image, drawInfo.ImageRect);
                }
            }

            private void DrawForClassicStyle(Graphics g, NWTabDrawInfo drawInfo)
            {
                ButtonState drawState = ButtonState.Normal;

                if (!drawInfo.Control.Enabled)
                {
                    drawState = ButtonState.Inactive;
                }
                else if (drawInfo.Control.Selected)
                {
                    drawState = ButtonState.Pushed;
                }

                ControlPaint.DrawButton(g, drawInfo.Control.AbsoluteBounds, drawState);
            }

            private VisualStyleRenderer GetRenderer(NWTab tab)
            {
                if (tab.Selected)
                {
                    return new VisualStyleRenderer(VisualStyleElement.Tab.TabItem.Pressed);
                }
                else if (tab.Hot)
                {
                    return new VisualStyleRenderer(VisualStyleElement.Tab.TabItem.Hot);
                }

                return new VisualStyleRenderer(VisualStyleElement.Tab.TabItem.Normal);
            }

            #endregion
        }

        public class NWTabButtonDrawerVisualStyle : NWControlDrawer
        {
            #region ** イベントハンドラ

            /// <summary>
            /// コントロールを描画します。
            /// </summary>
            /// <param name="g">Graphics オブジェクト</param>
            /// <param name="drawInfo">コントロールの描画情報</param>
            protected override void OnDraw(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!(drawInfo.Control is NWTabButton))
                {
                    Debug.Fail("target is not NWTabButton.");
                    return;
                }
                if (!drawInfo.Control.Visible) { return; }

                if (VisualStyleRenderer.IsSupported)
                {
                    DrawForVisualStyle(g, drawInfo);
                }
                else
                {
                    DrawForClassicStyle(g, drawInfo);
                }
            }

            #endregion

            #region ** メソッド

            private void DrawForVisualStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                VisualStyleRenderer renderer = null;

                if (!drawInfo.Control.Enabled)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.Window.SmallCloseButton.Disabled);
                }
                else if (drawInfo.Control.Selected)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.Window.SmallCloseButton.Pressed);
                }
                else if (drawInfo.Control.Hot)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.Window.SmallCloseButton.Hot);
                }
                else
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.Window.SmallCloseButton.Normal);
                }

                renderer.DrawBackground(g, drawInfo.Control.AbsoluteBounds);
            }

            private void DrawForClassicStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!drawInfo.Control.Enabled)
                {
                    ControlPaint.DrawCaptionButton(g, drawInfo.Control.AbsoluteBounds, CaptionButton.Close, ButtonState.Inactive);
                }
                else if (drawInfo.Control.Selected)
                {
                    ControlPaint.DrawCaptionButton(g, drawInfo.Control.AbsoluteBounds, CaptionButton.Close, ButtonState.Pushed);
                }
                else if (drawInfo.Control.Hot)
                {
                    ControlPaint.DrawCaptionButton(g, drawInfo.Control.AbsoluteBounds, CaptionButton.Close, ButtonState.Normal);
                }
                else
                {
                    ControlPaint.DrawCaptionButton(g, drawInfo.Control.AbsoluteBounds, CaptionButton.Close, ButtonState.Flat);
                }
            }

            #endregion
        }

        public class NWTabScrollButtonDrawerVisualStyle : NWControlDrawer
        {
            #region ** イベントハンドラ

            /// <summary>
            /// コントロールを描画します。
            /// </summary>
            /// <param name="g">Graphics オブジェクト</param>
            /// <param name="drawInfo">コントロールの描画情報</param>
            protected override void OnDraw(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!drawInfo.Control.Visible) { return; }

                if (drawInfo.Control is ScrollRightButton)
                {
                    DrawScrollRightButton(g, drawInfo);
                }
                else if (drawInfo.Control is ScrollLeftButton)
                {
                    DrawScrollLeftButton(g, drawInfo);
                }
                else if (drawInfo.Control is ScrollDownButton)
                {
                    DrawScrollDownButton(g, drawInfo);
                }
                else if (drawInfo.Control is ScrollUpButton)
                {
                    DrawScrollUpButton(g, drawInfo);
                }
                else
                {
                    Debug.Fail("target is not NWTabButton.");
                }
            }

            #endregion

            #region ** メソッド

            public void DrawScrollRightButton(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (VisualStyleRenderer.IsSupported)
                {
                    DrawScrollRightButtonForVisualStyle(g, drawInfo);
                }
                else
                {
                    DrawScrollRightButtonForClassicStyle(g, drawInfo);
                }
            }

            public void DrawScrollLeftButton(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (VisualStyleRenderer.IsSupported)
                {
                    DrawScrollLeftButtonForVisualStyle(g, drawInfo);
                }
                else
                {
                    DrawScrollLeftButtonForClassicStyle(g, drawInfo);
                }
            }

            public void DrawScrollDownButton(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (VisualStyleRenderer.IsSupported)
                {
                    DrawScrollDownButtonForVisualStyle(g, drawInfo);
                }
                else
                {
                    DrawScrollDownButtonForClassicStyle(g, drawInfo);
                }
            }

            public void DrawScrollUpButton(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (VisualStyleRenderer.IsSupported)
                {
                    DrawScrollUpButtonForVisualStyle(g, drawInfo);
                }
                else
                {
                    DrawScrollUpButtonForClassicStyle(g, drawInfo);
                }
            }

            private void DrawScrollRightButtonForVisualStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                VisualStyleRenderer renderer = null;

                if (!drawInfo.Control.Enabled)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.RightDisabled);
                }
                else if (drawInfo.Control.Selected)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.RightPressed);
                }
                else if (drawInfo.Control.Hot)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.RightHot);
                }
                else
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.RightNormal);
                }

                renderer.DrawBackground(g, drawInfo.Control.AbsoluteBounds);
            }

            private void DrawScrollLeftButtonForVisualStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                VisualStyleRenderer renderer = null;

                if (!drawInfo.Control.Enabled)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.LeftDisabled);
                }
                else if (drawInfo.Control.Selected)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.LeftPressed);
                }
                else if (drawInfo.Control.Hot)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.LeftHot);
                }
                else
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.LeftNormal);
                }

                renderer.DrawBackground(g, drawInfo.Control.AbsoluteBounds);
            }

            private void DrawScrollRightButtonForClassicStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!drawInfo.Control.Enabled)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Right, ButtonState.Inactive);
                }
                else if (drawInfo.Control.Selected)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Right, ButtonState.Pushed);
                }
                else if (drawInfo.Control.Hot)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Right, ButtonState.Normal);
                }
                else
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Right, ButtonState.Flat);
                }
            }

            private void DrawScrollLeftButtonForClassicStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!drawInfo.Control.Enabled)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Left, ButtonState.Inactive);
                }
                else if (drawInfo.Control.Selected)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Left, ButtonState.Pushed);
                }
                else if (drawInfo.Control.Hot)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Left, ButtonState.Normal);
                }
                else
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Left, ButtonState.Flat);
                }
            }

            private void DrawScrollDownButtonForVisualStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                VisualStyleRenderer renderer = null;

                if (!drawInfo.Control.Enabled)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.DownDisabled);
                }
                else if (drawInfo.Control.Selected)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.DownPressed);
                }
                else if (drawInfo.Control.Hot)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.DownHot);
                }
                else
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.DownNormal);
                }

                renderer.DrawBackground(g, drawInfo.Control.AbsoluteBounds);
            }

            private void DrawScrollUpButtonForVisualStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                VisualStyleRenderer renderer = null;

                if (!drawInfo.Control.Enabled)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.UpDisabled);
                }
                else if (drawInfo.Control.Selected)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.UpPressed);
                }
                else if (drawInfo.Control.Hot)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.UpHot);
                }
                else
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ScrollBar.ArrowButton.UpNormal);
                }

                renderer.DrawBackground(g, drawInfo.Control.AbsoluteBounds);
            }

            private void DrawScrollDownButtonForClassicStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!drawInfo.Control.Enabled)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Down, ButtonState.Inactive);
                }
                else if (drawInfo.Control.Selected)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Down, ButtonState.Pushed);
                }
                else if (drawInfo.Control.Hot)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Down, ButtonState.Normal);
                }
                else
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Down, ButtonState.Flat);
                }
            }

            private void DrawScrollUpButtonForClassicStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!drawInfo.Control.Enabled)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Up, ButtonState.Inactive);
                }
                else if (drawInfo.Control.Selected)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Up, ButtonState.Pushed);
                }
                else if (drawInfo.Control.Hot)
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Up, ButtonState.Normal);
                }
                else
                {
                    ControlPaint.DrawScrollButton(g, drawInfo.Control.AbsoluteBounds,
                                                   System.Windows.Forms.ScrollButton.Up, ButtonState.Flat);
                }
            }

            #endregion
        }

        public class NWDropDownButtonDrawerVisualStyle : NWControlDrawer
        {
            #region ** イベントハンドラ

            /// <summary>
            /// コントロールを描画します。
            /// </summary>
            /// <param name="g">Graphics オブジェクト</param>
            /// <param name="drawInfo">コントロールの描画情報</param>
            protected override void OnDraw(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!drawInfo.Control.Visible) { return; }

                if (VisualStyleRenderer.IsSupported)
                {
                    DrawButtonForVisualStyle(g, drawInfo);
                }
                else
                {
                    DrawButtonForClassicStyle(g, drawInfo);
                }
            }

            #endregion

            #region ** メソッド

            private void DrawButtonForVisualStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                VisualStyleRenderer renderer = null;

                if (!drawInfo.Control.Enabled)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ComboBox.DropDownButton.Disabled);
                }
                else if (drawInfo.Control.Selected)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ComboBox.DropDownButton.Pressed);
                }
                else if (drawInfo.Control.Hot)
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ComboBox.DropDownButton.Hot);
                }
                else
                {
                    renderer = new VisualStyleRenderer(VisualStyleElement.ComboBox.DropDownButton.Normal);
                }

                renderer.DrawBackground(g, drawInfo.Control.AbsoluteBounds);
            }

            private void DrawButtonForClassicStyle(Graphics g, NWControlDrawInfo drawInfo)
            {
                if (!drawInfo.Control.Enabled)
                {
                    ControlPaint.DrawComboButton(g, drawInfo.Control.AbsoluteBounds, ButtonState.Inactive);
                }
                else if (drawInfo.Control.Selected)
                {
                    ControlPaint.DrawComboButton(g, drawInfo.Control.AbsoluteBounds, ButtonState.Pushed);
                }
                else if (drawInfo.Control.Hot)
                {
                    ControlPaint.DrawComboButton(g, drawInfo.Control.AbsoluteBounds, ButtonState.Normal);
                }
                else
                {
                    ControlPaint.DrawComboButton(g, drawInfo.Control.AbsoluteBounds, ButtonState.Flat);
                }
            }

            #endregion
        }

        #endregion

        #endregion
    }
}
