﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.IO;
    using System.Text.RegularExpressions;

    /// <summary>
    /// アイテムの名前を付けます。
    /// </summary>
    public static class ItemNamer
    {

        /// <summary>
        /// プロジェクトで一意な名前を作成します。
        /// </summary>
        /// <param name="components">コンポーネントの一覧。</param>
        /// <param name="prefix">接頭辞。</param>
        /// <returns>一意な名前。</returns>
        public static string CreateUniqueName(ComponentDictionary components, string prefix)
        {
            return CreateUniqueName(components, prefix, string.Empty, true);
        }

        /// <summary>
        /// プロジェクトで一意な名前を作成します。
        /// </summary>
        /// <param name="components">コンポーネントの一覧。</param>
        /// <param name="prefix">接頭辞。</param>
        /// <param name="suffix">接尾辞。</param>
        /// <returns>一意な名前。</returns>
        public static string CreateUniqueName(ComponentDictionary components,
                                               string prefix, string suffix)
        {
            return CreateUniqueName(components, prefix, suffix, true);
        }

        /// <summary>
        /// プロジェクトで一意な名前を作成します。
        /// </summary>
        /// <param name="components">コンポーネントの一覧。</param>
        /// <param name="prefix">接頭辞。</param>
        /// <param name="suffix">接尾辞。</param>
        /// <param name="numbered">必ず末尾に数字を付加する場合は true。</param>
        /// <returns>一意な名前。</returns>
        public static string CreateUniqueName(ComponentDictionary components,
                                               string prefix, string suffix, bool numbered)
        {
            if (null == prefix) { throw new ArgumentNullException("prefix"); }
            if (null == suffix) { throw new ArgumentNullException("suffix"); }

            string name = prefix + suffix;

            if (null == components) { return name; }
            if (!numbered && !components.Contains(name)) { return name; }

            int count = 0;

            while (true)
            {
                name = prefix + suffix + count.ToString();

                if (!components.Contains(name))
                {
                    return name;
                }

                count++;
            }
        }

        /// <summary>
        /// ファイル名からプロジェクトで一意な名前を作成します。
        /// </summary>
        /// <param name="components">コンポーネントの一覧。</param>
        /// <param name="settings">命名設定。</param>
        /// <param name="prefix">接頭辞。</param>
        /// <param name="suffix">接尾辞。</param>
        /// <param name="fileName">ファイル名。</param>
        /// <returns>一意な名前。</returns>
        public static string CreateUniqueNameFromFileName(ComponentDictionary components, ItemNamingSettings settings,
                                                           string prefix, string suffix, string fileName)
        {
            if (null == settings) { throw new ArgumentNullException("settings"); }
            if (null == prefix) { throw new ArgumentNullException("prefix"); }
            if (null == suffix) { throw new ArgumentNullException("suffix"); }
            if (null == fileName) { throw new ArgumentNullException("fileName"); }


            string baseName = Path.GetFileNameWithoutExtension(fileName);

            //
            if (settings.EnabledNameDelimiter != false)
            {
                string[] replaceChars = new String[] {
                    ".",
                };

                string delimiter = settings.NameDelimiter;
                foreach (string replaceChar in replaceChars)
                {
                    delimiter = delimiter.Replace(replaceChar, "\\" + replaceChar);
                }

                string expression = String.Format("^.*?(?={0})", delimiter);
                MatchCollection matches = Regex.Matches(baseName, expression);
                if (matches.Count > 0)
                {
                    baseName = matches[0].Value;
                }
            }

            //
            switch (settings.InvalidCharChange)
            {
                case InvalidCharChangeType.ReplaceToUnderscore:
                    baseName = ItemNameValidator.InvalidCharsRegex.Replace(baseName, "_");
                    break;

                case InvalidCharChangeType.RemoveChar:
                    baseName = ItemNameValidator.InvalidCharsRegex.Replace(baseName, string.Empty);
                    break;
            }

            switch (settings.CaseChange)
            {
                case CaseChangeType.ToUpper:
                    baseName = baseName.ToUpper();
                    break;

                case CaseChangeType.ToLower:
                    baseName = baseName.ToLower();
                    break;
            }

            if (settings.HasPrefix)
            {
                baseName = prefix + baseName;
            }

            Regex regex = new Regex("^[a-zA-Z_][0-9a-zA-Z_]*$");
            if (regex.IsMatch(baseName) == false)
            {
                baseName = "_" + baseName;
            }

            return CreateUniqueName(components, baseName, suffix, false);
        }
    }
}
