﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.IO;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;
    using LegacyFormat = NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat;

    /// <summary>
    /// RVL サウンドセットドキュメントをサウンドセットドキュメントに変換します。
    /// </summary>
    public class SoundSetRvlToSoundSetDocumentConverter : IDocumentConverter
    {
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits;
        private IObjectFactory<Type, Component> componentFactory;

        public SoundSetRvlToSoundSetDocumentConverter(
            SoundIntermediateOutputTraits intermediateOutputTraits,
            IObjectFactory<Type, Component> componentFactory)
        {
            Ensure.Argument.NotNull(intermediateOutputTraits);
            Ensure.Argument.NotNull(componentFactory);

            this.intermediateOutputTraits = intermediateOutputTraits;
            this.componentFactory = componentFactory;
        }

        public string InputDocumentTypeName
        {
            get { return Platforms.Rvl.SoundSetDocument; }
        }

        public string OutputDocumentTypeName
        {
            get { return this.intermediateOutputTraits.SoundSetDocumentTypeName; }
        }

        public void Convert(Document inputDocument, Document outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);
            this.ConvertInternal(
                inputDocument as SoundSetRvlDocument,
                outputDocument as SoundSetDocument);
        }

        //-----------------------------------------------------------------
        // サウンドセットアイテムのコンバート
        //-----------------------------------------------------------------

        protected virtual StreamSoundBase ConvertStreamSound(LegacyFormat.Nw4rSoundSetStrm source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Stream Sound: ({0})", source.Label);

            if (source.TrackList.Count == 0)
            {
                throw new Exception("invalid stream sound found.");
            }

            StreamSoundBase dest = componentFactory.Create(typeof(StreamSoundBase)) as StreamSoundBase;
            {
                dest.ActorPlayer = source.ActorPlayer;
                dest.ColorIndex = source.ColorIndex;
                dest.Comment = source.Comment == null ? string.Empty : source.Comment;
                dest.Encoding = this.ConvertWaveEncoding(source.TrackList[0].FileFormat);
                dest.IsEnabled = source.Enabled;
                dest.Name = source.Label;
                dest.PanCurve = this.ConvertPanCurve(source.PanCurve);
                dest.PanMode = this.ConvertPanMode(source.PanMode);
                dest.PlayerPriority = source.PlayerPrio;
                dest.PlayerReference = source.Player;
                dest.Sound3D = this.ConvertSound3D(source.Sound3DParam);
                dest.UserParameter = source.UserParam;
                dest.Volume = source.Volume;
            }

            foreach (LegacyFormat.Nw4rSoundSetStrm.Track track in source.TrackList)
            {
                dest.Children.Add(this.ConvertStreamSoundTrack(track));
            }

            return dest;
        }

        protected virtual SequenceSoundBase ConvertSequenceSound(LegacyFormat.Nw4rSoundSetSeq source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Sequence Sound: ({0})", source.Label);
            SequenceSoundBase dest = componentFactory.Create(typeof(SequenceSoundBase)) as SequenceSoundBase;
            {
                dest.ActorPlayer = source.ActorPlayer;
                dest.ChannelPriority = source.ChannelPrio;
                dest.ColorIndex = source.ColorIndex;
                dest.Comment = source.Comment == null ? string.Empty : source.Comment;
                dest.FilePath = source.FullPath;
                dest.FileType = this.ConvertSequenceSoundFileType(source.FileType);
                dest.IsEnabled = source.Enabled;
                dest.Name = source.Label;
                dest.PlayerPriority = source.PlayerPrio;
                dest.PlayerReference = source.Player;
                dest.ReleasePriorityFixed = source.ReleasePrioFixFlag;
                dest.Sound3D = this.ConvertSound3D(source.Sound3DParam);
                dest.StartPosition = source.StartPosition == null ? string.Empty : source.StartPosition;
                dest.UserParameter = source.UserParam;
                dest.Volume = source.Volume;
            }

            if (dest.FileType == SequenceSoundFileType.Text)
            {
                dest.FilePath = Path.ChangeExtension(
                    dest.FilePath,
                    this.intermediateOutputTraits.TextSequenceSoundFileExtension
                    );
            }

            dest.SoundSetBankReferences.Add(
                new ComponentReference()
                {
                    TargetName = source.Bank,
                }
                );

            return dest;
        }

        protected virtual WaveSoundBase ConvertWaveSound(LegacyFormat.Nw4rSoundSetWaveSound source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Wave Sound: ({0})", source.Label);
            WaveSoundBase dest = componentFactory.Create(typeof(WaveSoundBase)) as WaveSoundBase;
            {
                dest.ActorPlayer = source.ActorPlayer;
                dest.ChannelPriority = source.ChannelPrio;
                dest.ColorIndex = source.ColorIndex;
                dest.Comment = source.Comment == null ? string.Empty : source.Comment;
                dest.Encoding = this.ConvertWaveEncoding(source.FileFormat);
                dest.FilePath = source.FullPath;
                dest.IsEnabled = source.Enabled;
                dest.Name = source.Label;
                dest.Pan = source.Pan;
                dest.PanCurve = this.ConvertPanCurve(source.PanCurve);
                dest.PanMode = this.ConvertPanMode(source.PanMode);
                dest.Pitch = (float)source.Pitch;
                dest.PlayerPriority = source.PlayerPrio;
                dest.PlayerReference = source.Player;
                dest.ReleaseEnvelope = source.AdsrEnvelope.Release;
                dest.ReleasePriorityFixed = source.ReleasePrioFixFlag;
                dest.Sends = new Sends()
                {
                    AuxASend = source.FxSendA,
                    AuxBSend = source.FxSendB,
                    MainSend = source.MainSend,
                };
                dest.Sound3D = this.ConvertSound3D(source.Sound3DParam);
                dest.SurroundPan = source.SurroundPan;
                dest.UserParameter = source.UserParam;
                dest.Volume = source.Volume;
            }

            return dest;
        }


        //-----------------------------------------------------------------
        // パラメータのコンバート
        //-----------------------------------------------------------------

        protected Sound3D ConvertSound3D(LegacyFormat.Nw4rSound3DParam source)
        {
            Assertion.Argument.NotNull(source);

            return new Sound3D()
            {
                DecayCurve3D = this.ConvertDecayCurve3D(source.DecayCurve),
                DecayRatio3D = (float)source.DecayRatio,
                DopplerFactor3D = source.DopplerFactor,
                Enable3DFilter = source.FilterCtrl,
                Enable3DPan = source.PanCtrl,
                Enable3DPriority = source.PriorityCtrl,
                Enable3DSurroundPan = source.SurroundPanCtrl,
                Enable3DVolume = source.VolumeCtrl,
            };
        }

        //-----------------------------------------------------------------
        // 列挙値のコンバート
        //-----------------------------------------------------------------

        protected PanCurve ConvertPanCurve(LegacyFormat.Nw4rXmlPanCurve source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rXmlPanCurve.LINEAR:
                    return PanCurve.Linear;

                case LegacyFormat.Nw4rXmlPanCurve.LINEAR_0DB:
                    return PanCurve.Linear_0Db;

                case LegacyFormat.Nw4rXmlPanCurve.LINEAR_0DB_CLAMP:
                    return PanCurve.Linear_0Db_Clamp;

                case LegacyFormat.Nw4rXmlPanCurve.SINCOS:
                    return PanCurve.SinCos;

                case LegacyFormat.Nw4rXmlPanCurve.SINCOS_0DB:
                    return PanCurve.SinCos_0Db;

                case LegacyFormat.Nw4rXmlPanCurve.SINCOS_0DB_CLAMP:
                    return PanCurve.SinCos_0Db_Clamp;

                case LegacyFormat.Nw4rXmlPanCurve.SQRT:
                    return PanCurve.Sqrt;

                case LegacyFormat.Nw4rXmlPanCurve.SQRT_0DB:
                    return PanCurve.Sqrt_0Db;

                case LegacyFormat.Nw4rXmlPanCurve.SQRT_0DB_CLAMP:
                    return PanCurve.Sqrt_0Db_Clamp;
            }

            return PanCurve.Sqrt;
        }

        protected PanMode ConvertPanMode(LegacyFormat.Nw4rXmlPanMode source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rXmlPanMode.BALANCE:
                    return PanMode.Balance;

                case LegacyFormat.Nw4rXmlPanMode.DUAL:
                    return PanMode.Dual;
            }

            return PanMode.Dual;
        }

        protected WaveEncoding ConvertWaveEncoding(LegacyFormat.Nw4rSoundSetStrmFileFormat source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rSoundSetStrmFileFormat.Adpcm:
                    return WaveEncoding.Adpcm;

                case LegacyFormat.Nw4rSoundSetStrmFileFormat.Pcm16:
                    return WaveEncoding.Pcm16;

                case LegacyFormat.Nw4rSoundSetStrmFileFormat.Pcm8:
                    return WaveEncoding.Pcm8;
            }

            return WaveEncoding.Adpcm;
        }

        protected WaveEncoding ConvertWaveEncoding(LegacyFormat.Nw4rSoundSetWaveSoundFileFormat source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rSoundSetWaveSoundFileFormat.Adpcm:
                    return WaveEncoding.Adpcm;

                case LegacyFormat.Nw4rSoundSetWaveSoundFileFormat.Pcm16:
                    return WaveEncoding.Pcm16;

                case LegacyFormat.Nw4rSoundSetWaveSoundFileFormat.Pcm8:
                    return WaveEncoding.Pcm8;
            }

            return WaveEncoding.Adpcm;
        }

        protected StreamSoundTrackBase ConvertStreamSoundTrack(LegacyFormat.Nw4rSoundSetStrm.Track source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Stream Sound Track: ({0})", source.FilePath);
            StreamSoundTrackBase dest = this.componentFactory.Create(typeof(StreamSoundTrackBase)) as StreamSoundTrackBase;
            dest.FilePath = source.FullPath;
            dest.Pan = source.Pan;
            dest.Volume = source.Volume;

            return dest;
        }

        protected SequenceSoundFileType ConvertSequenceSoundFileType(LegacyFormat.Nw4rSoundSetSeqFileType source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rSoundSetSeqFileType.Smf:
                    return SequenceSoundFileType.Smf;

                case LegacyFormat.Nw4rSoundSetSeqFileType.Text:
                    return SequenceSoundFileType.Text;
            }

            return SequenceSoundFileType.Text;
        }

        private void ConvertInternal(SoundSetRvlDocument inputDocument, SoundSetDocument outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);

            outputDocument.SoundSet = this.ConvertSoundSet(inputDocument.SoundSet);

            if (inputDocument.Resource is FileResource)
            {
                outputDocument.Resource = new FileResource(
                    Path.ChangeExtension(inputDocument.Resource.Key, this.intermediateOutputTraits.SoundSetFileExtension));
            }
        }

        private SoundSet ConvertSoundSet(LegacyFormat.Nw4rSoundSet source)
        {
            Assertion.Argument.NotNull(source);

            SoundSet dest = this.componentFactory.Create(typeof(SoundSet)) as SoundSet;
            Ensure.Operation.ObjectNotNull(dest);

            dest.Children.Add(this.ConvertStreamSoundFolder(source.StrmList));
            dest.Children.Add(this.ConvertWaveSoundSetFolder(source.WaveSoundSetList));
            dest.Children.Add(this.ConvertSequenceSoundSetFolder(source.SeqSoundSetList));
            dest.Children.Add(this.ConvertSequenceSoundFolder(source.SeqList));
            dest.Children.Add(this.ConvertSoundSetBankFolder(source.BankList));
            dest.Children.Add(new WaveArchivePack());
            dest.Children.Add(this.ConvertGroupFolder(source.GroupList));
            dest.Children.Add(this.ConvertPlayerFolder(source.PlayerList));

            return dest;
        }

        //-----------------------------------------------------------------
        // サウンドセットアイテムフォルダのコンバート
        //-----------------------------------------------------------------

        private StreamSoundPack ConvertStreamSoundFolder(LegacyFormat.Nw4rSoundSetStrmList source)
        {
            Assertion.Argument.NotNull(source);

            StreamSoundPack dest = new StreamSoundPack();

            foreach (LegacyFormat.Nw4rSoundSetStrm sourceStreamSound in source)
            {
                dest.Children.Add(this.ConvertStreamSound(sourceStreamSound));
            }

            return dest;
        }

        private WaveSoundSetPack ConvertWaveSoundSetFolder(LegacyFormat.Nw4rSoundSetWaveSoundSetList source)
        {
            Assertion.Argument.NotNull(source);

            WaveSoundSetPack dest = new WaveSoundSetPack();

            foreach (LegacyFormat.Nw4rSoundSetWaveSoundSet sourceWaveSoundSet in source)
            {
                dest.Children.Add(this.ConvertWaveSoundSet(sourceWaveSoundSet));
            }

            return dest;
        }

        private SequenceSoundSetPack ConvertSequenceSoundSetFolder(LegacyFormat.Nw4rSoundSetSeqSoundSetList source)
        {
            Assertion.Argument.NotNull(source);

            SequenceSoundSetPack dest = new SequenceSoundSetPack();

            foreach (LegacyFormat.Nw4rSoundSetSeqSoundSet sourceSequenceSoundSet in source)
            {
                dest.Children.Add(this.ConvertSequenceSoundSet(sourceSequenceSoundSet));
            }

            return dest;
        }

        private SequenceSoundPack ConvertSequenceSoundFolder(LegacyFormat.Nw4rSoundSetSeqList source)
        {
            Assertion.Argument.NotNull(source);

            SequenceSoundPack dest = new SequenceSoundPack();

            foreach (LegacyFormat.Nw4rSoundSetSeq sourceSequenceSound in source)
            {
                dest.Children.Add(this.ConvertSequenceSound(sourceSequenceSound));
            }

            return dest;
        }

        private SoundSetBankPack ConvertSoundSetBankFolder(LegacyFormat.Nw4rSoundSetBankList source)
        {
            Assertion.Argument.NotNull(source);

            SoundSetBankPack dest = new SoundSetBankPack();

            foreach (LegacyFormat.Nw4rSoundSetBank sourceSoundSetBank in source)
            {
                dest.Children.Add(this.ConvertSoundSetBank(sourceSoundSetBank));
            }

            return dest;
        }

        private PlayerPack ConvertPlayerFolder(LegacyFormat.Nw4rSoundSetPlayerList source)
        {
            Assertion.Argument.NotNull(source);

            PlayerPack dest = new PlayerPack();

            foreach (LegacyFormat.Nw4rSoundSetPlayer sourcePlayer in source)
            {
                dest.Children.Add(this.ConvertPlayer(sourcePlayer));
            }

            return dest;
        }

        private GroupPack ConvertGroupFolder(LegacyFormat.Nw4rSoundSetGroupList source)
        {
            Assertion.Argument.NotNull(source);

            GroupPack dest = new GroupPack();

            foreach (LegacyFormat.Nw4rSoundSetGroup sourceGroup in source)
            {
                dest.Children.Add(this.ConvertGroup(sourceGroup));
            }

            return dest;
        }

        private WaveSoundSetBase ConvertWaveSoundSet(LegacyFormat.Nw4rSoundSetWaveSoundSet source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Wave Sound Set: ({0})", source.Label);
            WaveSoundSetBase dest = new WaveSoundSetBase()
            {
                ColorIndex = source.ColorIndex,
                Comment = source.Comment == null ? string.Empty : source.Comment,
                IsEnabled = source.Enabled,
                Name = source.Label,
                WaveArchiveReference = WaveArchiveConsts.AutoShared,
            };

            foreach (LegacyFormat.Nw4rSoundSetWaveSound sourceWaveSound in source)
            {
                dest.Children.Add(this.ConvertWaveSound(sourceWaveSound));
            }

            return dest;
        }

        private SequenceSoundSetBase ConvertSequenceSoundSet(LegacyFormat.Nw4rSoundSetSeqSoundSet source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Sequence Sound Set: ({0})", source.Label);
            SequenceSoundSetBase dest = new SequenceSoundSetBase()
            {
                ColorIndex = source.ColorIndex,
                Comment = source.Comment == null ? string.Empty : source.Comment,
                IsEnabled = source.Enabled,
                Name = source.Label,
            };

            foreach (LegacyFormat.Nw4rSoundSetSeq sourceSequenceSound in source)
            {
                dest.Children.Add(this.ConvertSequenceSound(sourceSequenceSound));
            }

            return dest;
        }

        private SoundSetBankBase ConvertSoundSetBank(LegacyFormat.Nw4rSoundSetBank source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Bank: ({0})", source.Label);
            SoundSetBankBase dest = componentFactory.Create(typeof(SoundSetBankBase)) as SoundSetBankBase;
            {
                dest.ColorIndex = source.ColorIndex;
                dest.Comment = source.Comment == null ? string.Empty : source.Comment;
                dest.FilePath =
                    Path.ChangeExtension(source.FullPath,
                                         this.intermediateOutputTraits.BankFileExtension);
                dest.IsEnabled = source.Enabled;
                dest.Name = source.Label;
                dest.WaveArchiveReference = WaveArchiveConsts.AutoShared;
            }

            return dest;
        }

        private PlayerBase ConvertPlayer(LegacyFormat.Nw4rSoundSetPlayer source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Player: ({0})", source.Label);
            PlayerBase dest = new PlayerBase()
            {
                ColorIndex = source.ColorIndex,
                Comment = source.Comment == null ? string.Empty : source.Comment,
                HeapSize = source.HeapSize,
                IsEnabled = source.Enabled,
                Name = source.Label,
                SoundLimit = source.SoundLimit,
            };

            return dest;
        }

        private GroupBase ConvertGroup(LegacyFormat.Nw4rSoundSetGroup source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Group: ({0})", source.Label);
            GroupBase dest = new GroupBase()
            {
                ColorIndex = source.ColorIndex,
                Comment = source.Comment == null ? string.Empty : source.Comment,
                IsEnabled = source.Enabled,
                OutputType = source.OutputFlag ?
                    GroupOutputType.Embedding : GroupOutputType.None,
                Name = source.Label,
            };

            foreach (LegacyFormat.Nw4rSoundSetGroupItem groupItem in source)
            {
                dest.Children.Add(this.ConvertGroupItem(groupItem));
            }

            return dest;
        }

        private GroupItemBase ConvertGroupItem(LegacyFormat.Nw4rSoundSetGroupItem source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Group Item: ({0})", source.Label);
            GroupItemBase dest = new GroupItemBase()
            {
                TargetItemReference = source.Label,
            };

            return dest;
        }

        private DecayCurve3D ConvertDecayCurve3D(LegacyFormat.Nw4rDecayCurve source)
        {
            switch (source)
            {
                case LegacyFormat.Nw4rDecayCurve.Log:
                    return DecayCurve3D.Log;

                case LegacyFormat.Nw4rDecayCurve.Linear:
                    return DecayCurve3D.Linear;
            }

            return DecayCurve3D.Log;
        }
    }
}
