﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;

    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;
    using MsgRsrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// インポートのログを提供します。
    /// </summary>
    public static class ImportLog
    {
        private static StreamWriter writer = null;

        public static string LogFilePath
        {
            get; set;
        }

        public static void OpenAppend(string path)
        {
            Close();

            ImportLog.LogFilePath = path;
            writer = new StreamWriter(path, true);
        }

        public static void Open(string path)
        {
            Close();

            ImportLog.LogFilePath = path;
            if (File.Exists(path) == true)
            {
                File.Delete(path);
            }
            writer = new StreamWriter(path);
        }

        public static void Close()
        {
            if (writer != null)
            {
                writer.Close();
                writer = null;
            }
        }

        public static void WriteLine(string format, params object[] args)
        {
            if (writer != null)
            {
                string str = string.Empty;

                if (args.Length == 0)
                {
                    str = format;
                }
                else
                {
                    str = string.Format(format, args);
                }

                writer.WriteLine(str);
            }
        }

        public static void Error(string format, params object[] args)
        {
            string str = string.Empty;

            if (args.Length == 0)
            {
                str = format;
            }
            else
            {
                str = string.Format(format, args);
            }

            WriteLine(MsgRsrc.MessageResource.Message_ImportError, str);
        }

        public static void Warning(string format, params object[] args)
        {
            string str = string.Empty;

            if (args.Length == 0)
            {
                str = format;
            }
            else
            {
                str = string.Format(format, args);
            }

            WriteLine(MsgRsrc.MessageResource.Message_ImportWarning, str);
        }
    }

    /// <summary>
    /// ドキュメントのインポート機能を提供します。
    /// </summary>
    public class DocumentImportService
    {
        private readonly DocumentImportServiceTraits traits;

        public DocumentImportService(DocumentImportServiceTraits traits)
        {
            Ensure.Argument.NotNull(traits);
            this.traits = traits;
        }

        /// <summary>
        /// DocumentImportService の特性を取得または設定します。
        /// </summary>
        public DocumentImportServiceTraits Traits
        {
            get { return this.traits; }
        }

        /// <summary>
        /// ドキュメントをインポートする前に発生します。
        /// </summary>
        public event EventHandler<DocumentImportEventArgs> Importing;

        /// <summary>
        /// ドキュメントをインポート中にエラーになった場合に発生します。
        /// </summary>
        public event EventHandler<DocumentImportEventArgs> ImportError;

        /// <summary>
        /// ドキュメントをインポートした際に発生します。
        /// </summary>
        public event EventHandler<DocumentImportEventArgs> Imported;

        /// <summary>
        /// ドキュメントをインポートします。
        /// </summary>
        /// <param name="document">インポートするドキュメントを指定します。</param>
        /// <param name="destinationDocumentTypeName">
        /// インポート後のドキュメントタイプ名を指定します。
        /// </param>
        /// <returns>インポートしたドキュメントへの参照を返します。</returns>
        public DocumentReference Import(Document document, string destinationDocumentTypeName)
        {
            Ensure.Argument.NotNull(document);
            Ensure.Argument.StringNotEmpty(destinationDocumentTypeName);

            IDocumentImporter importer = this.Traits.ImporterFactory.Create(
                new DocumentImportServiceTraits.ImporterKey(
                                                            document.TypeName,
                                                            destinationDocumentTypeName
                                                            )
                );

            DocumentImportEventArgs importingEventArgs = new DocumentImportEventArgs(document);
            this.OnImporting(importingEventArgs);

            if (importingEventArgs.Cancel)
            {
                return null;
            }

            DocumentReference[] importedDocumentRefs = new DocumentReference[0];
            List<Document> importedDocuments = new List<Document>();
            try
            {
                if (importer == null)
                {
                    throw new Exception();
                }

                importedDocumentRefs = importer.Import(document);
                if (importedDocumentRefs == null)
                {
                    return null;
                }
                Ensure.Operation.True(
                                      importedDocumentRefs.Length > 0,
                                      "failed to import documents."
                                      );

                foreach (DocumentReference importedDocumentRef in importedDocumentRefs)
                {
                    importedDocuments.Add(importedDocumentRef.Document);
                }

                DocumentImportEventArgs importedEventArgs =
                    new DocumentImportEventArgs(importedDocuments.ToArray());
                this.OnImported(importedEventArgs);

                if (importedEventArgs.Cancel)
                {
                    return null;
                }

                return importedDocumentRefs[0].Clone();
            }
            catch
            {
                DocumentImportEventArgs importedEventArgs =
                    new DocumentImportEventArgs(importedDocuments.ToArray());
                throw;
            }
            finally
            {
                if (importedDocumentRefs != null)
                {
                    foreach (DocumentReference importedDocumentRef in importedDocumentRefs)
                    {
                        importedDocumentRef.Close();
                    }
                }
            }
        }

        public void OnImportError()
        {
            this.OnImportError(new DocumentImportEventArgs());
        }

        /// <summary>
        /// ドキュメントをインポートする前に発生します。
        /// </summary>
        /// <param name="e">インポートイベントデータを指定します。</param>
        protected virtual void OnImporting(DocumentImportEventArgs e)
        {
            Ensure.Argument.NotNull(e);

            if (this.Importing != null)
            {
                this.Importing(this, e);
            }
        }

        /// <summary>
        /// ドキュメントをインポートした際に発生します。
        /// </summary>
        /// <param name="e">インポートイベントデータを指定します。</param>
        protected virtual void OnImported(DocumentImportEventArgs e)
        {
            Ensure.Argument.NotNull(e);

            if (this.Imported != null)
            {
                this.Imported(this, e);
            }
        }

        /// <summary>
        /// ドキュメントをインポート中にエラーになった場合に発生します。
        /// </summary>
        /// <param name="e">インポートイベントデータを指定します。</param>
        protected virtual void OnImportError(DocumentImportEventArgs e)
        {
            Ensure.Argument.NotNull(e);

            if (this.ImportError != null)
            {
                this.ImportError(this, e);
            }
        }
    }
}
