﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


namespace NintendoWare.SoundFoundation.Projects
{
    using Core.IO;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Resources;
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;

    /// <summary>
    /// サウンドプロジェクトを表します。
    /// </summary>
    public class SoundProject : Component
    {
        public static int SoundArchivePlayerStreamBufferDefalutSize { get; set; }
        public static int SoundArchivePlayerStreamBufferExtensionSize { get; set; }

        // コンバート設定
        public const string ParameterNameOfConvertWarningFlags = "ConvertWarningFlags";
        public event EventHandler ModifiedProjectSetting;

        private readonly Dictionary<string, SoundArchiveOutputTypes> unresolvedSoundArchiveOutputTypes = new Dictionary<string, SoundArchiveOutputTypes>();
        private readonly Dictionary<SoundSet, SoundArchiveOutputTypes> soundArchiveOutputTypes = new Dictionary<SoundSet, SoundArchiveOutputTypes>();

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public SoundProject()
        {
            Router.Bindings.Add
                (new RoutingEventBinding<SoundDocumentBindingEventArgs>
                  (typeof(UpdateDocumentBindingEvent),
                    OnBroadcastUpdateComponentBinding,
                    OnBroadcastUpdateComponentBinding));

            this.SoundListOutputSettings = new SoundListOutputSettings();
            this.UserDataStructureSettings = new UserParameterStructureSettings();
        }

        //-----------------------------------------------------------------
        // パラメータへのアクセサ
        //-----------------------------------------------------------------

        /// <summary>
        /// SMFコンバートの四分音符分解能を取得または設定します。
        /// </summary>
        public int SmfConvertTimeBase
        {
            get { return GetIntValue(ProjectParameterNames.Conversion.SmfTimebase); }
            set { SetIntValue(ProjectParameterNames.Conversion.SmfTimebase, value); }
        }

        /// <summary>
        /// ビルトイン波形プリプロセス exe 利用の有無を取得または設定します。
        /// </summary>
        public bool DoUseBuiltInWavePreprocessExe
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.DoUseBuiltInWavePreprocessExe); }
            set { SetBoolValue(ProjectParameterNames.Conversion.DoUseBuiltInWavePreprocessExe, value); }
        }

        /// <summary>
        /// 波形プリプロセス exe のパスを取得または設定します。
        /// </summary>
        public string WavePreprocessExePath
        {
            get { return GetTextValue(ProjectParameterNames.Conversion.WavePreprocessExePath); }
            set { SetTextValue(ProjectParameterNames.Conversion.WavePreprocessExePath, value); }
        }

        /// <summary>
        /// 中間シーケンスファイルを残すかどうかを取得または設定します。
        /// </summary>
        public bool KeepIntermediateSequenceFile
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.KeepIntermediateTextSequence); }
            set { SetBoolValue(ProjectParameterNames.Conversion.KeepIntermediateTextSequence, value); }
        }

        /// <summary>
        /// 中間ファイルの出力先を取得または設定します。
        /// </summary>
        public string IntermediateOutputPath
        {
            get { return GetTextValue(ProjectParameterNames.Conversion.IntermediateOutputDirectoryPath); }
            set { SetTextValue(ProjectParameterNames.Conversion.IntermediateOutputDirectoryPath, value); }
        }

        /// <summary>
        /// インゲーム編集の出力先を取得または設定します。
        /// </summary>
        public string InGameEditCacheOutputPath
        {
            get { return GetTextValue(ProjectParameterNames.Conversion.InGameEditCacheOutputDirectoryPath); }
            set { SetTextValue(ProjectParameterNames.Conversion.InGameEditCacheOutputDirectoryPath, value); }
        }

        /// <summary>
        /// アイテム別ラベルを出力するかどうかを取得または設定します。
        /// </summary>
        public bool OutputLabel
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.OutputLabel); }
            set { SetBoolValue(ProjectParameterNames.Conversion.OutputLabel, value); }
        }

        /// <summary>
        /// 出力先を取得または設定します。
        /// </summary>
        public string OutputDirectoryPath
        {
            get { return GetTextValue(ProjectParameterNames.Conversion.OutputDirectoryPath); }
            set { SetTextValue(ProjectParameterNames.Conversion.OutputDirectoryPath, value); }
        }

        /// <summary>
        /// 外部ファイルの出力先を取得または設定します。
        /// </summary>
        public string ExternalFileOutputDirectoryPath
        {
            get { return GetTextValue(ProjectParameterNames.Conversion.ExternalFileDirectoryPath); }
            set { SetTextValue(ProjectParameterNames.Conversion.ExternalFileDirectoryPath, value); }
        }

        /// <summary>
        /// ユーザー管理ファイルの出力先を取得または設定します。
        /// </summary>
        public string UserManagementFileOutputDirectoryPath
        {
            get { return GetTextValue(ProjectParameterNames.Conversion.UserManagementFileDirectoryPath); }
            set { SetTextValue(ProjectParameterNames.Conversion.UserManagementFileDirectoryPath, value); }
        }

        /// <summary>
        /// '/' から始まる実機用コンテキストパスが、
        /// PC上でドライブルートを基準としたPC用コンテキストパスとして作用しないように
        /// 先頭の '/' を取り除いた外部ファイル出力先を取得します。
        /// </summary>
        public string ValidExternalFileOutputDirectoryPath
        {
            get
            {
                if (this.ExternalFileOutputDirectoryPath.Length == 0)
                {
                    return string.Empty;
                }

                char firstChar = this.ExternalFileOutputDirectoryPath[0];

                return firstChar == '\\' || firstChar == '/' ?
                    this.ExternalFileOutputDirectoryPath.Substring(1) : this.ExternalFileOutputDirectoryPath;
            }
        }

        /// <summary>
        /// '/' から始まる実機用コンテキストパスが、
        /// PC上でドライブルートを基準としたPC用コンテキストパスとして作用しないように
        /// 先頭の '/' を取り除いたユーザー管理ファイル出力先を取得します。
        /// </summary>
        public string ValidUserManagementFileOutputDirectoryPath
        {
            get
            {
                if (this.UserManagementFileOutputDirectoryPath.Length == 0)
                {
                    return string.Empty;
                }

                char firstChar = this.UserManagementFileOutputDirectoryPath[0];

                return firstChar == '\\' || firstChar == '/' ?
                    this.UserManagementFileOutputDirectoryPath.Substring(1) : this.UserManagementFileOutputDirectoryPath;
            }
        }

        /// <summary>
        /// 文字列テーブルを出力するかどうかを取得または設定します。
        /// </summary>
        public bool ExcludeStringTable
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.ExcludeStringTable); }
            set { SetBoolValue(ProjectParameterNames.Conversion.ExcludeStringTable, value); }
        }

        /// <summary>
        /// PC 版アプリ向けにストリーム AAC の代替バイナリファイル(.bfstm) 出力の有無を取得または設定します。
        /// </summary>
        public bool DoOutputPCBinariesForAAC
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.DoOutputPCBinariesForAAC); }
            set { SetBoolValue(ProjectParameterNames.Conversion.DoOutputPCBinariesForAAC, value); }
        }

        /// <summary>
        /// ターゲットでの Opus のデコードにはハードウェアを使用するように指定する。
        /// </summary>
        public bool DoUseHardwareOpusDecoder
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.DoUseHardwareOpusDecoder); }
            set { SetBoolValue(ProjectParameterNames.Conversion.DoUseHardwareOpusDecoder, value); }
        }

        /// <summary>
        /// このプロジェクトが出力していない出力バイナリファイルを削除するかどうかを取得または設定します。
        /// </summary>
        public bool DoDeleteGarbageOutputBinaries
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.DoDeleteGarbageOutputBinaries); }
            set { SetBoolValue(ProjectParameterNames.Conversion.DoDeleteGarbageOutputBinaries, value); }
        }

        /// <summary>
        /// 未使用のアイテムを警告するかどうかを取得または設定します。
        /// </summary>
        public bool DoWarnUnreferencedItems
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.DoWarnUnreferencedItems); }
            set { SetBoolValue(ProjectParameterNames.Conversion.DoWarnUnreferencedItems, value); }
        }

        /// <summary>
        /// グループアイテムの参照先が無効化されている場合に警告するかどうかを取得または設定します。
        /// </summary>
        public bool DoWarnDisableGroupItemTargets
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.DoWarnDisableGroupItemTargets); }
            set { SetBoolValue(ProjectParameterNames.Conversion.DoWarnDisableGroupItemTargets, value); }
        }

        /// <summary>
        /// ストリーム AAC ファイルに対応する波形ファイル (.aiff/.aif/.wav)が見つからない場合に
        /// 警告するかどうかを取得または設定します。
        /// </summary>
        public bool DoWarnPCBinariesForAACNotFound
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.DoWarnPCBinariesForAACNotFound); }
            set { SetBoolValue(ProjectParameterNames.Conversion.DoWarnPCBinariesForAACNotFound, value); }
        }

        /// <summary>
        /// コンバート前コマンドを実行するかどうかを取得または設定します。
        /// </summary>
        public bool IsPreConvertCommandsEnabled
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.IsPreConvertCommandsEnabled); }
            set { SetBoolValue(ProjectParameterNames.Conversion.IsPreConvertCommandsEnabled, value); }
        }

        /// <summary>
        /// コンバート後コマンドを実行するかどうかを取得または設定します。
        /// </summary>
        public bool IsPostConvertCommandsEnabled
        {
            get { return GetBoolValue(ProjectParameterNames.Conversion.IsPostConvertCommandsEnabled); }
            set { SetBoolValue(ProjectParameterNames.Conversion.IsPostConvertCommandsEnabled, value); }
        }

        /// <summary>
        /// コンバート前コマンドを取得または設定します。
        /// </summary>
        public List<string> PreConvertCommands
        {
            get { return this.GetValue<List<string>>(ProjectParameterNames.Conversion.PreConvertCommands); }
            set { this.SetValue<List<string>>(ProjectParameterNames.Conversion.PreConvertCommands, value); }
        }

        /// <summary>
        /// コンバート後コマンドを取得または設定します。
        /// </summary>
        public List<string> PostConvertCommands
        {
            get { return this.GetValue<List<string>>(ProjectParameterNames.Conversion.PostConvertCommands); }
            set { this.SetValue<List<string>>(ProjectParameterNames.Conversion.PostConvertCommands, value); }
        }

        /// <summary>
        /// サウンドアーカイブのストリームサウンド数を取得または設定します。
        /// </summary>
        public int SoundArchivePlayerStreamSoundCount
        {
            get { return GetIntValue(ProjectParameterNames.SoundArchivePlayer.StreamSoundCount); }
            set { SetIntValue(ProjectParameterNames.SoundArchivePlayer.StreamSoundCount, value); }
        }

        /// <summary>
        /// サウンドアーカイブのストリームチャンネル数を取得または設定します。
        /// </summary>
        public int SoundArchivePlayerStreamChannelCount
        {
            get { return GetIntValue(ProjectParameterNames.SoundArchivePlayer.StreamChannelCount); }
            set { SetIntValue(ProjectParameterNames.SoundArchivePlayer.StreamChannelCount, value); }
        }

        /// <summary>
        /// サウンドアーカイブのストリームバッファサイズの倍率を取得または設定します。
        /// </summary>
        public int SoundArchivePlayerStreamBufferTimes
        {
            get { return GetIntValue(ProjectParameterNames.SoundArchivePlayer.StreamBufferTimes); }
            set { SetIntValue(ProjectParameterNames.SoundArchivePlayer.StreamBufferTimes, value); }
        }

        /// <summary>
        /// サウンドアーカイブのウェーブサウンド数を取得または設定します。
        /// </summary>
        public int SoundArchivePlayerWaveSoundCount
        {
            get { return GetIntValue(ProjectParameterNames.SoundArchivePlayer.WaveSoundCount); }
            set { SetIntValue(ProjectParameterNames.SoundArchivePlayer.WaveSoundCount, value); }
        }

        /// <summary>
        /// サウンドアーカイブのシーケンスサウンド数を取得または設定します。
        /// </summary>
        public int SoundArchivePlayerSequenceSoundCount
        {
            get { return GetIntValue(ProjectParameterNames.SoundArchivePlayer.SequenceSoundCount); }
            set { SetIntValue(ProjectParameterNames.SoundArchivePlayer.SequenceSoundCount, value); }
        }

        /// <summary>
        /// サウンドアーカイブのシーケンストラック数を取得または設定します。
        /// </summary>
        public int SoundArchivePlayerSequenceTrackCount
        {
            get { return GetIntValue(ProjectParameterNames.SoundArchivePlayer.SequenceTrackCount); }
            set { SetIntValue(ProjectParameterNames.SoundArchivePlayer.SequenceTrackCount, value); }
        }

        /// <summary>
        /// ストリームサウンド名の接頭辞を取得または設定します。
        /// </summary>
        public string StreamSoundNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.StreamSoundNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.StreamSoundNamePrefix, value); }
        }

        /// <summary>
        /// ウェーブサウンドセット名の接頭辞を取得または設定します。
        /// </summary>
        public string WaveSoundSetNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.WaveSoundSetNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.WaveSoundSetNamePrefix, value); }
        }

        /// <summary>
        /// ウェーブサウンド名の接頭辞を取得または設定します。
        /// </summary>
        public string WaveSoundNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.WaveSoundNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.WaveSoundNamePrefix, value); }
        }

        /// <summary>
        /// シーケンスサウンドセット名の接頭辞を取得または設定します。
        /// </summary>
        public string SequenceSoundSetNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.SequenceSoundSetNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.SequenceSoundSetNamePrefix, value); }
        }

        /// <summary>
        /// シーケンスサウンド名の接頭辞を取得または設定します。
        /// </summary>
        public string SequenceSoundNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.SequenceSoundNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.SequenceSoundNamePrefix, value); }
        }

        /// <summary>
        /// バンク名の接頭辞を取得または設定します。
        /// </summary>
        public string BankNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.BankNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.BankNamePrefix, value); }
        }

        /// <summary>
        /// プレイヤー名の接頭辞を取得または設定します。
        /// </summary>
        public string PlayerNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.PlayerNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.PlayerNamePrefix, value); }
        }

        /// <summary>
        /// 波形アーカイブ名の接頭辞を取得または設定します。
        /// </summary>
        public string WaveArchiveNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.WaveArchiveNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.WaveArchiveNamePrefix, value); }
        }

        /// <summary>
        /// グループ名の接頭辞を取得または設定します。
        /// </summary>
        public string GroupNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.GroupNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.GroupNamePrefix, value); }
        }

        /// <summary>
        /// インストルメント名の接頭辞を取得または設定します。
        /// </summary>
        public string InstrumentNamePrefix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.InstrumentNamePrefix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.InstrumentNamePrefix, value); }
        }

        /// <summary>
        /// プレフィックスを持つかどうかを取得または設定します。
        /// </summary>
        public bool ItemNamingHasPrefix
        {
            get { return this.GetValue<bool>(ProjectParameterNames.ItemNaming.HasPrefix); }
            set { this.SetValue<bool>(ProjectParameterNames.ItemNaming.HasPrefix, value); }
        }

        /// <summary>
        /// 大文字小文字の扱い設定を取得または設定します。
        /// </summary>
        public CaseChangeType ItemNamingCaseChange
        {
            get { return this.GetValue<CaseChangeType>(ProjectParameterNames.ItemNaming.CaseChange); }
            set { this.SetValue<CaseChangeType>(ProjectParameterNames.ItemNaming.CaseChange, value); }
        }

        /// <summary>
        /// 禁則文字の扱い設定を取得または設定します。
        /// </summary>
        public InvalidCharChangeType ItemNamingInvalidCharChange
        {
            get { return this.GetValue<InvalidCharChangeType>(ProjectParameterNames.ItemNaming.InvalidCharChange); }
            set { this.SetValue<InvalidCharChangeType>(ProjectParameterNames.ItemNaming.InvalidCharChange, value); }
        }

        /// <summary>
        /// アイテム貼り付け時のポストフィックスを取得または設定します。
        /// </summary>
        public string ItemPastePostfix
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.ItemPastePostfix); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.ItemPastePostfix, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsFileSavePreCommandEnabled
        {
            get { return GetBoolValue(ProjectParameterNames.FileEvent.IsFileSavePreCommandEnabled); }
            set { SetBoolValue(ProjectParameterNames.FileEvent.IsFileSavePreCommandEnabled, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsFileSavePostCommandEnabled
        {
            get { return GetBoolValue(ProjectParameterNames.FileEvent.IsFileSavePostCommandEnabled); }
            set { SetBoolValue(ProjectParameterNames.FileEvent.IsFileSavePostCommandEnabled, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public string FileSavePreCommandPath
        {
            get { return GetTextValue(ProjectParameterNames.FileEvent.FileSavePreCommandPath); }
            set { SetTextValue(ProjectParameterNames.FileEvent.FileSavePreCommandPath, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public string FileSavePostCommandPath
        {
            get { return GetTextValue(ProjectParameterNames.FileEvent.FileSavePostCommandPath); }
            set { SetTextValue(ProjectParameterNames.FileEvent.FileSavePostCommandPath, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public bool EnabledNameDelimiter
        {
            get { return this.GetValue<bool>(ProjectParameterNames.ItemNaming.EnabledNameDelimiter); }
            set { this.SetValue<bool>(ProjectParameterNames.ItemNaming.EnabledNameDelimiter, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public string NameDelimiter
        {
            get { return GetTextValue(ProjectParameterNames.ItemNaming.NameDelimiter); }
            set { SetTextValue(ProjectParameterNames.ItemNaming.NameDelimiter, value); }
        }

        /// <summary>
        /// リストのコメントカラム名の設定を取得または設定します。
        /// </summary>
        public string CommentColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment, value); }
        }

        public string Comment1ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment1); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment1, value); }
        }

        public string Comment2ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment2); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment2, value); }
        }

        public string Comment3ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment3); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment3, value); }
        }

        public string Comment4ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment4); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment4, value); }
        }

        public string Comment5ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment5); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment5, value); }
        }

        public string Comment6ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment6); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment6, value); }
        }

        public string Comment7ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment7); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment7, value); }
        }

        public string Comment8ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment8); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment8, value); }
        }

        public string Comment9ColumnText
        {
            get { return GetTextValue(ProjectParameterNames.CommentColumnText.Comment9); }
            set { SetTextValue(ProjectParameterNames.CommentColumnText.Comment9, value); }
        }

        /// <summary>
        /// カラーコメントの設定を取得または設定します。
        /// </summary>
        public string ColorComment0
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment0); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment0, value); }
        }

        ///
        public string ColorComment1
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment1); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment1, value); }
        }

        ///
        public string ColorComment2
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment2); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment2, value); }
        }

        ///
        public string ColorComment3
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment3); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment3, value); }
        }

        ///
        public string ColorComment4
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment4); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment4, value); }
        }

        ///
        public string ColorComment5
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment5); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment5, value); }
        }

        ///
        public string ColorComment6
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment6); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment6, value); }
        }

        ///
        public string ColorComment7
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment7); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment7, value); }
        }

        ///
        public string ColorComment8
        {
            get { return GetTextValue(ProjectParameterNames.ColorComment.Comment8); }
            set { SetTextValue(ProjectParameterNames.ColorComment.Comment8, value); }
        }

        ///
        public string ProjectComment
        {
            get { return GetTextValue(ProjectParameterNames.Project.ProjectComment); }
            set { SetTextValue(ProjectParameterNames.Project.ProjectComment, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public UserCommand[] UserCommands
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        public int SyncPort
        {
            get { return GetIntValue(ProjectParameterNames.SndEditSetting.SyncPort); }
            set { SetIntValue(ProjectParameterNames.SndEditSetting.SyncPort, value); }
        }

        ///
        public int FuncPort
        {
            get { return GetIntValue(ProjectParameterNames.SndEditSetting.FuncPort); }
            set { SetIntValue(ProjectParameterNames.SndEditSetting.FuncPort, value); }
        }

        ///
        public int CtrlPort
        {
            get { return GetIntValue(ProjectParameterNames.SndEditSetting.CtrlPort); }
            set { SetIntValue(ProjectParameterNames.SndEditSetting.CtrlPort, value); }
        }

        /// <summary>
        ///
        /// </summary>
        public int SyncChannel
        {
            get { return GetIntValue(ProjectParameterNames.SndEditSetting.SyncChannel); }
            set { SetIntValue(ProjectParameterNames.SndEditSetting.SyncChannel, value); }
        }

        ///
        public int FuncChannel
        {
            get { return GetIntValue(ProjectParameterNames.SndEditSetting.FuncChannel); }
            set { SetIntValue(ProjectParameterNames.SndEditSetting.FuncChannel, value); }
        }

        ///
        public int CtrlChannel
        {
            get { return GetIntValue(ProjectParameterNames.SndEditSetting.CtrlChannel); }
            set { SetIntValue(ProjectParameterNames.SndEditSetting.CtrlChannel, value); }
        }

        // 接続、切断の設定の有効無効
        public bool EnabledConnectionTimeOut
        {
            get { return GetBoolValue(ProjectParameterNames.Connection.EnabledConnectionTimeOut); }
            set { SetBoolValue(ProjectParameterNames.Connection.EnabledConnectionTimeOut, value); }
        }

        // 接続中に反応がなくなった場合、切断するまでのタイムアウト（秒）です。
        public int ConnectionTimeOut
        {
            get { return GetIntValue(ProjectParameterNames.Connection.ConnectionTimeOut); }
            set { SetIntValue(ProjectParameterNames.Connection.ConnectionTimeOut, value); }
        }

        // <summary>
        // 「最大平均ラウドネス値を超える場合は警告する」の有効無効を設定または取得します。
        // </summary>
        public bool EnabledIntegratedLoudnessThreshold
        {
            get { return GetBoolValue(ProjectParameterNames.Statistics.EnabledIntegratedLoudnessThreshold); }
            set { SetBoolValue(ProjectParameterNames.Statistics.EnabledIntegratedLoudnessThreshold, value); }
        }

        // <summary>
        // 最大平均ラウドネス値
        // </summary>
        public float IntegratedLoudnessThreshold
        {
            get { return GetFloatValue(ProjectParameterNames.Statistics.IntegratedLoudnessThreshold); }
            set
            {
                // 有効桁数小数点以下一桁（二桁目は四捨五入）
                float threshold = (float)Math.Round(value, 1, MidpointRounding.AwayFromZero);
                SetFloatValue(ProjectParameterNames.Statistics.IntegratedLoudnessThreshold, threshold);
            }
        }

        // <summary>
        // シーケンスサウンドの最大計測時間（平均ラウドネス値の計測で必要）（単位：秒）
        // </summary>
        public int SequenceMaxMeasureDuration
        {
            get { return GetIntValue(ProjectParameterNames.Statistics.SequenceMaxMeasureDuration); }
            set { SetIntValue(ProjectParameterNames.Statistics.SequenceMaxMeasureDuration, value); }
        }

        //-----------------------------------------------------------------
        // その他のプロパティ
        //-----------------------------------------------------------------

        /// <summary>
        /// HTMLサウンドリスト設定を取得または設定します。
        /// </summary>
        public SoundListOutputSettings SoundListOutputSettings { get; set; }

        /// <summary>
        /// 読み込み時に SoundArchiveOutputTypesを作るために一時的に使われます。
        /// 読み込み完了後には使われません。
        /// </summary>
        public Dictionary<string, SoundArchiveOutputTypes> UnresolvedSoundArchiveOutputTypes
        {
            get
            {
                return this.unresolvedSoundArchiveOutputTypes;
            }
        }

        public Dictionary<SoundSet, SoundArchiveOutputTypes> SoundArchiveOutputTypes
        {
            get
            {
                return this.soundArchiveOutputTypes;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public UserParameterStructureSettings UserDataStructureSettings { get; set; }

        //-----------------------------------------------------------------
        // メソッドとイベントハンドラ
        //-----------------------------------------------------------------

        public string GetWavePreprocessExeFullPath(string basePath)
        {
            var path = Environment.ExpandEnvironmentVariables(this.WavePreprocessExePath);
            return string.IsNullOrEmpty(path)
                ? string.Empty
                : Path.IsPathRooted(path) ? path : PathUtility.GetFullPath(Path.Combine(basePath, path));
        }

        public void OnModifiedProjectSetting()
        {
            if (this.ModifiedProjectSetting != null)
            {
                ModifiedProjectSetting(this, new EventArgs());
            }
        }

        // <summary>
        // 全ての ProjectSoundSetComponent のファイルパスを取得します。
        // </summary>
        public IEnumerable<string> GetProjectSoundSetComponentFilePaths()
        {
            return this.GetProjectSoundSetComponentFilePaths(this);
        }

        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            // コンバート設定
            SetInstanceIntValue(ProjectParameterNames.Conversion.SmfTimebase, 96);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.DoUseBuiltInWavePreprocessExe, true);
            SetInstanceTextValue(ProjectParameterNames.Conversion.WavePreprocessExePath, string.Empty);
            SetInstanceTextValue(ProjectParameterNames.Conversion.IntermediateOutputDirectoryPath, "cache");
            SetInstanceTextValue(ProjectParameterNames.Conversion.InGameEditCacheOutputDirectoryPath, "editCache");
            SetInstanceBoolValue(ProjectParameterNames.Conversion.KeepIntermediateTextSequence, true);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.OutputLabel, false);
            SetInstanceTextValue(ProjectParameterNames.Conversion.OutputDirectoryPath, "output");
            SetInstanceTextValue(ProjectParameterNames.Conversion.ExternalFileDirectoryPath, "stream");
            SetInstanceTextValue(ProjectParameterNames.Conversion.UserManagementFileDirectoryPath, "userManagementFiles");

            SetInstanceBoolValue(ProjectParameterNames.Conversion.ExcludeStringTable, false);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.DoOutputPCBinariesForAAC, true);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.DoUseHardwareOpusDecoder, false);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.DoDeleteGarbageOutputBinaries, false);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.DoWarnUnreferencedItems, false);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.DoWarnDisableGroupItemTargets, false);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.DoWarnPCBinariesForAACNotFound, true);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.IsPreConvertCommandsEnabled, true);
            SetInstanceBoolValue(ProjectParameterNames.Conversion.IsPostConvertCommandsEnabled, true);

            this.Parameters.AddValue(
                ProjectParameterNames.Conversion.PreConvertCommands,
                new ObjectParameterValue<List<string>>(new List<string>()));

            this.Parameters.AddValue(
                ProjectParameterNames.Conversion.PostConvertCommands,
                new ObjectParameterValue<List<string>>(new List<string>()));

            SetInstanceBoolValue(ProjectParameterNames.FileEvent.IsFileSavePreCommandEnabled, false);
            SetInstanceBoolValue(ProjectParameterNames.FileEvent.IsFileSavePostCommandEnabled, false);
            SetInstanceTextValue(ProjectParameterNames.FileEvent.FileSavePreCommandPath, string.Empty);
            SetInstanceTextValue(ProjectParameterNames.FileEvent.FileSavePostCommandPath, string.Empty);

            // サウンドアーカイブ設定
            SetInstanceIntValue(ProjectParameterNames.SoundArchivePlayer.StreamSoundCount, 4, 0, 999);
            SetInstanceIntValue(ProjectParameterNames.SoundArchivePlayer.StreamChannelCount, 8, 0, 32);
            SetInstanceIntValue(ProjectParameterNames.SoundArchivePlayer.StreamBufferTimes, 1, 1, 4);
            SetInstanceIntValue(ProjectParameterNames.SoundArchivePlayer.WaveSoundCount, 64, 0, 999);
            SetInstanceIntValue(ProjectParameterNames.SoundArchivePlayer.SequenceSoundCount, 64, 0, 999);
            SetInstanceIntValue(ProjectParameterNames.SoundArchivePlayer.SequenceTrackCount, 64, 0, 999);

            // アイテム名の接頭辞
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.StreamSoundNamePrefix, "STRM_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.WaveSoundSetNamePrefix, "WSDSET_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.WaveSoundNamePrefix, "WSD_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.SequenceSoundSetNamePrefix, "SEQSET_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.SequenceSoundNamePrefix, "SEQ_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.BankNamePrefix, "BANK_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.PlayerNamePrefix, "PLAYER_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.WaveArchiveNamePrefix, "WARC_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.GroupNamePrefix, "GROUP_");
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.InstrumentNamePrefix, "INST_");

            // アイテムの命名方法
            SetInstanceBoolValue(ProjectParameterNames.ItemNaming.HasPrefix, true);
            SetInstanceParameterValue(ProjectParameterNames.ItemNaming.CaseChange,
                new CaseChangeTypeParameterValue());
            SetInstanceParameterValue(ProjectParameterNames.ItemNaming.InvalidCharChange,
                new InvalidCharChangeTypeParameterValue());

            SetInstanceTextValue(ProjectParameterNames.ItemNaming.ItemPastePostfix, "_copy");

            SetInstanceBoolValue(ProjectParameterNames.ItemNaming.EnabledNameDelimiter, true);
            SetInstanceTextValue(ProjectParameterNames.ItemNaming.NameDelimiter, ".");

            // コメントカラム名の設定
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment,
                                 MessageResource.ColumnText_Comment);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment1,
                                 MessageResource.ColumnText_Comment1);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment2,
                                 MessageResource.ColumnText_Comment2);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment3,
                                 MessageResource.ColumnText_Comment3);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment4,
                                 MessageResource.ColumnText_Comment4);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment5,
                                 MessageResource.ColumnText_Comment5);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment6,
                                 MessageResource.ColumnText_Comment6);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment7,
                                 MessageResource.ColumnText_Comment7);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment8,
                                 MessageResource.ColumnText_Comment8);
            SetInstanceTextValue(ProjectParameterNames.CommentColumnText.Comment9,
                                 MessageResource.ColumnText_Comment9);

            //
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment0, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment1, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment2, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment3, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment4, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment5, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment6, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment7, String.Empty);
            SetInstanceTextValue(ProjectParameterNames.ColorComment.Comment8, String.Empty);

            //
            UserCommands = new UserCommand[UserCommand.MaxCount];
            for (int index = 0; index < UserCommand.MaxCount; index++)
            {
                UserCommands[index] = new UserCommand();
            }

            //
            SetInstanceTextValue(ProjectParameterNames.Project.ProjectComment, String.Empty);

            // sndedit 設定
            SetInstanceIntValue(ProjectParameterNames.SndEditSetting.SyncPort, 54086, 0, 65535);
            SetInstanceIntValue(ProjectParameterNames.SndEditSetting.FuncPort, 54087, 0, 65535);
            SetInstanceIntValue(ProjectParameterNames.SndEditSetting.CtrlPort, 54088, 0, 65535);

            SetInstanceIntValue(ProjectParameterNames.SndEditSetting.SyncChannel, 0x0A, 0, 0x0F);
            SetInstanceIntValue(ProjectParameterNames.SndEditSetting.FuncChannel, 0x0B, 0, 0x0F);
            SetInstanceIntValue(ProjectParameterNames.SndEditSetting.CtrlChannel, 0x0C, 0, 0x0F);

            // 接続、切断の設定
            SetInstanceBoolValue(ProjectParameterNames.Connection.EnabledConnectionTimeOut, false);
            SetInstanceIntValue(ProjectParameterNames.Connection.ConnectionTimeOut, 5);

            // 統計
            SetInstanceBoolValue(ProjectParameterNames.Statistics.EnabledIntegratedLoudnessThreshold, false);
            SetInstanceFloatValue(ProjectParameterNames.Statistics.IntegratedLoudnessThreshold, -24.0F);
            SetInstanceIntValue(ProjectParameterNames.Statistics.SequenceMaxMeasureDuration, 180);
        }

        private void OnBroadcastUpdateComponentBinding(object sender, SoundDocumentBindingEventArgs e)
        {
            Debug.Assert(e.Components.Count() == 1, "Component must be one.");

            switch (e.Action)
            {
                case SoundDocumentChangedAction.Added: // サウンドセットの追加
                    {
                        if (e.Components[0] is SoundSet == false)
                        {
                            return;
                        }

                        Component parentComponent = null;
                        SoundSet targetComponent = e.Components[0] as SoundSet;

                        // 追加されたサウンドセットと同じファイルパスの ProjectSoundSetComponent を取得します。
                        var projectSoundSetComponent = this.FindProjectSoundSetComponent(e.Document.Resource.Key, this);
                        Debug.Assert(projectSoundSetComponent != null);
                        // 追加されたサウンドセットと ProjectSoundSetComponent を置き換える準備をします。
                        parentComponent = projectSoundSetComponent.Parent;
                        int insertIndex = parentComponent.Children.IndexOf(projectSoundSetComponent);
                        // ProjectSoundSetComponent をプロジェクトツリーから削除する。
                        parentComponent.Children.Remove(projectSoundSetComponent);
                        string name = projectSoundSetComponent.Name;

                        if (targetComponent.Name != name)
                        {
                            targetComponent.Name = name;
                        }
                        // サウンドセットを追加する（ProjectSoundSetComponent と置き換え）。
                        if (insertIndex >= 0 && insertIndex <= parentComponent.Children.Count)
                        {
                            parentComponent.Children.Insert(insertIndex, targetComponent);
                        }
                        else
                        {
                            parentComponent.Children.Add(targetComponent);
                        }
                        break;
                    }

                case SoundDocumentChangedAction.Removed: // サウンドセットの削除
                    {
                        if (e.Components[0] is SoundSet == false)
                        {
                            return;
                        }
                        SoundSet targetComponent = e.Components[0] as SoundSet;
                        var parentComponent = targetComponent.Parent;
                        parentComponent.Children.Remove(targetComponent);
                        break; ;
                    }
            }
        }

        private ProjectSoundSetComponent FindProjectSoundSetComponent(string soundSetFilePath, Component component)
        {
            if (component is ProjectSoundSetComponent == true)
            {
                var projectSoundSetComponent = component as ProjectSoundSetComponent;
                if (projectSoundSetComponent.TargetFilePath == soundSetFilePath)
                {
                    return projectSoundSetComponent;
                }
            }
            else if (component is FolderComponent == true ||
                     component is SoundProject == true)
            {
                foreach (var child in component.Children)
                {
                    var projectSoundSetComponent = this.FindProjectSoundSetComponent(soundSetFilePath, child);
                    if (projectSoundSetComponent != null)
                    {
                        return projectSoundSetComponent;
                    }
                }
            }

            return null;
        }

        private IEnumerable<string> GetProjectSoundSetComponentFilePaths(Component component)
        {
            if (component is ProjectSoundSetComponent == true)
            {
                return new string[] { (component as ProjectSoundSetComponent).TargetFilePath, };
            }
            else if (component is FolderComponent == true ||
                     component is SoundProject == true)
            {
                var soundSetPathList = new List<string>();

                foreach (var child in component.Children)
                {
                    var soundSetPaths = this.GetProjectSoundSetComponentFilePaths(child);
                    if (soundSetPaths != null)
                    {
                        soundSetPathList.AddRange(soundSetPaths);
                    }
                }

                return soundSetPathList;
            }

            return null;
        }
    }
}
