﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Specialized;
    using Core;
    using Core.Parameters;

    public class Sound : SoundSetItem
    {
        private string savedPlayer = String.Empty;
        private PlayerBase targetPlayer = null;

        public Sound()
        {
            this.IsListenQueryComponentDirty = true;
            this.IsListenQuerySoundProjectItemBinding = true;
            this.IsListenResetComponentDirty = true;
            this.IsListenUpdateSoundProjectItemBinding = true;
            this.IsListenUpdateRenamedSoundProjectItemBinding = true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool LinkedPlayer
        {
            get { return this.targetPlayer != null ? true : false; }
        }

        /// <summary>
        /// 参照するプレイヤーを取得します。
        /// </summary>
        public PlayerBase TargetPlayer
        {
            get { return this.targetPlayer; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override bool IsReference(Component component)
        {
            if (base.IsReference(component) == true)
            {
                return true;
            }

            if (this.targetPlayer == null) { return false; }
            return this.targetPlayer == component ? true : false;
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパス
        /// </summary>
        public string FilePath
        {
            get { return GetFilePathValue(ProjectParameterNames.FilePath); }
            set { SetFilePathValue(ProjectParameterNames.FilePath, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 音量
        /// </summary>
        public int Volume
        {
            get { return GetIntValue(ProjectParameterNames.Volume); }
            set { SetIntValue(ProjectParameterNames.Volume, value); }
        }

        ///--------------------------------
        /// <summary>
        /// ラウドネス値の計測状態を取得または設定します。
        /// </summary>
        public IntegratedLoudnessStatus IntegratedLoudnessStatus
        {
            get { return this.GetValue<IntegratedLoudnessStatus>(ProjectParameterNames.IntegratedLoudnessStatus); }
            set { this.SetValue<IntegratedLoudnessStatus>(ProjectParameterNames.IntegratedLoudnessStatus, value); }
        }

        ///--------------------------------
        /// <summary>
        /// ラウドネス値を取得または設定します。
        /// </summary>
        public float IntegratedLoudness
        {
            get { return this.GetFloatValue(ProjectParameterNames.IntegratedLoudness); }
            set { this.SetFloatValue(ProjectParameterNames.IntegratedLoudness, value); }
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤー優先度
        /// </summary>
        public int PlayerPriority
        {
            get { return GetIntValue(ProjectParameterNames.Sound.PlayerPriority); }
            set { SetIntValue(ProjectParameterNames.Sound.PlayerPriority, value); }
        }

        ///--------------------------------
        /// <summary>
        /// プレイヤー参照
        /// </summary>
        public string PlayerReference
        {
            get { return GetTextValue(ProjectParameterNames.Sound.PlayerReference); }
            set { SetTextValue(ProjectParameterNames.Sound.PlayerReference, value); }
        }

        ///--------------------------------
        /// <summary>
        /// アクタープレイヤー
        /// </summary>
        public int ActorPlayer
        {
            get { return GetIntValue(ProjectParameterNames.Sound.ActorPlayer); }
            set { SetIntValue(ProjectParameterNames.Sound.ActorPlayer, value); }
        }

        /// <summary>
        /// 3Dサウンド設定を取得または設定します。
        /// </summary>
        public Sound3D Sound3D
        {
            get { return this.GetValue<Sound3D>(ProjectParameterNames.Sound.Sound3DParams); }
            set { this.SetValue<Sound3D>(ProjectParameterNames.Sound.Sound3DParams, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 単一再生の種類
        /// </summary>
        public SinglePlayType SinglePlayType
        {
            get { return GetValue<SinglePlayType>(ProjectParameterNames.Sound.SinglePlayType); }
            set { SetValue<SinglePlayType>(ProjectParameterNames.Sound.SinglePlayType, value); }
        }

        ///--------------------------------
        /// <summary>
        /// 単一再生の有効時間
        /// </summary>
        public int SinglePlayEffectiveDuration
        {
            get { return GetIntValue(ProjectParameterNames.Sound.SinglePlayEffectiveDuration); }
            set { SetIntValue(ProjectParameterNames.Sound.SinglePlayEffectiveDuration, value); }
        }

        ///--------------------------------
        /// <summary>
        /// ユーザー設定
        /// </summary>
        public ulong UserParameter
        {
            get { return GetULongValue(ProjectParameterNames.Sound.UserParameter); }
            set { SetULongValue(ProjectParameterNames.Sound.UserParameter, value); }
        }

        public ulong UserParameter1
        {
            get { return GetULongValue(ProjectParameterNames.Sound.UserParameter1); }
            set { SetULongValue(ProjectParameterNames.Sound.UserParameter1, value); }
        }

        public ulong UserParameter2
        {
            get { return GetULongValue(ProjectParameterNames.Sound.UserParameter2); }
            set { SetULongValue(ProjectParameterNames.Sound.UserParameter2, value); }
        }

        public ulong UserParameter3
        {
            get { return GetULongValue(ProjectParameterNames.Sound.UserParameter3); }
            set { SetULongValue(ProjectParameterNames.Sound.UserParameter3, value); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool SuspendParameterValueChangedEventBubbling { get; set; }

        ///--------------------------------
        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceFilePathValue(ProjectParameterNames.FilePath, string.Empty);
            SetInstanceIntValue(ProjectParameterNames.Volume, 96, 0, 255);
            SetInstanceFloatValue(ProjectParameterNames.IntegratedLoudness, 0);
            SetInstanceIntegratedLoudnessStatusValue(ProjectParameterNames.IntegratedLoudnessStatus, IntegratedLoudnessStatus.Unmeasured);
            SetInstanceIntValue(ProjectParameterNames.Sound.PlayerPriority, 64, 0, 127);
            SetInstanceTextValue(ProjectParameterNames.Sound.PlayerReference, string.Empty);
            SetInstanceIntValue(ProjectParameterNames.Sound.ActorPlayer, 0, 0, 3);
            SetInstanceULongValue(ProjectParameterNames.Sound.UserParameter, 0, 0, uint.MaxValue);
            SetInstanceULongValue(ProjectParameterNames.Sound.UserParameter1, 0, 0, uint.MaxValue);
            SetInstanceULongValue(ProjectParameterNames.Sound.UserParameter2, 0, 0, uint.MaxValue);
            SetInstanceULongValue(ProjectParameterNames.Sound.UserParameter3, 0, 0, uint.MaxValue);
            Sound3DParameterValue sound3DParam = new Sound3DParameterValue(new Sound3D());
            Parameters.AddValue(ProjectParameterNames.Sound.Sound3DParams, sound3DParam);
            SetInstanceParameterValue(ProjectParameterNames.Sound.SinglePlayType, new SinglePlayTypeParameterValue(SinglePlayType.None));
            SetInstanceIntValue(ProjectParameterNames.Sound.SinglePlayEffectiveDuration, 0, 0, 65535);

            // サウンドアーカイブバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.Name, ParameterAttributes.ComputeSoundArchiveHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.PlayerReference, ParameterAttributes.ComputeSoundArchiveHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.UserParameter, ParameterAttributes.ComputeSoundArchiveHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.UserParameter1, ParameterAttributes.ComputeSoundArchiveHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.UserParameter2, ParameterAttributes.ComputeSoundArchiveHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.UserParameter3, ParameterAttributes.ComputeSoundArchiveHash);

            // sndedit 転送対象パラメータ
            this.SetParameterAttributes(ProjectParameterNames.Volume, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.PlayerPriority, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.ActorPlayer, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.Sound3DParams, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.ReleasePriorityFixed, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.SinglePlayType, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.SinglePlayEffectiveDuration, ParameterAttributes.ComputeSndeditTransferHash);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected PanMode GetPanModeValue(string name)
        {
            return (PanModeParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected PanCurve GetPanCurveValue(string name)
        {
            return (PanCurveParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected DecayCurve3D GetDecayCurve3DValue(string name)
        {
            return (DecayCurve3DParameterValue)Parameters.GetValue(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetPanModeValue(string name, PanMode value)
        {
            Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetPanCurveValue(string name, PanCurve value)
        {
            Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetDecayCurve3DValue(string name, DecayCurve3D value)
        {
            Parameters[name].Value = value;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstancePanModeValue(string name, PanMode value)
        {
            Parameters.AddValue(name, new PanModeParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstancePanCurveValue(string name, PanCurve value)
        {
            Parameters.AddValue(name, new PanCurveParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceDecayCurve3DValue(string name, DecayCurve3D value)
        {
            Parameters.AddValue(name, new DecayCurve3DParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceIntegratedLoudnessStatusValue(string name, IntegratedLoudnessStatus value)
        {
            SetInstanceParameterValue(name, new IntegratedLoudnessStatusParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnParameterValueChanged(ParameterEventArgs e)
        {
            if (SuspendParameterValueChangedEventBubbling)
            {
                return;
            }

            base.OnParameterValueChanged(e);

            if (e.Key == ProjectParameterNames.Sound.PlayerReference)
            {
                if (this.targetPlayer != null && this.targetPlayer.Name != PlayerReference)
                {
                    this.targetPlayer = null;
                }

                Router.BubbleEvent(new RoutingEventArgs
                                    (new RequestUpdateSoundProjectItemBindingEvent(), this));
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnChildrenChanged(NotifyCollectionChangedEventArgs e)
        {
            base.OnChildrenChanged(e);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnUpdateSoundProjectItemBinding(ComponentManagerEventArgs e)
        {
            string[] playerNames = (null == this.targetPlayer) ?
                new string[] { PlayerReference } :
                new string[] { PlayerReference, this.targetPlayer.Name };

            this.targetPlayer = FindComponent<PlayerBase>(e.ComponentManager, playerNames);

            if (this.targetPlayer != null)
            {
                SuspendParameterValueChangedEventBubbling = true;
                PlayerReference = this.targetPlayer.Name;
                SuspendParameterValueChangedEventBubbling = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnUpdateRenamedSoundProjectItemBinding(ComponentManagerRenamedComponentEventArgs e)
        {
            string targetName = PlayerReference;
            if (targetName == e.NewName)
            {
                targetName = e.OldName;
            }

            this.targetPlayer = targetName != String.Empty ? FindComponent<PlayerBase>(e.ComponentManager, targetName) : null;

            if (this.targetPlayer != null)
            {
                SuspendParameterValueChangedEventBubbling = true;
                PlayerReference = this.targetPlayer.Name;
                SuspendParameterValueChangedEventBubbling = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnQuerySoundProjectItemBinding(QuerySoundProjectItemBindingEventArgs e)
        {
            if (e.BindingComponents.Contains(this)) { return; }

            if (e.Component == this.targetPlayer)
            {
                e.BindingComponents.Add(this);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnQueryComponentDirty(QueryComponentDirtyEventArgs e)
        {
            if (this.savedPlayer != PlayerReference)
            {
                e.Result = true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnResetComponentDirty(RoutingEventArgs e)
        {
            this.savedPlayer = PlayerReference;
        }
    }
}
