﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// バンクに含まれるコンポーネントを管理します。
    /// </summary>
    public class BankService : ComponentService
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="documentService">ドキュメントサービス。</param>
        /// <param name="intermediateOutputTraits">サウンド中間出力の特性を指定します。</param>
        public BankService(
            DocumentService documentService,
            SoundIntermediateOutputTraits intermediateOutputTraits)
            : base(documentService, new BankComponentManager())
        {
            Ensure.Argument.NotNull(intermediateOutputTraits);
            this.IntermediateOutputTraits = intermediateOutputTraits;
        }

        /// <summary>
        /// バンクファイルが再読み込みされると発生します。
        /// </summary>
        public event EventHandler Reloaded;

        /// <summary>
        /// 開かれているバンクドキュメントを取得します。
        /// </summary>
        public BankDocument BankDocument
        {
            get { return ComponentManager.BankDocument; }
        }

        /// <summary>
        /// バンクを取得します。
        /// </summary>
        public Bank Bank
        {
            get { return ComponentManager.Bank; }
        }

        /// <summary>
        /// バンクに含まれるインストルメントの一覧を取得します。
        /// </summary>
        public Instrument[] Instruments
        {
            get { return ComponentManager.Instruments.ToArray(); }
        }

        /// <summary>
        /// バンクに含まれる VelocityRegion の一覧を取得します。
        /// </summary>
        public VelocityRegion[] VelocityRegions
        {
            get { return ComponentManager.VelocityRegions.ToArray(); }
        }

        /// <summary>
        /// コンポーネントマネージャを取得します。
        /// </summary>
        protected new BankComponentManager ComponentManager
        {
            get { return base.ComponentManager as BankComponentManager; }
        }

        /// <summary>
        /// サウンド中間出力の特性を取得します。
        /// </summary>
        private SoundIntermediateOutputTraits IntermediateOutputTraits { get; set; }

        /// <summary>
        /// 新しいバンクを作成します。
        /// </summary>
        /// <param name="filePath">バンクファイルパス。</param>
        public void Create(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            Create(new FileResource(filePath));
        }

        /// <summary>
        /// バンクファイルを開きます。
        /// </summary>
        /// <param name="filePath">バンクファイルパス。</param>
        public void Open(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }
            Open(new FileResource(filePath));
        }

        /// <summary>
        /// バンクファイルを再読み込みします。
        /// </summary>
        public void Reload()
        {
            if (null == BankDocument) { return; }

            if ((BankDocument.Resource as FileResource).LastWriteTime ==
                             File.GetLastWriteTime(BankDocument.Resource.Key)) { return; }

            // リロードできるか GetDocumentReader で試す。
            IDocumentReader reader = DocumentService.Traits.GetDocumentReader(BankDocument.Resource);
            if (reader == null)
            {
                throw new FileNotSupportedException(BankDocument.Resource.Key);
            }

            using (DocumentReference documentRef =
                        DocumentService.OpenDocument(new FileResource(BankDocument.Resource.Key)))
            {
                try
                {
                    this.BeginDelayComponentsRemoveEvent();
                    ComponentManager.RemoveDocument(BankDocument.Resource.Key);
                    this.EndDelayComponentsRemoveEvent();
                }
                catch
                {
                    this.CancelDelayComponentsRemoveEvent();
                    throw;
                }

                DocumentService.ReloadDocument(documentRef.Document);

                try
                {
                    this.BeginDelayComponentsAddEvent();
                    ComponentManager.AddDocument(documentRef);
                    this.EndDelayComponentsAddEvent();
                }
                catch
                {
                    this.CancelDelayComponentsAddEvent();
                    throw;
                }
            }

            OnReloaded(EventArgs.Empty);
        }

        /// <summary>
        /// バンクファイルが再読み込みされると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnReloaded(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != Reloaded)
            {
                Reloaded(this, e);
            }
        }

        /// <summary>
        /// バンクリソースを作成します。
        /// </summary>
        /// <param name="filePath">バンクファイルリソース。</param>
        protected override void CreateInternal(IStreamResource resource)
        {
            if (!(resource is FileResource)) { throw new ArgumentException("resource must be FileResource."); }
            if (null == resource.Key) { throw new ArgumentException("resource.Key must not be null."); }

            if (null != BankDocument)
            {
                throw new Exception("bank is already opened.");
            }

            using (DocumentReference documentRef = DocumentService.CreateDocument(
                this.IntermediateOutputTraits.BankDocumentTypeName))
            {

                if (null == documentRef ||
                    !(documentRef.Document is BankDocument))
                {
                    throw new ArgumentException("documentTypeName is invalid.");
                }

                documentRef.Document.Resource = resource;

                var bank = new Bank();
                (documentRef.Document as BankDocument).Bank = bank;

                DocumentService.SaveDocument(documentRef.Document);
                ComponentManager.AddDocument(documentRef);
            }
        }

        /// <summary>
        /// バンクリソースを開きます。
        /// </summary>
        /// <param name="filePath">バンクファイルリソース。</param>
        protected override bool OpenInternal(IStreamResource resource, bool readOnly = false)
        {
            if (!(resource is FileResource)) { throw new ArgumentException("resource must be FileResource."); }
            if (null == resource.Key) { throw new ArgumentException("resource.Key must not be null."); }

            if (null != BankDocument)
            {
                throw new Exception("bank is already opened.");
            }

            using (DocumentReference documentRef =
                        DocumentService.OpenDocument(new FileResource(resource.Key)))
            {

                if (null == documentRef ||
                    documentRef.Document is BankDocument == false ||
                    documentRef.Document.TypeName != this.IntermediateOutputTraits.BankDocumentTypeName)
                {
                    throw new FileNotSupportedException(resource.Key);
                }

                ComponentManager.AddDocument(documentRef);
            }

            return true;
        }

        /// <summary>
        /// バンクリソースに保存します。
        /// </summary>
        protected override void SaveInternal()
        {
            if (null == BankDocument)
            {
                throw new Exception("souind project is not opened.");
            }
            if (!BankDocument.IsDirty) { return; }

            DocumentService.SaveDocument(BankDocument);
        }

        /// <summary>
        /// バンクリソースを閉じます。
        /// </summary>
        protected override void CloseInternal()
        {
            ComponentManager.ClearDocuments();
        }
    }
}
