﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using NintendoWare.SoundFoundation.Core;
using NintendoWare.SoundFoundation.Documents;

namespace NintendoWare.SoundFoundation.Projects
{
    /// <summary>
    /// ドキュメントからサウンドプロジェクトツリーを構築します。
    /// </summary>
    public class SoundProjectComponentManager : ComponentManager
    {
        /// <summary>
        /// 開かれているプロジェクトドキュメントへの参照。
        /// </summary>
        private DocumentReference _projectDocumentReference = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SendResetComponentDirtyEvent()
        {
            SendBroadcastEvent(new RoutingEventArgs
                                (new ResetComponentDirtyEvent()));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void SendUpdateSoundProjectItemBindingEvent()
        {
            SendBroadcastEvent(new ComponentManagerEventArgs
                                (new UpdateSoundProjectItemBindingEvent(), this));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Component[] QueryComponentBindings(Component component)
        {
            if (null == component) { throw new ArgumentNullException("component"); }

            QuerySoundProjectItemBindingEventArgs e = new QuerySoundProjectItemBindingEventArgs(component);
            SendBroadcastEvent(e);

            return e.BindingComponents.ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// 開かれているプロジェクトを取得します。
        /// </summary>
        public SoundProjectDocument ProjectDocument
        {
            get
            {
                if (null == _projectDocumentReference) { return null; }
                return _projectDocumentReference.Document as SoundProjectDocument;
            }
        }

        /// <summary>
        /// 開かれているプロジェクトを取得します。
        /// </summary>
        public SoundProject Project
        {
            get
            {
                if (null == ProjectDocument) { return null; }
                return ProjectDocument.Project;
            }
        }

        /// <summary>
        /// 開かれているプロジェクトを取得します。
        /// </summary>
        public IEnumerable<SoundSetDocument> SoundSetDocuments
        {
            get
            {
                return from document in Documents.Values
                       where document is SoundSetDocument
                       select document as SoundSetDocument;
            }
        }

        /// <summary>
        /// プロジェクトに含まれるサウンドセットを列挙します。
        /// </summary>
        public IEnumerable<SoundSet> SoundSets
        {
            get
            {
                return FindSoundSet(Project);
            }
        }

        private IEnumerable<SoundSet> FindSoundSet(Component component)
        {
            if (component is SoundSet)
            {
                yield return component as SoundSet;
                yield break;
            }

            foreach (var childComponent in component.Children)
            {
                foreach (var resultComponent in FindSoundSet(childComponent))
                {
                    yield return resultComponent as SoundSet;
                }
            }
        }

        /// <summary>
        /// プロジェクトに含まれるサウンドセットアイテムを列挙します。
        /// </summary>
        public IEnumerable<SoundSetItem> SoundSetItems
        {
            get
            {
                foreach (SoundSet soundSet in SoundSets)
                {
                    foreach (SoundSetItem pack in soundSet.Children)
                    {
                        if (pack is WaveSoundSetPack || pack is SequenceSoundSetPack)
                        {
                            foreach (SoundSetItem itemSet in pack.Children)
                            {
                                yield return itemSet;

                                foreach (SoundSetItem item in itemSet.Children)
                                {
                                    yield return item;
                                }
                            }
                        }
                        else
                        {
                            foreach (SoundSetItem item in pack.Children)
                            {
                                yield return item;
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// プロジェクトに含まれるサウンドを列挙します。
        /// </summary>
        public IEnumerable<Sound> Sounds
        {
            get
            {
                foreach (SoundSet soundSet in SoundSets)
                {

                    foreach (Component pack in soundSet.Children)
                    {

                        if (pack is StreamSoundPack ||
                             pack is SequenceSoundPack)
                        {

                            foreach (Sound sound in pack.Children)
                            {
                                yield return sound;
                            }

                        }

                        if (pack is WaveSoundSetPack ||
                             pack is SequenceSoundSetPack)
                        {

                            foreach (Component sounds in pack.Children)
                            {

                                foreach (Sound sound in sounds.Children)
                                {
                                    yield return sound;
                                }

                            }
                        }

                    }

                }
            }
        }

        /// <summary>
        /// プロジェクトに含まれる StreamSoundBase を列挙します。
        /// </summary>
        public IEnumerable<StreamSoundBase> StreamSounds
        {
            get
            {
                return from SoundSet soundSet in SoundSets
                       from Component pack in soundSet.Children
                       where pack is StreamSoundPack
                       from StreamSoundBase sound in pack.Children
                       select sound;
            }
        }

        /// <summary>
        /// プロジェクトに含まれる WaveSoundSetBase を列挙します。
        /// </summary>
        public IEnumerable<WaveSoundSetBase> WaveSoundSets
        {
            get
            {
                return from SoundSet soundSet in SoundSets
                       from Component pack in soundSet.Children
                       where pack is WaveSoundSetPack
                       from WaveSoundSetBase waveSoundSet in pack.Children
                       select waveSoundSet;
            }
        }

        /// <summary>
        /// プロジェクトに含まれる WaveSoundBase を列挙します。
        /// </summary>
        public IEnumerable<WaveSoundBase> WaveSounds
        {
            get
            {
                return from SoundSet soundSet in SoundSets
                       from Component pack in soundSet.Children
                       where pack is WaveSoundSetPack
                       from WaveSoundSetBase sounds in pack.Children
                       from WaveSoundBase sound in sounds.Children
                       select sound;
            }
        }

        /// <summary>
        /// プロジェクトに含まれる SequenceSoundSetBase を列挙します。
        /// </summary>
        public IEnumerable<SequenceSoundSetBase> SequenceSoundSets
        {
            get
            {
                return from SoundSet soundSet in SoundSets
                       from Component pack in soundSet.Children
                       where pack is SequenceSoundSetPack
                       from SequenceSoundSetBase sequenceSoundSet in pack.Children
                       select sequenceSoundSet;
            }
        }

        /// <summary>
        /// プロジェクトに含まれる SequenceSoundBase を列挙します。
        /// </summary>
        public IEnumerable<SequenceSoundBase> SequenceSounds
        {
            get
            {
                foreach (SoundSet soundSet in SoundSets)
                {
                    foreach (Component pack in soundSet.Children)
                    {
                        if (pack is SequenceSoundSetPack)
                        {
                            foreach (SequenceSoundSetBase sequenceSoundSet in pack.Children)
                            {
                                foreach (SequenceSoundBase sound in sequenceSoundSet.Children)
                                {
                                    yield return sound;
                                }
                            }
                        }
                        else if (pack is SequenceSoundPack)
                        {
                            foreach (SequenceSoundBase sound in pack.Children)
                            {
                                yield return sound;
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// プロジェクトに含まれるバンクを列挙します。
        /// </summary>
        public IEnumerable<SoundSetBankBase> SoundSetBanks
        {
            get
            {
                return from SoundSet soundSet in SoundSets
                       from Component pack in soundSet.Children
                       where pack is SoundSetBankPack
                       from SoundSetBankBase bank in pack.Children
                       select bank;
            }
        }

        /// <summary>
        /// プロジェクトに含まれる WaveArchiveBase を列挙します。
        /// </summary>
        public IEnumerable<WaveArchiveBase> WaveArchives
        {
            get
            {
                return from SoundSet soundSet in SoundSets
                       from Component pack in soundSet.Children
                       where pack.Name == "@WaveArchives"               // ★仮の条件
                       from WaveArchiveBase waveArchive in pack.Children
                       select waveArchive;
            }
        }

        /// <summary>
        /// プロジェクトに含まれるプレイヤーを列挙します。
        /// </summary>
        public IEnumerable<PlayerBase> Players
        {
            get
            {
                return from SoundSet soundSet in SoundSets
                       from Component pack in soundSet.Children
                       where pack is PlayerPack
                       from PlayerBase player in pack.Children
                       select player;
            }
        }

        /// <summary>
        /// プロジェクトに含まれるグループを列挙します。
        /// </summary>
        public IEnumerable<GroupBase> Groups
        {
            get
            {
                return from SoundSet soundSet in SoundSets
                       from Component pack in soundSet.Children
                       where pack is GroupPack
                       from GroupBase itemGroup in pack.Children
                       select itemGroup;
            }
        }

        /// <summary>
        /// ProjectDocument が変更されると発生します。
        /// </summary>
        public event EventHandler ProjectDocumentChanged;

        /// <summary>
        /// ProjectDocument が変更されると発生します。
        /// </summary>
        public event EventHandler SoundSetDocumentsChanged;

        /// <summary>
        /// 指定コンポーネントがルートコンポーネントかどうか調べます。
        /// </summary>
        /// <param name="component">対象コンポーネント。</param>
        /// <returns>ルートコンポーネントの場合は true、それ以外の場合は false。</returns>
        protected override bool IsComponentTreeRoot(Component component)
        {
            return (component is SoundProject);
        }

        /// <summary>
        /// ドキュメント参照が追加されると発生します。
        /// </summary>
        /// <param name="e">DocumentReference イベントデータ。</param>
        protected override void OnAddDocumentReference(DocumentReferenceEventArgs e)
        {
            base.OnAddDocumentReference(e);

            if (null == e.DocumentReference) { return; }

            if (e.DocumentReference.Document is SoundSetDocument)
            {
                OnSoundSetDocumentsChanged(new EventArgs());
                return;
            }

            if (e.DocumentReference.Document is SoundProjectDocument)
            {

                if (null != _projectDocumentReference)
                {
                    throw new Exception("project document is already existing.");  // ★仮の例外クラス
                }

                _projectDocumentReference = e.DocumentReference;

                if (null != Project)
                {
                    OnProjectDocumentChanged(new EventArgs());
                }

            }
        }

        /// <summary>
        /// ドキュメント参照が削除されると発生します。
        /// </summary>
        /// <param name="e">DocumentReference イベントデータ。</param>
        protected override void OnRemoveDocumentReference(DocumentReferenceEventArgs e)
        {
            base.OnRemoveDocumentReference(e);

            if (null == e.DocumentReference) { return; }

            if (e.DocumentReference.Document is SoundSetDocument)
            {
                OnSoundSetDocumentsChanged(new EventArgs());
                return;
            }

            if (e.DocumentReference == _projectDocumentReference)
            {
                _projectDocumentReference = null;
                OnProjectDocumentChanged(new EventArgs());
            }
        }

        /// <summary>
        /// プロジェクトドキュメントが変更されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnProjectDocumentChanged(EventArgs e)
        {
            if (null != ProjectDocumentChanged)
            {
                ProjectDocumentChanged(this, e);
            }
        }

        /// <summary>
        /// サウンドセットドキュメント一覧が変更されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected virtual void OnSoundSetDocumentsChanged(EventArgs e)
        {
            if (null != SoundSetDocumentsChanged)
            {
                SoundSetDocumentsChanged(this, e);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SendBroadcastEvent(RoutingEventArgs eventArgs)
        {
            IEventRoutable routable = null;

            if ((routable = RootComponent as IEventRoutable) != null)
            {
                routable.Router.DispatchEvent(eventArgs);
            }
        }
    }
}
