﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.Wave
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.IO;
    using System.Text;

    public class WaveFileWav : WaveFile
    {
        private enum FormatTag : short
        {
            Unknown = 0,
            Pcm = 1,
            Adpcm = 2,
            IeeeFloat = 3,
            BigEndianPcm = -510, // =0xFE02 : SoundForge10 で BigEndian の .wav を出力した場合の値
        }

        private WavFormatChunk formatChunk = new WavFormatChunk();
        private WavSamplerChunk samplerChunk = new WavSamplerChunk();
        private WavCueChunk cueChunk = new WavCueChunk();
        private WavListChunk listChunk = new WavListChunk();
        private long numSampleFrames;
        private List<IRegionInfo> regions = new List<IRegionInfo>();
        private List<IMarkerInfo> markers = new List<IMarkerInfo>();

        public WavFormatChunk FormatChunk
        {
            get { return formatChunk; }
            set { formatChunk = value; }
        }

        public WavSamplerChunk SamplerChunk
        {
            get { return samplerChunk; }
            set { samplerChunk = value; }
        }

        public WavCueChunk CueChunk
        {
            get { return cueChunk; }
            set { cueChunk = value; }
        }

        public WavListChunk ListChunk
        {
            get { return listChunk; }
            set { listChunk = value; }
        }

        public class WavFormatChunk
        {
            private FormatTag wFormatTag = FormatTag.Pcm;
            private ushort wChannels = 1;
            private ulong dwSamplesPerSec = 32000;
            private ulong dwAvgBytesPerSec = 64000;
            private ushort wBlockAlign = 2;
            private ushort wBitsPerSample = 16;

            public ushort ChannelCount
            {
                get { return wChannels; }
                set
                {
                    wChannels = value;
                    UpdateField();
                }
            }
            public ushort SampleBit
            {
                get { return wBitsPerSample; }
                set
                {
                    wBitsPerSample = value;
                    UpdateField();
                }
            }
            public ulong SampleRate
            {
                get { return dwSamplesPerSec; }
                set
                {
                    dwSamplesPerSec = value;
                    UpdateField();
                }
            }
            public void Read(BinaryReader reader)
            {
                wFormatTag = (FormatTag)reader.ReadInt16();
                wChannels = reader.ReadUInt16();
                dwSamplesPerSec = reader.ReadUInt32();
                dwAvgBytesPerSec = reader.ReadUInt32();
                wBlockAlign = reader.ReadUInt16();
                wBitsPerSample = reader.ReadUInt16();
            }

            public bool IsBigEndianPcm()
            {
                return (wFormatTag == FormatTag.BigEndianPcm);
            }

            private void UpdateField()
            {
                wBlockAlign = (ushort)(wChannels * ((wBitsPerSample + 7) % 8));
                dwAvgBytesPerSec = wBlockAlign * dwSamplesPerSec;
            }
        }

        public class SampleLoop
        {
            private Int32 dwIdentifier;
            private Int32 dwType;
            private Int32 dwStart;
            private Int32 dwEnd;
            private Int32 dwFraction;
            private Int32 dwPlayCount;

            public Int32 StartFrame
            {
                get { return dwStart; }
                set { dwStart = value; }
            }
            public Int32 EndFrame
            {
                get { return dwEnd; }
                set { dwEnd = value; }
            }

            public void Read(BinaryReader reader)
            {
                dwIdentifier = reader.ReadInt32();
                dwType = reader.ReadInt32();
                dwStart = reader.ReadInt32();
                dwEnd = reader.ReadInt32();
                dwFraction = reader.ReadInt32();
                dwPlayCount = reader.ReadInt32();
            }
        }

        public class WavSamplerChunk
        {
            private Int32 dwManufacturer = 0;
            private Int32 dwProduct = 0;
            private Int32 dwSamplePeriod = 0;
            private Int32 dwMIDIUnityNote = 60;
            private Int32 dwMIDIPitchFraction = 0;
            private Int32 dwSMPTEFormat = 0;
            private Int32 dwSMPTEOffset = 0;
            //private Int32 cSampleLoops = 0;
            private Int32 cbSamplerData = 0;

            private SampleLoop[] loops = new SampleLoop[0];

            public bool IsLoop
            {
                get { return loops.Length > 0; }
            }
            public long LoopStartFrame
            {
                get
                {
                    if (loops.Length == 0) return -1;
                    return loops[0].StartFrame;
                }
                set
                {
                    if (loops.Length == 0)
                    {
                        loops = new[] { new SampleLoop() };
                    }
                    loops[0].StartFrame = (int)value;
                }
            }
            public long LoopEndFrame
            {
                get
                {
                    if (loops.Length == 0) return 0;
                    return loops[0].EndFrame;
                }
                set
                {
                    if (loops.Length == 0)
                    {
                        loops = new[] { new SampleLoop() };
                    }
                    loops[0].EndFrame = (int)value;
                }
            }
            public int OriginalKey
            {
                get
                {
                    return (int)dwMIDIUnityNote;
                }
                set
                {
                    dwMIDIUnityNote = (Int32)value;
                }
            }

            public void Read(BinaryReader reader)
            {
                dwManufacturer = reader.ReadInt32();
                dwProduct = reader.ReadInt32();
                dwSamplePeriod = reader.ReadInt32();
                dwMIDIUnityNote = reader.ReadInt32();
                dwMIDIPitchFraction = reader.ReadInt32();
                dwSMPTEFormat = reader.ReadInt32();
                dwSMPTEOffset = reader.ReadInt32();
                Int32 cSampleLoops = reader.ReadInt32();
                cbSamplerData = reader.ReadInt32();

                loops = new SampleLoop[cSampleLoops];
                for (int i = 0; i < cSampleLoops; i++)
                {
                    SampleLoop loop = new SampleLoop();
                    loop.Read(reader);
                    loops[i] = loop;
                }
            }
        }

        public class WavCueChunk
        {
            private WavCuePointList cuePoints = new WavCuePointList();

            public WavCuePointList CuePoints
            {
                get { return this.cuePoints; }
            }

            public void Read(BinaryReader reader, uint chunkSize)
            {
                if (chunkSize < 4)
                {
                    reader.ReadBytes((int)chunkSize);
                    return;
                }

                int cuePointCount = (int)reader.ReadUInt32();
                int readLength = 4;

                if (cuePointCount != (chunkSize - 4) / WavCuePoint.Size)
                {
                    throw new WaveFileReaderException("Invalid format");
                }

                for (int cueIndex = 0; cueIndex < cuePointCount; ++cueIndex)
                {
                    var newCuePoint = new WavCuePoint();
                    newCuePoint.Read(reader);

                    this.cuePoints.Add(newCuePoint);

                    readLength += WavCuePoint.Size;
                }
            }
        }

        public class WavCuePoint
        {
            public const int Size = 4 + 4 + 4 + 4 + 4 + 4;

            public UInt32 ID { get; set; }
            public UInt32 Position { get; set; }
            public char[] DataChunkID { get; set; }
            public UInt32 ChunkStart { get; set; }
            public UInt32 BlockStart { get; set; }
            public UInt32 SampleOffset { get; set; }

            public void Read(BinaryReader reader)
            {
                this.ID = reader.ReadUInt32();
                this.Position = reader.ReadUInt32();
                this.DataChunkID = reader.ReadChars(4);
                this.ChunkStart = reader.ReadUInt32();
                this.BlockStart = reader.ReadUInt32();
                this.SampleOffset = reader.ReadUInt32();
            }
        }

        public class WavCuePointList : KeyedCollection<uint, WavCuePoint>
        {
            protected override uint GetKeyForItem(WavCuePoint item)
            {
                return item.ID;
            }
        }

        public class WavListChunk
        {
            private List<WavLabelChunk> labelChunks = new List<WavLabelChunk>();
            private List<WavLabeledTextChunk> labeledTextChunks = new List<WavLabeledTextChunk>();

            public List<WavLabelChunk> LabelChunks
            {
                get { return this.labelChunks; }
            }

            public List<WavLabeledTextChunk> LabeledTextChunks
            {
                get { return this.labeledTextChunks; }
            }

            public void Read(BinaryReader reader, uint chunkSize)
            {
                var typeID = new string(reader.ReadChars(4));

                switch (typeID)
                {
                    case "adtl":
                        break;

                    case "INFO":
                        reader.ReadBytes((int)chunkSize - 4);
                        return;

                    default:
                        throw new WaveFileReaderException("Invalid format");
                }

                uint readSize = 4;

                while (readSize < chunkSize)
                {
                    string subChunkType = new string(reader.ReadChars(4));
                    UInt32 subChunkSize = reader.ReadUInt32();

                    readSize += 8;

                    switch (subChunkType)
                    {
                        case "labl":
                            {
                                var newChunk = new WavLabelChunk();
                                newChunk.Read(reader, subChunkSize);

                                this.labelChunks.Add(newChunk);
                            }
                            break;

                        case "ltxt":
                            {
                                var newChunk = new WavLabeledTextChunk();
                                newChunk.Read(reader, subChunkSize);

                                this.labeledTextChunks.Add(newChunk);
                            }
                            break;

                        default:
                            reader.ReadBytes((int)subChunkSize);
                            break;
                    }

                    // 2バイトアライン
                    uint alignment = 1 - ((subChunkSize + 0x01) & 0x01);
                    reader.ReadBytes((int)alignment);

                    readSize += subChunkSize + alignment;
                }
            }
        }

        public class WavLabelChunk
        {
            public WavLabelChunk()
            {
                this.Text = string.Empty;
            }

            public int CuePointID { get; set; }
            public string Text { get; set; }

            public void Read(BinaryReader reader, uint chunkSize)
            {
                this.CuePointID = (int)reader.ReadUInt32();

                int restSize = (int)chunkSize - 4;

                if (restSize > 0)
                {
                    this.Text = new string(reader.ReadChars(restSize)).TrimEnd('\0');
                }
            }
        }

        public class WavLabeledTextChunk
        {
            public uint CuePointID { get; set; }
            public uint SampleLength { get; set; }
            public char[] PurposeID { get; set; }

            public void Read(BinaryReader reader, uint chunkSize)
            {
                if (chunkSize < 12)
                {
                    reader.ReadBytes((int)chunkSize);
                    return;
                }

                this.CuePointID = reader.ReadUInt32();
                this.SampleLength = reader.ReadUInt32();
                this.PurposeID = reader.ReadChars(4);

                reader.ReadBytes((int)chunkSize - 12);
            }
        }

        public class RegionInfo : IRegionInfo
        {
            private string name = string.Empty;

            public string Name
            {
                get { return this.name; }
                set { this.name = value ?? string.Empty; }
            }

            public uint StartFrame { get; set; }
            public uint EndFrame { get; set; }
        }

        public class MarkerInfo : IMarkerInfo
        {
            private string name = string.Empty;

            public string Name
            {
                get { return this.name; }
                set { this.name = value ?? string.Empty; }
            }

            public uint Position { get; set; }
        }

        public override bool IsLoop
        {
            get { return samplerChunk.IsLoop; }
        }
        public override long LoopStartFrame
        {
            get { return samplerChunk.LoopStartFrame; }
            set { samplerChunk.LoopStartFrame = value; }
        }
        public override long LoopEndFrame
        {
            get
            {
                if (samplerChunk.LoopEndFrame < 0)
                {
                    return 0;
                }
                else
                {
                    return samplerChunk.LoopEndFrame + 1;
                }
            }
            set
            {
                if (value > 0)
                {
                    samplerChunk.LoopEndFrame = value - 1;
                }
                else
                {
                    samplerChunk.LoopEndFrame = 0;
                }
            }
        }

        public override int ChannelCount
        {
            get { return formatChunk.ChannelCount; }
            set { formatChunk.ChannelCount = (ushort)value; }
        }
        public override long FrameCount
        {
            get { return numSampleFrames; }
            set { numSampleFrames = value; }
        }
        public override int SampleBit
        {
            get
            {
                return formatChunk.SampleBit;
            }
            set
            {
                formatChunk.SampleBit = (ushort)value;
            }
        }
        public override int SampleRate
        {
            get
            {
                return (int)formatChunk.SampleRate;
            }
            set
            {
                formatChunk.SampleRate = (ushort)value;
            }
        }
        public override int OriginalKey
        {
            get
            {
                return samplerChunk.OriginalKey;
            }
            set
            {
                samplerChunk.OriginalKey = value;
            }
        }

        public override double WaveTime { get; set; }

        public override IList<IRegionInfo> Regions
        {
            get { return this.regions; }
        }

        public override IList<IMarkerInfo> Markers
        {
            get { return this.markers; }
        }
    }

}
