﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.Wave
{
    using System;
    using SoundFoundation.Utilities;

    public class WaveFileUtility
    {
        /// <summary>
        /// 再生時間(Time)のフォーマットされたループ有無テキストを取得します。
        /// </summary>
        public static string GetWaveTimeStringWithLoop(double waveTime, bool isLoop)
        {
            string times = GetWaveTimeString(waveTime);
            if (isLoop == true)
            {
                times = "L " + times;
            }

            return times;
        }

        /// <summary>
        /// 再生時間(Time)のフォーマットされたループ有無テキストを取得します。
        /// </summary>
        public static string GetWaveTimeStringWithLoop(WaveFile waveFile)
        {
            if (waveFile == null)
            {
                return NotAvailable.Text;
            }

            return GetWaveTimeStringWithLoop(waveFile.WaveTime, waveFile.IsLoop);
        }

        /// <summary>
        /// 再生時間(Time)のフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetWaveTimeString(double waveTime)
        {
            waveTime += 5; // 秒の下３桁目を四捨五入。0.005(秒) * 1000(ミリ秒単位) = 5

            return TimeSpan.FromMilliseconds(waveTime).ToString("m\\:ss\\.ff");
        }

        /// <summary>
        /// 再生時間(Tick)のフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetWaveTickString(WaveFile waveFile)
        {
            if (waveFile == null) return NotAvailable.Text;

            int smfConvertTimeBase = 48; // 四分音符分解能デフォルト値 48 を使う
            double frameCount = (double)(waveFile.IsLoop ?
                                          waveFile.LoopEndFrame :
                                          waveFile.FrameCount);
            double timed = frameCount / waveFile.SampleRate;
            timed += 0.005; // 下３桁目を四捨五入
            long tick = (long)(timed * (120 / 60) * smfConvertTimeBase);
            // 120 : テンポデフォルト値
            // 120/60 : 1秒間に四分音符がいくつあるか
            return tick.ToString();
        }

        /// <summary>
        /// サンプルレートのフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetSampleRateString(WaveFile waveFile)
        {
            if (waveFile == null)
            {
                return NotAvailable.Text;
            }

            return GetSampleRateString(waveFile.SampleRate);
        }

        /// <summary>
        /// サンプルレートのフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetSampleRateString(int sampleRate)
        {
            return String.Format("{0} Hz", sampleRate);
        }

        /// <summary>
        /// ビットレートのフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetWaveBitRateString(WaveFile waveFile)
        {
            if (waveFile == null)
            {
                return NotAvailable.Text;
            }

            return GetWaveBitRateString(waveFile, waveFile.SampleRate);
        }

        /// <summary>
        /// 指定したサンプルレートを用いてビットレートのフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetWaveBitRateString(WaveFile waveFile, int sampleRate)
        {
            if (waveFile == null)
            {
                return NotAvailable.Text;
            }

            float bitrate = sampleRate * waveFile.ChannelCount * waveFile.SampleBit / 1000f;

            if (bitrate < 1)
            {
                return String.Format("{0:g} kbps", bitrate);
            }
            else
            {
                return String.Format("{0:f0} kbps", bitrate); // f0 小数点以下は表示しない
            }
        }

        /// <summary>
        /// 量子化ビット数のフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetWaveSampleBitString(WaveFile waveFile)
        {
            if (waveFile == null)
            {
                return NotAvailable.Text;
            }

            return String.Format("{0} bit", waveFile.SampleBit);
        }

        /// <summary>
        /// チャンネル数のフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetChannelString(WaveFile waveFile)
        {
            if (waveFile == null)
            {
                return NotAvailable.Text;
            }

            return GetChannelString(waveFile.ChannelCount);
        }

        /// <summary>
        /// チャンネル数のフォーマットされたテキストを取得します。
        /// </summary>
        public static string GetChannelString(int channelCount)
        {
            if (channelCount <= 0)
            {
                return NotAvailable.Text;
            }

            switch (channelCount)
            {
                case 1:
                    return "Mono";
                case 2:
                    return "Stereo";
                default:
                    return channelCount.ToString();
            }
        }
    }
}
