﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.FileFormats.Wave
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.IO;

    public class WaveFileAiff : WaveFile
    {
        private AiffCommonChunk commonChunk = new AiffCommonChunk();
        private AiffInstChunk instChunk = new AiffInstChunk();
        private AiffSoundDataChunk soundDataChunk = new AiffSoundDataChunk();
        private AiffMarkerChunk markerChunk = new AiffMarkerChunk();
        private List<IRegionInfo> regions = new List<IRegionInfo>();
        private List<IMarkerInfo> markers = new List<IMarkerInfo>();

        public AiffCommonChunk CommonChunk
        {
            get { return commonChunk; }
            set { commonChunk = value; }
        }
        public AiffInstChunk InstChunk
        {
            get { return instChunk; }
            set { instChunk = value; }
        }
        public AiffSoundDataChunk SoundDataChunk
        {
            get { return soundDataChunk; }
            set { soundDataChunk = value; }
        }
        public AiffMarkerChunk MarkerChunk
        {
            get { return markerChunk; }
            set { markerChunk = value; }
        }

        public class AiffCommonChunk
        {
            private Int16 numChannels = 1;
            private UInt32 numSampleFrames = 0;
            private Int16 sampleSize = 16;
            private double sampleRate = 32000.0;

            public Int16 NumChannels
            {
                get { return this.numChannels; }
                set { this.numChannels = value; }
            }

            public UInt32 NumSampleFrames
            {
                get { return this.numSampleFrames; }
                set { this.numSampleFrames = value; }
            }

            public Int16 SampleSize
            {
                get { return this.sampleSize; }
                set { this.sampleSize = value; }
            }

            public double SampleRate
            {
                get { return this.sampleRate; }
                set { this.sampleRate = value; }
            }

            public void Read(BinaryReader reader)
            {
                this.numChannels = reader.ReadInt16();
                this.numSampleFrames = reader.ReadUInt32();
                this.sampleSize = reader.ReadInt16();
                UInt16 head = reader.ReadUInt16();
                UInt32 hiMant = reader.ReadUInt32();
                UInt32 loMant = reader.ReadUInt32();

                IeeeExtended ie = new IeeeExtended(head, hiMant, loMant);
                this.sampleRate = ie.ToDouble();
            }
        }

        public class AiffInstChunk
        {
            private byte baseNote = 60;
            private byte detune = 0;
            private byte lowNote = 0;
            private byte highNote = 127;
            private byte lowVelocity = 0;
            private byte highVelocity = 127;
            private Int16 gain = 0;
            private AiffLoop sustainLoop = new AiffLoop();
            private AiffLoop releaseLoop = new AiffLoop();

            public byte BaseNote
            {
                get { return this.baseNote; }
                set { this.baseNote = value; }
            }

            public byte Detune
            {
                get { return this.detune; }
                set { this.detune = value; }
            }

            public byte LowNote
            {
                get { return this.lowNote; }
                set { this.lowNote = value; }
            }

            public byte HighNote
            {
                get { return this.highNote; }
                set { this.highNote = value; }
            }

            public byte LowVelocity
            {
                get { return this.lowVelocity; }
                set { this.lowVelocity = value; }
            }

            public byte HighVelocity
            {
                get { return this.highVelocity; }
                set { this.highVelocity = value; }
            }

            public Int16 Gain
            {
                get { return this.gain; }
                set { this.gain = value; }
            }

            public AiffLoop SustainLoop
            {
                get { return this.sustainLoop; }
                set { this.sustainLoop = value; }
            }

            public AiffLoop ReleaseLoop
            {
                get { return this.releaseLoop; }
                set { this.releaseLoop = value; }
            }

            public void Read(BinaryReader reader)
            {
                baseNote = reader.ReadByte();
                detune = reader.ReadByte();
                lowNote = reader.ReadByte();
                highNote = reader.ReadByte();
                lowVelocity = reader.ReadByte();
                highVelocity = reader.ReadByte();
                gain = reader.ReadInt16();
                sustainLoop.Read(reader);
                releaseLoop.Read(reader);
            }
        }

        public class AiffSoundDataChunk
        {
            private UInt32 dataOffset = 0;
            private UInt32 dataBlockSize = 0;

            public UInt32 DataOffset
            {
                get { return this.dataOffset; }
                set { this.dataOffset = value; }
            }

            public UInt32 DataBlockSize
            {
                get { return this.dataBlockSize; }
                set { this.dataBlockSize = value; }
            }


            public void Read(BinaryReader reader)
            {
                dataOffset = reader.ReadUInt32();
                dataBlockSize = reader.ReadUInt32();
            }
        }

        public class AiffLoop
        {
            public enum Mode : short
            {
                NoLoop = 0,
                Forward = 1,
                ForwardBackward = 2,
            }

            private Mode playMode = Mode.NoLoop;
            private Int16 beginLoop = 0;
            private Int16 endLoop = 0;

            public Mode PlayMode
            {
                get { return this.playMode; }
                set { this.playMode = value; }
            }

            public Int16 BeginLoop
            {
                get { return this.beginLoop; }
                set { this.beginLoop = value; }
            }

            public Int16 EndLoop
            {
                get { return this.endLoop; }
                set { this.endLoop = value; }
            }

            public void Read(BinaryReader reader)
            {
                playMode = (AiffLoop.Mode)reader.ReadInt16();
                beginLoop = reader.ReadInt16();
                endLoop = reader.ReadInt16();
            }
        }

        public class AiffMarker
        {
            private UInt32 position;
            private string name;

            public UInt32 Position
            {
                get { return this.position; }
                set { this.position = value; }
            }

            public string Name
            {
                get { return this.name; }
                set { this.name = value; }
            }

        }

        public class AiffMarkerChunk
        {
            private SortedList list = new SortedList();

            public AiffMarker GetMarker(Int16 id)
            {
                return (AiffMarker)list[id];
            }

            public void Read(BinaryReader reader)
            {
                list.Clear();

                UInt16 numMakers = reader.ReadUInt16();
                for (int i = 0; i < numMakers; i++)
                {
                    Int16 id = reader.ReadInt16();
                    UInt32 position = reader.ReadUInt32();
                    Byte nameLen = reader.ReadByte();

                    AiffMarker marker = new AiffMarker();
                    marker.Position = position;
                    marker.Name = new string(reader.ReadChars(nameLen));

                    list.Add(id, marker);

                    reader.BaseStream.Position = ((reader.BaseStream.Position + 0x01) & ~0x01);
                }
            }
        }

        public override bool IsLoop
        {
            get
            {
                if (instChunk.SustainLoop.PlayMode == AiffLoop.Mode.NoLoop) return false;
                AiffMarker marker = markerChunk.GetMarker(instChunk.SustainLoop.BeginLoop);
                if (marker == null) return false;
                marker = markerChunk.GetMarker(instChunk.SustainLoop.EndLoop);
                if (marker == null) return false;
                return true;
            }
        }
        public override long LoopStartFrame
        {
            get
            {
                if (instChunk.SustainLoop.PlayMode == AiffLoop.Mode.NoLoop) return 0;
                Int16 markID = instChunk.SustainLoop.BeginLoop;
                AiffMarker marker = markerChunk.GetMarker(markID);
                return marker == null ? 0 : marker.Position;
            }
            set
            {
                if (instChunk.SustainLoop.PlayMode == AiffLoop.Mode.NoLoop) return;
                Int16 markID = instChunk.SustainLoop.BeginLoop;
                AiffMarker marker = markerChunk.GetMarker(markID);
                marker.Position = (uint)value;
            }
        }
        public override long LoopEndFrame
        {
            get
            {
                if (instChunk.SustainLoop.PlayMode == AiffLoop.Mode.NoLoop) return 0;
                Int16 markID = instChunk.SustainLoop.EndLoop;
                AiffMarker marker = markerChunk.GetMarker(markID);
                return marker == null ? 0 : marker.Position;
            }
            set
            {
                if (instChunk.SustainLoop.PlayMode == AiffLoop.Mode.NoLoop) return;
                Int16 markID = instChunk.SustainLoop.EndLoop;
                AiffMarker marker = markerChunk.GetMarker(markID);
                marker.Position = (uint)value;
            }
        }

        public override int ChannelCount
        {
            get
            {
                return commonChunk.NumChannels;
            }
            set
            {
                commonChunk.NumChannels = (short)value;
            }
        }
        public override long FrameCount
        {
            get
            {
                return commonChunk.NumSampleFrames;
            }
            set
            {
                commonChunk.NumSampleFrames = (uint)value;
            }
        }
        public override int SampleBit
        {
            get
            {
                return commonChunk.SampleSize;
            }
            set
            {
                commonChunk.SampleSize = (short)value;
            }
        }
        public override int SampleRate
        {
            get
            {
                return (int)commonChunk.SampleRate;
            }
            set
            {
                commonChunk.SampleRate = value;
            }
        }
        public override int OriginalKey
        {
            get
            {
                return (int)instChunk.BaseNote;
            }
            set
            {
                instChunk.BaseNote = (byte)value;
            }
        }

        public override double WaveTime { get; set; }

        public override IList<IRegionInfo> Regions
        {
            get { return this.regions; }
        }

        public override IList<IMarkerInfo> Markers
        {
            get { return this.markers; }
        }
    }

    internal class IeeeExtended
    {
        private UInt16 head;
        private UInt32 hiMant;
        private UInt32 loMant;

        public IeeeExtended(UInt16 h, UInt32 hi, UInt32 lo)
        {
            head = h;
            hiMant = hi;
            loMant = lo;
        }

        public double ToDouble()
        {
            double f;
            int expon;

            expon = head & 0x7fff;

            if (expon == 0 && hiMant == 0 && loMant == 0)
            {
                f = 0;
            }
            else
            {
                if (expon == 0x7FFF)
                {    /* Infinity or NaN */
                    f = Double.MaxValue;
                }
                else
                {
                    expon -= 0x3fff;

                    f = hiMant * Math.Pow(2.0, (expon -= 31));
                    f += loMant * Math.Pow(2.0, (expon -= 32));
                }
            }

            if ((head & 0x8000) != 0)
                return -f;
            else
                return f;
        }
    }
}
