﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate
{
    using System;
    using Core.Parameters;
    using Projects;
    using ToolDevelopmentKit;

    internal class BankItemParamTranslator : CommonItemParamTranslator
    {
        public BankItemParamTranslator()
        {
            // インストルメント
            this.AddHandler(
               ProjectParameterNames.Instrument.EnvelopeMode, typeof(XmlInstrumentEnvelopeModeParam),
               XmlInstrumentEnvelopeModeParam2Model, Model2XmlInstrumentEnvelopeModeParam);
            this.AddHandler(ProjectParameterNames.Instrument.ProgramNo, typeof(XmlProgramNoParam));

            // キーリージョン
            this.AddHandler(ProjectParameterNames.KeyRegion.KeyMax, typeof(XmlKeyMaxParam));
            this.AddHandler(ProjectParameterNames.KeyRegion.KeyMin, typeof(XmlKeyMinParam));

            // ベロシティリージョン
            this.AddHandler(ProjectParameterNames.VelocityRegion.KeyGroup, typeof(XmlKeyGroupParam));
            this.AddHandler(
               ProjectParameterNames.VelocityRegion.InterpolationType,
               typeof(XmlInstrumentEnvelopeModeParam),
               XmlInterpolationTypeParam2Model, Model2XmlInterpolationTypeParam);
            this.AddHandler(ProjectParameterNames.VelocityRegion.OriginalKey, typeof(XmlOriginalKeyParam));
            this.AddHandler(
               XmlParameterNames.InstrumentNoteOffMode, typeof(XmlInstrumentEnvelopeModeParam),
               XmlInstrumentNoteOffModeParam2Model, InvalidOperation);
            this.AddHandler(
               ProjectParameterNames.VelocityRegion.PercussionMode, typeof(XmlInstrumentEnvelopeModeParam),
               InvalidOperation, Model2XmlInstrumentNoteOffModeParam);
            this.AddHandler(ProjectParameterNames.VelocityRegion.PitchCents, typeof(XmlPitchCentsParam));
            this.AddHandler(ProjectParameterNames.VelocityRegion.PitchSemitones, typeof(XmlPitchSemitonesParam));
            this.AddHandler(ProjectParameterNames.VelocityRegion.VelocityMax, typeof(XmlVelocityMaxParam));
            this.AddHandler(ProjectParameterNames.VelocityRegion.VelocityMin, typeof(XmlVelocityMinParam));
        }

        //-----------------------------------------------------------------
        // InstrumentEnvelopeMode
        //-----------------------------------------------------------------

        private void XmlInstrumentEnvelopeModeParam2Model(XmlParameter paramXml, IParameterProvider dest)
        {
            Assertion.Argument.NotNull(paramXml);
            Assertion.Argument.NotNull(dest);

            XmlInstrumentEnvelopeModeParam src = paramXml as XmlInstrumentEnvelopeModeParam;
            InstrumentEnvelopeMode value = InstrumentEnvelopeModeEx.Parse(src.Value);

            this.SetParameterValue(dest, src.Name, value);
        }

        private void Model2XmlInstrumentEnvelopeModeParam(
            string paramName, IParameterValue value, Type xmlParamType, IXmlParameterProvider dest)
        {
            Assertion.Argument.NotNull(paramName);
            Assertion.Argument.NotNull(value);
            Assertion.Argument.True(value is InstrumentEnvelopeModeParameterValue);
            Assertion.Argument.NotNull(xmlParamType);
            Assertion.Argument.NotNull(dest);

            XmlInstrumentEnvelopeModeParam destValue = new XmlInstrumentEnvelopeModeParam()
            {
                Value = (value as InstrumentEnvelopeModeParameterValue).Value.ToText(),
            };

            dest.Parameters.Add(destValue);
        }

        //-----------------------------------------------------------------
        // PercussionMode / InstrumentNoteOffMode
        //-----------------------------------------------------------------

        private void XmlInstrumentNoteOffModeParam2Model(XmlParameter paramXml, IParameterProvider dest)
        {
            Assertion.Argument.NotNull(paramXml);
            Assertion.Argument.NotNull(dest);

            XmlInstrumentNoteOffModeParam src = paramXml as XmlInstrumentNoteOffModeParam;
            bool isPercussionMode = false;

            switch (src.Value)
            {
                case XmlInstrumentNoteOffMode.Ignore:
                    isPercussionMode = true;
                    break;

                case XmlInstrumentNoteOffMode.Release:
                    isPercussionMode = false;
                    break;

                default:
                    throw new Exception("invalid InstrumentNoteOffModeParam.");
            }

            this.SetParameterValue(dest, ProjectParameterNames.VelocityRegion.PercussionMode, isPercussionMode);
        }

        private void Model2XmlInstrumentNoteOffModeParam(
            string paramName, IParameterValue value, Type xmlParamType, IXmlParameterProvider dest)
        {
            Assertion.Argument.NotNull(paramName);
            Assertion.Argument.NotNull(value);
            Assertion.Argument.True(value is BoolParameterValue);
            Assertion.Argument.NotNull(xmlParamType);
            Assertion.Argument.NotNull(dest);

            XmlInstrumentNoteOffModeParam destValue = new XmlInstrumentNoteOffModeParam()
            {
                Value = ((bool)value.Value) ? XmlInstrumentNoteOffMode.Ignore : XmlInstrumentNoteOffMode.Release,
            };

            dest.Parameters.Add(destValue);
        }

        //-----------------------------------------------------------------
        // InterpolationType
        //-----------------------------------------------------------------

        private void XmlInterpolationTypeParam2Model(XmlParameter paramXml, IParameterProvider dest)
        {
            Assertion.Argument.NotNull(paramXml);
            Assertion.Argument.NotNull(dest);

            XmlInterpolationTypeParam src = paramXml as XmlInterpolationTypeParam;
            InterpolationType value = InterpolationTypeEx.Parse(src.Value);

            this.SetParameterValue(dest, src.Name, value);
        }

        private void Model2XmlInterpolationTypeParam(
            string paramName, IParameterValue value, Type xmlParamType, IXmlParameterProvider dest)
        {
            Assertion.Argument.NotNull(paramName);
            Assertion.Argument.NotNull(value);
            Assertion.Argument.True(value is InterpolationTypeParameterValue);
            Assertion.Argument.NotNull(xmlParamType);
            Assertion.Argument.NotNull(dest);

            XmlInterpolationTypeParam destValue = new XmlInterpolationTypeParam()
            {
                Value = (value as InterpolationTypeParameterValue).Value.ToText(),
            };

            dest.Parameters.Add(destValue);
        }
    }
}
