﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.IO;
    using Core;
    using Core.Parameters;
    using Projects;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.Conversion;
    using System.Linq;

    internal class SoundProjectXml2ModelTranslator : Xml2ModelTranslator
    {
        public SoundProjectXml2ModelTranslator(IObjectFactory<Type, Component> componentFactory)
            : base(componentFactory, new SoundProjectItemParamTranslator())
        {
            this.AddConverter<XmlSoundProject, SoundProject>(XmlSoundProjectToSoundProject);
            this.AddConverter<XmlProjectSoundSet, ProjectSoundSetComponent>(XmlProjectSoundSetToProjectSoundSetComponent);
            this.AddConverter<XmlFolder, FolderComponent>(XmlFolderToFolderComponent);
            this.AddConverter<XmlSoundArchivePlayer, SoundProject>(XmlSoundArchivePlayerToSoundProject);
            this.AddConverter<XmlConvert, SoundProject>(XmlConvertToSoundProject);
            this.AddConverter<XmlItemNaming, SoundProject>(XmlItemNamingToSoundProject);
            this.AddConverter<XmlCommentColumnText, SoundProject>(XmlCommentColumnTextToSoundProject);
            this.AddConverter<XmlColorComment, SoundProject>(XmlColorCommentToSoundProject);
            this.AddConverter<string, SoundListOutputType>(StringToSoundListOutputType);
            this.AddConverter<XmlProjectSetting, SoundProject>(XmlProjectSettingToSoundProject);
            this.AddDefaultConverter<XmlSoundListOutput, SoundListOutput>();
            this.AddDefaultConverter<XmlListOutput, ListOutput>();
            this.AddDefaultConverter<XmlListColumnOutput, ListColumnOutput>();

            this.AddConverter<string, StructureTypes>(StringToStructureTypes);
            this.AddDefaultConverter<XmlUserParameterSetting, UserParameterStructureSetting>();
            this.AddConverter<XmlFileEvent, SoundProject>(XmlFileEventToSoundProject);

            this.AddConverter<XmlSndEdit, SoundProject>(XmlSndEditToSoundProject);
            this.AddConverter<XmlConnection, SoundProject>(XmlConnectionToSoundProject);

            this.AddConverter<XmlStatistics, SoundProject>(XmlStatisticsToSoundProject);
        }

        public SoundProject Run(XmlSoundProject xmlSoundProject)
        {
            return this.Run<SoundProject>(xmlSoundProject) as SoundProject;
        }

        /// <summary>
        /// フォルダ対応前の旧フォーマット用の読み込み処理です。
        ///
        /// <ProjectSoundSets>
        ///   < ProjectSoundSet Name = "Common" >
        ///     < Parameters >
        ///       < FilePath > Common.fsst </ FilePath >
        ///     </ Parameters >
        ///   </ ProjectSoundSet >
        /// </ ProjectSoundSets >
        /// </summary>
        private void TranslateProjectSoundSetComponents(
            IEnumerable srcChildren, SoundProject dest, Converter converter)
        {
            Assertion.Argument.NotNull(srcChildren);
            Assertion.Argument.NotNull(dest);
            Assertion.Argument.NotNull(converter);

            foreach (object item in srcChildren)
            {
                ProjectSoundSetComponent projectSoundSetComponent = converter.Convert(item, typeof(ProjectSoundSetComponent)) as ProjectSoundSetComponent;

                if (projectSoundSetComponent == null)
                {
                    throw new Exception("invalid type");
                }

                dest.Children.Add(projectSoundSetComponent);
            }
        }

        private void TranslateProjectSoundSetRoot(XmlProjectSoundSetRoot root, SoundProject dest, Converter converter)
        {
            TranslateProjectItems(dest, root.Items, dest, converter);
        }

        private void TranslateProjectItems(Component parentComponent, IEnumerable items, SoundProject soundProject, Converter converter)
        {
            foreach (var item in items)
            {
                if (item is XmlFolder)
                {
                    var component = converter.Convert(item, null) as FolderComponent;
                    parentComponent.Children.Add(component);

                    var xmlFolder = item as XmlFolder;
                    TranslateProjectItems(component, xmlFolder.Items, soundProject, converter);
                }
                else if (item is XmlProjectSoundSet)
                {
                    var component = converter.Convert(item, typeof(ProjectSoundSetComponent)) as ProjectSoundSetComponent;
                    if (component == null)
                    {
                        throw new Exception("invalid type");
                    }
                    parentComponent.Children.Add(component);
                }
            }
        }

        /// <summary>
        /// XmlSoundProject から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlSoundProjectToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlSoundProject);
            Assertion.Argument.True(dstType == typeof(SoundProject));

            XmlSoundProject xmlSrc = src as XmlSoundProject;

            SoundProject dest = new SoundProject();
            this.Context.SoundProject = dest;

            if (xmlSrc.ProjectSoundSetRoot != null)
            {
                // フォルダ構成とサウンドセットの情報
                TranslateProjectSoundSetRoot(xmlSrc.ProjectSoundSetRoot, dest, converter);
            }
            else
            {
                // フォルダ対応前の旧フォーマット用の読み込み処理です。
                TranslateProjectSoundSetComponents(xmlSrc.ProjectSoundSets, dest, converter);
            }

            // サウンドアーカイブプレイヤー設定
            if (xmlSrc.SoundArchivePlayer != null)
            {
                converter.Convert(xmlSrc.SoundArchivePlayer, typeof(SoundProject));
            }

            // コンバート設定
            if (xmlSrc.Convert != null)
            {
                converter.Convert(xmlSrc.Convert, typeof(SoundProject));

                if (xmlSrc.Convert.PreConvertCommands != null)
                {
                    dest.PreConvertCommands = xmlSrc.Convert.PreConvertCommands;
                }
                if (xmlSrc.Convert.PostConvertCommands != null)
                {
                    dest.PostConvertCommands = xmlSrc.Convert.PostConvertCommands;
                }
            }

            // アイテムの命名設定
            if (xmlSrc.ItemNaming != null)
            {
                converter.Convert(xmlSrc.ItemNaming, typeof(SoundProject));
            }

            // サウンドリスト出力設定
            if (xmlSrc.SoundListOutputs != null)
            {
                foreach (XmlSoundListOutput xmlSoundListOutput in xmlSrc.SoundListOutputs)
                {
                    SoundListOutput soundListOutput =
                        converter.Convert(xmlSoundListOutput, typeof(SoundListOutput)) as SoundListOutput;
                    dest.SoundListOutputSettings.ListOutputs.Add(soundListOutput);
                }
            }

            // コメントカラム名の設定
            if (xmlSrc.CommentColumnText != null)
            {
                converter.Convert(xmlSrc.CommentColumnText, typeof(SoundProject));
            }

            // カラーコメントの設定
            if (xmlSrc.ColorComment != null)
            {
                converter.Convert(xmlSrc.ColorComment, typeof(SoundProject));
            }

            // ユーザーコマンドの設定
            if (xmlSrc.UserCommands != null)
            {
                dest.UserCommands = ConvrtUserCommands(xmlSrc.UserCommands, converter);
            }

            // ユーザーパラメータの設定
            if (xmlSrc.UserParameterSettings != null)
            {
                for (int index = 0; index < dest.UserDataStructureSettings.Settings.Count; index++)
                {
                    if (index >= xmlSrc.UserParameterSettings.Count)
                    {
                        break;
                    }

                    UserParameterStructureSetting setting =
                        converter.Convert
                        (xmlSrc.UserParameterSettings[index],
                          typeof(UserParameterStructureSetting)) as UserParameterStructureSetting;
                    dest.UserDataStructureSettings.Settings[index].CopyFrom(setting);
                }
            }

            // プロジェクト設定
            if (xmlSrc.ProjectSetting != null)
            {
                converter.Convert(xmlSrc.ProjectSetting, typeof(SoundProject));
            }

            // ファイルイベントの設定
            if (xmlSrc.FileEvent != null)
            {
                converter.Convert(xmlSrc.FileEvent, typeof(SoundProject));
            }

            // sndedit 設定
            if (xmlSrc.SndEdit != null)
            {
                converter.Convert(xmlSrc.SndEdit, typeof(SoundProject));
            }

            // 接続、切断の 設定
            if (xmlSrc.Connection != null)
            {
                converter.Convert(xmlSrc.Connection, typeof(SoundProject));
            }

            // 統計の設定
            if (xmlSrc.Statistics != null)
            {
                converter.Convert(xmlSrc.Statistics, typeof(SoundProject));
            }

            // サウンドアーカイブの出力設定
            if (xmlSrc.SoundArchiveSettings != null)
            {
                foreach (var xmlSetting in xmlSrc.SoundArchiveSettings)
                {
                    dest.UnresolvedSoundArchiveOutputTypes.Add(xmlSetting.Name, SoundArchiveOutputTypesEx.Parse(xmlSetting.OutputType));
                }
            }

            return dest;
        }

        /// <summary>
        /// XmlProjectSoundSet から ProjectSoundSetComponent に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlProjectSoundSetToProjectSoundSetComponent(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlProjectSoundSet);
            Assertion.Argument.True(dstType == typeof(ProjectSoundSetComponent));

            XmlProjectSoundSet xmlSrc = src as XmlProjectSoundSet;

            XmlFilePathParam filePathParam =
                xmlSrc.Parameters.Find((XmlParameter item) => item.Name == XmlParameterNames.FilePath)
                as XmlFilePathParam;

            if (filePathParam == null)
            {
                throw new Exception("filePath not found.");
            }

            if (this.Context.SoundProject == null)
            {
                throw new Exception("soundProject not found.");
            }

            var dest = new ProjectSoundSetComponent(xmlSrc.Name, this.PathResolver.GetFullPath(filePathParam.Value));

            return dest;
        }

        private object XmlFolderToFolderComponent(object src, Type dstType, Converter converter)
        {
            var xmlFolder = src as XmlFolder;
            var dest = new FolderComponent() { Name = xmlFolder.Name };
            return dest;
        }

        /// <summary>
        /// XmlSoundArchivePlayer から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlSoundArchivePlayerToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlSoundArchivePlayer);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlSoundArchivePlayer, this.Context.SoundProject, converter);
            return null;
        }

        /// <summary>
        /// XmlConvert から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlConvertToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlConvert);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlConvert, this.Context.SoundProject, converter);
            return null;
        }

        /// <summary>
        /// XmlItemNaming から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlItemNamingToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlItemNaming);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlItemNaming, this.Context.SoundProject, converter);
            return null;
        }

        /// <summary>
        /// XmlCommentColumnText から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlCommentColumnTextToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlCommentColumnText);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlCommentColumnText, this.Context.SoundProject, converter);
            return null;
        }

        /// <summary>
        /// XmlColorComment から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlColorCommentToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlCommentColumnText);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlColorComment, this.Context.SoundProject, converter);
            return null;
        }

        /// <summary>
        /// String から SoundListOutputType に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object StringToSoundListOutputType(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is string);
            Assertion.Argument.True(dstType == typeof(SoundListOutputType));

            return SoundListOutputTypeEx.Parse(src as string);
        }

        /// <summary>
        /// String から StructureTypes に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object StringToStructureTypes(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is string);
            Assertion.Argument.True(dstType == typeof(StructureTypes));

            return StructureTypesEx.Parse(src as string);
        }

        /// <summary>
        ///
        /// </summary>
        private UserCommand[] ConvrtUserCommands(IEnumerable<XmlUserCommand> xmlUserCommands, Converter converter)
        {
            List<UserCommand> list = new List<UserCommand>();

            foreach (XmlUserCommand xmlUserCommand in xmlUserCommands)
            {
                UserCommand userCommand = converter.Convert
                    (xmlUserCommand, typeof(UserCommand)) as UserCommand;
                list.Add(userCommand);
            }

            while (list.Count < UserCommand.MaxCount)
            {
                list.Add(new UserCommand());
            }

            return list.ToArray();
        }

        /// <summary>
        /// XmlProjectSetting から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlProjectSettingToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlProjectSetting);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlProjectSetting, this.Context.SoundProject, converter);

            this.Context.SoundProject.ProjectComment =
                this.Context.SoundProject.ProjectComment.Replace("\n", "\r\n");

            return null;
        }

        /// <summary>
        /// XmlFileEvent から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlFileEventToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlFileEvent);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlFileEvent, this.Context.SoundProject, converter);

            return null;
        }

        /// <summary>
        /// XmlSndEdit から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlSndEditToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlSndEdit);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlSndEdit, this.Context.SoundProject, converter);
            return null;
        }

        /// <summary>
        /// XmlConnection から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlConnectionToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlConnection);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            XmlConnection connection = (src as XmlConnection);

            if (connection.Parameters.Find(p => p is XmlTimeOutParam) != null)
            {
                this.Context.SoundProject.EnabledConnectionTimeOut = true;
            }
            TranslateParameters(connection, this.Context.SoundProject, converter);

            return null;
        }

        /// <summary>
        /// XmlStatistics から SoundProject に変換します。
        /// </summary>
        /// <param name="src">入力です。</param>
        /// <param name="dstType">出力型です。</param>
        /// <param name="converter">変換の状態です。Converterから渡されます。</param>
        /// <returns>出力です。</returns>
        private object XmlStatisticsToSoundProject(object src, Type dstType, Converter converter)
        {
            Assertion.Argument.True(src is XmlStatistics);
            Assertion.Argument.True(dstType == typeof(SoundProject));
            Ensure.Operation.True(this.Context.SoundProject != null);

            TranslateParameters(src as XmlStatistics, this.Context.SoundProject, converter);

            return null;
        }
    }
}
