﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.FileFormats.Audio
{
    using System.IO;
    using NintendoWare.ToolDevelopmentKit;

    public class AdtsHeader
    {
        private byte[] data = new byte[0];

        private enum SamplesPerSecType
        {
            Type_96000 = 0,
            Type_88200 = 1,
            Type_64000 = 2,
            Type_48000 = 3,
            Type_44100 = 4,
            Type_32000 = 5,
            Type_24000 = 6,
            Type_22050 = 7,
            Type_16000 = 8,
            Type_12000 = 9,
            Type_11025 = 10,
            Type_8000 = 11,
            Type_7350 = 12,
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="data">7 or 9 byte のバイナリデータを指定します。</param>
        public AdtsHeader(byte[] data)
        {
            Ensure.Argument.True(data.Length == 7 || data.Length == 9);
            this.data = data;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// Syncword の有無を取得します。
        /// </summary>
        public bool HasSyncword
        {
            get
            {
                return
                    this.data.Length >= 2 &&
                    this.data[0] == 0xff &&
                    (this.data[1] & 0xf0) == 0xf0;
            }
        }

        /// <summary>
        /// CRC の有無を取得します。
        /// </summary>
        public bool HasCrc
        {
            get
            {
                return
                    this.data.Length >= 2 &&
                    (this.data[1] & 0x01) == 0x00;
            }
        }

        /// <summary>
        /// チャンネル数を取得します。
        /// </summary>
        public int Channel
        {
            get
            {
                if (this.data.Length < 4) { return -1; }
                return ((this.data[2] & 0x01) << 2) | ((this.data[3] & 0xC0) >> 6);
            }
        }

        public int SamplesPerSec
        {
            get
            {
                if (this.data.Length < 3) { return -1; }

                SamplesPerSecType type = (SamplesPerSecType)((this.data[2] & 0x3C) >> 2);

                switch (type)
                {
                    case SamplesPerSecType.Type_96000:
                        return 96000;
                    case SamplesPerSecType.Type_88200:
                        return 88200;
                    case SamplesPerSecType.Type_64000:
                        return 64000;
                    case SamplesPerSecType.Type_48000:
                        return 48000;
                    case SamplesPerSecType.Type_44100:
                        return 44100;
                    case SamplesPerSecType.Type_32000:
                        return 32000;
                    case SamplesPerSecType.Type_24000:
                        return 24000;
                    case SamplesPerSecType.Type_22050:
                        return 22050;
                    case SamplesPerSecType.Type_16000:
                        return 16000;
                    case SamplesPerSecType.Type_12000:
                        return 12000;
                    case SamplesPerSecType.Type_11025:
                        return 11025;
                    case SamplesPerSecType.Type_8000:
                        return 8000;
                    case SamplesPerSecType.Type_7350:
                        return 7350;
                }

                return 0;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ストリームから ADTS ヘッダを解析します。
        /// </summary>
        /// <param name="stream">ストリームを指定します。</param>
        /// <returns>
        /// AdtsHeader を返します。
        /// ヘッダサイズ分が読み取れなかった場合は、例外が発生します。
        /// </returns>
        public static AdtsHeader Parse(Stream stream)
        {
            Ensure.Argument.NotNull(stream);

            using (var reader = new BinaryReader(stream))
            {
                return new AdtsHeader(reader.ReadBytes(9));
            }
        }

        public bool IsValid()
        {
            if (!this.HasSyncword) { return false; }
            if (this.data.Length == 7 && this.HasCrc) { return false; }
            if (this.Channel < 0 || 7 < this.Channel) { return false; }
            return true;
        }
    }
}
