﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Documents
{
    using System;
    using NintendoWare.SoundFoundation.Core.Resources;

    /// <summary>
    /// ドキュメント管理機能を提供します。
    /// </summary>
    public interface IDocumentService
    {
        /// <summary>
        /// ドキュメントが開かれると発生します。
        /// </summary>
        event EventHandler<DocumentEventArgs> DocumentOpened;

        /// <summary>
        /// 開かれているドキュメントの一覧を取得します。
        /// </summary>
        Document[] Documents { get; }

        /// <summary>
        /// 指定した種類のドキュメントを作成します。
        /// </summary>
        /// <param name="typeName">作成するドキュメントタイプを指定します。</param>
        /// <returns>作成したドキュメントへの参照を返します。</returns>
        DocumentReference CreateDocument(string typeName);

        /// <summary>
        /// 指定したリソースのドキュメントを開きます。
        /// </summary>
        /// <param name="resouce">リソースを指定します。</param>
        /// <returns>開いたドキュメントへの参照を返します。</returns>
        DocumentReference OpenDocument(IStreamResource resouce);

        /// <summary>
        /// 指定したドキュメントを保存します。
        /// </summary>
        /// <param name="document">ドキュメントを指定します。</param>
        void SaveDocument(Document document);

        /// <summary>
        /// 指定したドキュメントのクローンを作成します。
        /// </summary>
        /// <param name="document">ドキュメントを指定します。</param>
        /// <returns>ドキュメントのクローンへの参照を返します。</returns>
        DocumentReference CloneDocument(Document document);

        /// <summary>
        /// ドキュメントを指定した種類のドキュメントに変換します。
        /// </summary>
        /// <param name="inputDocument">変換元のドキュメントを指定します。</param>
        /// <param name="outputDocumentTypeName">変換後のドキュメントタイプを指定します。</param>
        /// <returns>変換されたドキュメントへの参照を返します。</returns>
        DocumentReference ConvertDocument(Document inputDocument, string outputDocumentTypeName);

        /// <summary>
        /// 指定したリソースのドキュメントタイプを調べます。
        /// </summary>
        /// <param name="resource">リソースを指定します。</param>
        /// <returns>ドキュメントタイプを返します。</returns>
        string GetDocumentTypeName(IStreamResource resource);

        /// <summary>
        /// 指定したドキュメントタイプから変換可能なドキュメントタイプを調べます。
        /// </summary>
        /// <param name="typeName">ドキュメントタイプを指定します。</param>
        /// <returns>変換可能なドキュメントタイプの配列を返します。</returns>
        string[] FindConvertibleDocumentTypeNames(string typeName);

        /// <summary>
        /// 指定したドキュメントタイプ間の変換が可能かどうかを調べます。
        /// </summary>
        /// <param name="inputDocumentTypeName">変換元のドキュメントタイプを指定します。</param>
        /// <param name="outputDocumentTypeName">変換後のドキュメントタイプを指定します。</param>
        /// <returns>変換可能な場合は true、不可能な場合は false を返します。</returns>
        bool CanConvertDocument(string inputDocumentTypeName, string outputDocumentTypeName);
    }
}
