﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NintendoWare.SoundFoundation.Core.Resources;
using NintendoWare.SoundFoundation.Operations;

namespace NintendoWare.SoundFoundation.Documents
{
    /// <summary>
    /// ドキュメント情報を管理します。
    /// </summary>
    public abstract class Document : ICloneable, IDisposable
    {
        private IStreamResource _Resource = null;
        private OperationHistory _OperationHistory = new OperationHistory();

        public Document()
        {
            _OperationHistory.DirtyChanged += OnOperationHistoryDirtyChanged;
        }

        /// <summary>
        /// リソースが変更されると発生します。
        /// </summary>
        public event EventHandler ResourceChanged;

        /// <summary>
        /// ドキュメントの Dirty 状態が変更されると発生します。
        /// </summary>
        public event EventHandler DirtyChanged;

        ///--------------------------------
        /// <summary>
        /// ドキュメントの種類を取得します。
        /// </summary>
        public abstract string TypeName { get; }

        /// <summary>
        /// ドキュメントが編集されているかどうかを調べます。
        /// </summary>
        public bool IsDirty
        {
            get
            {
                if (_OperationHistory.IsDirty) { return true; }
                return IsDirtyOtherFactor;
            }
        }

        /// <summary>
        /// オペレーション履歴を取得します。
        /// </summary>
        public OperationHistory OperationHistory
        {
            get { return _OperationHistory; }
        }

        ///--------------------------------
        /// <summary>
        /// リソースを取得または設定します。
        /// </summary>
        public IStreamResource Resource
        {
            get { return _Resource; }
            set
            {
                if (value == _Resource) { return; }

                _Resource = value;
                OnResourceChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// 他の要因でドキュメントが編集されているかどうかを調べます。
        /// </summary>
        protected virtual bool IsDirtyOtherFactor
        {
            get { return false; }
        }

        ///--------------------------------
        /// <summary>
        /// ドキュメントのクローンを作成します。
        /// </summary>
        public abstract object Clone();

        /// <summary>
        /// Dirty 状態をリセットします。
        /// </summary>
        public void ResetDirty()
        {
            bool old = IsDirty;

            _OperationHistory.ResetSavedTransaction();

            if (old != IsDirty)
            {
                OnDirtyChanged(EventArgs.Empty);
            }

            OnResetDirty();
        }

        /// <summary>
        /// ドキュメントを破棄します。
        /// </summary>
        public void Close()
        {
            (this as IDisposable).Dispose();
        }

        void IDisposable.Dispose()
        {
            _Resource = null;
            _OperationHistory.Clear();

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// ドキュメントの Dirty 状態が変更されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected void OnDirtyChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != DirtyChanged)
            {
                DirtyChanged(this, e);
            }
        }

        /// <summary>
        /// Dirty 状態がリセットされると発生します。
        /// </summary>
        protected virtual void OnResetDirty()
        {
        }

        /// <summary>
        /// ドキュメントのリソースが変更されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        private void OnResourceChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (ResourceChanged != null)
            {
                ResourceChanged(this, new EventArgs());
            }
        }

        /// <summary>
        /// オペレーション履歴の Dirty 状態が変更されると発生します。
        /// </summary>
        /// <param name="sender">イベントの発行元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnOperationHistoryDirtyChanged(object sender, EventArgs e)
        {
            if (IsDirtyOtherFactor) { return; }
            OnDirtyChanged(EventArgs.Empty);
        }
    }
}
