﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Core
{
    using System.Collections.Generic;
    using ToolDevelopmentKit;

    /// <summary>
    /// オブジェクトファクトリを集約します。
    /// </summary>
    /// <typeparam name="TParam">パラメータを指定します。</typeparam>
    /// <typeparam name="TValue">生成するオブジェクトの型を指定します。</typeparam>
    public class ObjectAggregateFactory<TParam, TValue> : IObjectFactory<TParam, TValue>
    {
        private Dictionary<TParam, IObjectFactory<TParam, TValue>> factories =
            new Dictionary<TParam, IObjectFactory<TParam, TValue>>();

        /// <summary>
        /// 内包するオブジェクトファクトリのディクショナリを取得します。
        /// </summary>
        public IDictionary<TParam, IObjectFactory<TParam, TValue>> Factories
        {
            get { return this.factories; }
        }

        /// <summary>
        /// 指定されたパラメータからオブジェクトのインスタンスを作成します。
        /// </summary>
        /// <param name="param">パラメータを指定します。</param>
        /// <returns>生成したオブジェクトのインスタンスを返します。</returns>
        public TValue Create(TParam param)
        {
            Ensure.Argument.NotNull(param);

            IObjectFactory<TParam, TValue> factory = null;
            this.factories.TryGetValue(param, out factory);

            if (factory == null)
            {
                return this.CreateDefaultValue(param);
            }

            return factory.Create(param);
        }

        /// <summary>
        /// デフォルト値を作成します。
        /// 指定されたパラメータが見つからなかった場合に実行されます。
        /// </summary>
        /// <param name="param">パラメータを指定します。</param>
        /// <returns>デフォルト値を返します。</returns>
        protected virtual TValue CreateDefaultValue(TParam param)
        {
            return default(TValue);
        }
    }
}
