﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Core.IO
{
    using System;
    using System.Collections;
    using System.IO;

    /// <summary>
    /// Path の拡張機能を提供します。
    /// </summary>
    public static class PathEx
    {
        public static string MakeRelative(string strFilePath, string strBaseDir)
        {
            return PathInfo.MakeRelative(strFilePath, strBaseDir);
        }

        public static string Normalize(string strFilePath)
        {
            PathInfo pathInfo = new PathInfo(strFilePath);
            return pathInfo.GetFilePath();
        }

        private class PathInfo
        {
            private enum PathType
            {
                Absolute,
                Relative,
                Unc
            }

            private PathType type;
            private string volume = String.Empty;
            private string uncHost = String.Empty;
            private string fileBase = String.Empty;
            private string fileExt = String.Empty;
            private ArrayList dirList = new ArrayList();

            public PathInfo(string strFilePath)
            {
                int next = 0;

                int volSep = strFilePath.IndexOf(Path.VolumeSeparatorChar);
                if (volSep != -1)
                {
                    // ボリューム指定
                    this.type = PathType.Absolute;
                    this.volume = strFilePath.Substring(0, volSep);

                    next = volSep + 1;
                    while (next < strFilePath.Length)
                    {
                        if (!IsDirSep(strFilePath[next])) break;
                        next++;
                    }
                }
                else if (IsDirSep(strFilePath[0]))
                {
                    if (IsDirSep(strFilePath[1]))
                    {
                        // UNC形式
                        this.type = PathType.Unc;

                        next = 2;
                        while (next < strFilePath.Length)
                        {
                            if (!IsDirSep(strFilePath[next])) break;
                            next++;
                        }

                        int hostBegin = next;
                        while (next < strFilePath.Length)
                        {
                            if (IsDirSep(strFilePath[next])) break;
                            next++;
                        }
                        this.uncHost = strFilePath.Substring(hostBegin, next - hostBegin);
                        while (next < strFilePath.Length)
                        {
                            if (!IsDirSep(strFilePath[next])) break;
                            next++;
                        }
                    }
                    else
                    {
                        // 絶対パス
                        this.type = PathType.Absolute;

                        next = 1;
                    }
                }
                else
                {
                    // 相対パス
                    this.type = PathType.Relative;
                }

                this.dirList.Clear();

                // ディレクトリの解析
                int prev = next;
                while ((next = strFilePath.IndexOfAny("\\/".ToCharArray(), prev)) != -1)
                {
                    string dirName = strFilePath.Substring(prev, next - prev);

                    if (dirName != ".")
                    {
                        if (dirName == ".." && this.dirList.Count != 0 && (string)this.dirList[this.dirList.Count - 1] != "..")
                        {
                            this.dirList.RemoveAt(this.dirList.Count - 1);
                        }
                        else
                        {
                            this.dirList.Add(dirName);
                        }
                    }

                    next++;
                    while (next < strFilePath.Length)
                    {
                        if (!IsDirSep(strFilePath[next])) break;
                        next++;
                    }
                    prev = next;
                }

                string fileName = strFilePath.Substring(prev);
                int dotPos = fileName.LastIndexOf('.');
                if (dotPos != -1)
                {
                    this.fileBase = fileName.Substring(0, dotPos);
                    this.fileExt = fileName.Substring(dotPos);
                }
                else
                {
                    this.fileBase = fileName;
                    this.fileExt = String.Empty;
                }
            }

            public static string MakeRelative(string strFilePath, string strBaseDir)
            {
                string strCurDir = Directory.GetCurrentDirectory();

                if (strBaseDir == String.Empty)
                {
                    strBaseDir = "./";
                }

                PathInfo path = new PathInfo(Path.Combine(strCurDir, strFilePath));
                PathInfo dir = new PathInfo(Path.Combine(strCurDir, strBaseDir + Path.DirectorySeparatorChar));

                if (path.type == PathType.Unc)
                {
                    if ((dir.type != PathType.Unc) ||
                         String.Compare(path.uncHost, dir.uncHost, true) != 0)
                    {
                        return path.GetFilePath();
                    }
                }
                else
                {
                    if ((dir.type == PathType.Unc) ||
                         String.Compare(path.volume, dir.volume, true) != 0)
                    {
                        return path.GetFilePath();
                    }
                }

                int count = 0;
                IEnumerator p = path.dirList.GetEnumerator();
                IEnumerator d = dir.dirList.GetEnumerator();
                while (p.MoveNext() && d.MoveNext())
                {
                    if (String.Compare((string)p.Current, (string)d.Current, true) != 0) break;
                    count++;
                }

                path.dirList.RemoveRange(0, count);
                dir.dirList.RemoveRange(0, count);

                for (int i = 0; i < dir.dirList.Count; i++)
                {
                    path.dirList.Insert(0, "..");
                }

                path.type = PathType.Relative;
                return path.GetFilePath();
            }

            public string GetFilePath()
            {
                string path = ConstructDirPath();
                path += fileBase;
                path += fileExt;
                return path;
            }
            public string GetDirPath()
            {
                string path = ConstructDirPath();
                if (path == String.Empty) return "./";
                return path;
            }
            public string GetFileName()
            {
                string path = fileBase;
                path += fileExt;
                return path;
            }
            public string GetFilePathWithoutExtension()
            {
                string path = ConstructDirPath();
                path += fileBase;
                return path;
            }
            public string GetFileNameWithoutExtension()
            {
                return fileBase;
            }

            private bool IsDirSep(char c)
            {
                if (c == '\\') return true;
                if (c == '/') return true;
                return false;
            }

            private string ConstructDirPath()
            {
                string path = String.Empty;

                switch (type)
                {
                    case PathType.Unc:
                        path += "//";
                        path += uncHost;
                        path += "/";
                        break;
                    case PathType.Absolute:
                        if (volume != String.Empty)
                        {
                            path += volume;
                            path += Path.VolumeSeparatorChar;
                        }
                        path += "/";
                        break;
                    case PathType.Relative:
                    default:
                        break;
                }

                foreach (string dir in dirList)
                {
                    path += dir;
                    path += '/';
                }

                return path;
            }
        }
    }
}
