﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading;
using NintendoWare.SoundFoundation.Core.Threading;
using NintendoWare.SoundFoundation.Logs;
using NintendoWare.ToolDevelopmentKit;

namespace NintendoWare.SoundFoundation.Conversion
{
    public class ConversionProcessScheduler
    {
        private const int ExitTimeout = 10 * 1000;

        private uint executionCountMax = 1;
        private bool isErrorQuit = true;

        private HashSet<IConversionProcessor> notReady = new HashSet<IConversionProcessor>();
        private Queue<IConversionProcessor> readyQueue = new Queue<IConversionProcessor>();
        private HashSet<IConversionProcessor> running = new HashSet<IConversionProcessor>();
        private object queueLock = new object();

        private Thread schedulingThread;
        private AutoResetEvent nextEvent = new AutoResetEvent(false);
        private ManualResetEvent stopEvent = new ManualResetEvent(true);
        private bool isCanceled = false;

        private ConversionContext context;

        public ConversionProcessScheduler()
        {
            this.IsForced = false;
        }

        public event EventHandler Completed;

        public bool IsForced { get; set; }

        public uint ExecutionCountMax
        {
            get { return this.executionCountMax; }
            set { this.executionCountMax = value; }
        }

        public int CompletedProcessCount { get; private set; }

        public int TotalProcessCount { get; private set; }

        public bool IsErrorQuit
        {
            get { return this.isErrorQuit; }
            set { this.isErrorQuit = value; }
        }

        public bool IsSucceeded
        {
            get
            {
                if (this.context == null)
                {
                    return false;
                }

                if (this.IsFailed) { return false; }

                lock (this.queueLock)
                {
                    return (this.notReady.Count == 0 && this.running.Count == 0 && this.readyQueue.Count == 0);
                }
            }
        }

        public bool IsFailed
        {
            get
            {
                if (this.context == null)
                {
                    return false;
                }

                return this.context.IsFailed;
            }
        }

        public bool IsCanceled
        {
            get { return this.isCanceled; }
        }

        private bool IsEndThread
        {
            get
            {
                if (this.isErrorQuit && this.IsFailed) { return true; }
                if (this.isCanceled) { return true; }
                return this.IsSucceeded;
            }
        }

        public void Run(ConversionContext context)
        {
            Ensure.Operation.True(this.schedulingThread == null);

            this.context = context;

            try
            {
                this.Initialize(context.ConversionProcessors);
                this.Start();
            }
            catch
            {
                this.context = null;
                throw;
            }
        }

        public void Wait()
        {
            if (this.schedulingThread == null) { return; }
            Stop();
        }

        public void Cancel()
        {
            if (this.IsEndThread == true)
            {
                return;
            }
            this.isCanceled = true;
            this.nextEvent.Set();

            Stop();
        }

        protected virtual void OnCompleted(EventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.Completed != null)
            {
                this.Completed(this, e);
            }
        }

        private void Initialize(IEnumerable<IConversionProcessor> processors)
        {
            Assertion.Argument.NotNull(processors);

            lock (this.queueLock)
            {
                this.notReady.Clear();
                this.readyQueue.Clear();
                this.running.Clear();

                this.TotalProcessCount = 0;
                this.CompletedProcessCount = 0;

                foreach (var processor in processors)
                {
                    ++this.TotalProcessCount;

                    switch (processor.State)
                    {
                        case ConversionProcessState.NotReady:
                            this.notReady.Add(processor);
                            this.ListenProcessorState(processor);
                            break;

                        case ConversionProcessState.Ready:
                            this.Enqueue(processor);
                            break;

                        case ConversionProcessState.Running:
                        case ConversionProcessState.Error:
                            throw new Exception("internal error : invalid processor state.");
                    }
                }

                if (this.readyQueue.Count == 0 && this.notReady.Count > 0)
                {
                    throw new Exception("internal error");
                }
            }
        }

        private void Start()
        {
            try
            {
                this.nextEvent.Reset();
                this.stopEvent.Reset();
                this.isCanceled = false;

                this.schedulingThread = new Thread(Schedule);
                this.schedulingThread.Start();
            }
            catch
            {
                this.stopEvent.Set();
                throw;
            }
        }

        private void Stop()
        {
            this.stopEvent.WaitOne();
        }

        private void ListenProcessorState(IConversionProcessor processor)
        {
            Assertion.Argument.NotNull(processor);
            processor.StateChanged += OnProcessorStateChanged;
        }

        private void Enqueue(IConversionProcessor processor)
        {
            Assertion.Argument.NotNull(processor);

            lock (this.queueLock)
            {
                if (this.notReady.Contains(processor))
                {
                    this.notReady.Remove(processor);
                }

                this.readyQueue.Enqueue(processor);
            }
        }

        //-----------------------------------------------------------------
        // スケジューリングスレッドからのみコールされるメソッド
        //-----------------------------------------------------------------

        private void Schedule()
        {
            Thread.CurrentThread.InitializeCurrentUICulture();

            while (!IsEndThread)
            {
                if (NextProcess())
                {
                    continue;
                }

                while (true)
                {
                    if (this.nextEvent.WaitOne(1000))
                    {
                        break;
                    }
                }
            }

            this.WaitForProcessToComplete();

            this.schedulingThread = null;

            try
            {
                this.OnCompleted(EventArgs.Empty);
            }
            finally
            {
                this.stopEvent.Set();
            }
        }

        /// <summary>
        /// 実行中の処理が全て終了するまで待機します。
        /// </summary>
        private void WaitForProcessToComplete()
        {
            Stopwatch stopWatch = new Stopwatch();
            stopWatch.Start();

            while (this.running.Count > 0)
            {
                Thread.Sleep(100);

                if (stopWatch.ElapsedMilliseconds > ConversionProcessScheduler.ExitTimeout)
                {
                    break;
                }
            }

            stopWatch.Stop();
        }

        private bool NextProcess()
        {
            IConversionProcessor processor = null;

            lock (this.queueLock)
            {
                if (this.readyQueue.Count == 0) { return false; }
                if (this.executionCountMax > 0 &&
                    this.running.Count >= this.executionCountMax) { return false; }

                processor = this.readyQueue.Dequeue();
                processor.IsForced = this.IsForced;

                this.running.Add(processor);
            }

            new Thread(RunProcess).Start(processor);

            return true;
        }

        //-----------------------------------------------------------------
        // 処理スレッドからのみコールされるメソッド
        //-----------------------------------------------------------------

        private void RunProcess(object obj)
        {
            Thread.CurrentThread.InitializeCurrentUICulture();

            IConversionProcessor processor = obj as IConversionProcessor;
            Assertion.Argument.NotNull(processor);

            try
            {
                processor.Run(this.context);
            }
            catch (ConversionException exception)
            {
                foreach (OutputLine line in exception.Lines)
                {
                    this.context.Logger.AddLine(line);
                }
            }
            catch (Exception exception)
            {
                this.context.Logger.AddLine(new InternalErrorLine(exception.ToString()));
            }
            finally
            {
                ++this.CompletedProcessCount;
                ProcessDone(processor);
            }
        }

        private void ProcessDone(IConversionProcessor processor)
        {
            Assertion.Argument.NotNull(processor);

            lock (this.queueLock)
            {
                this.running.Remove(processor);
            }

            this.nextEvent.Set();
        }

        //-----------------------------------------------------------------
        // イベントハンドラ
        //-----------------------------------------------------------------

        private void OnProcessorStateChanged(object sender, ConversionProcessStateEventArgs e)
        {
            IConversionProcessor processor = sender as IConversionProcessor;
            Ensure.Argument.NotNull(processor);

            if (processor.State == ConversionProcessState.Ready)
            {
                this.Enqueue(processor);
            }
        }
    }
}
