﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareReport
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Text;
    using System.Xml;
    using Projects;
    using ToolDevelopmentKit;

    internal class SoundInfoProvider : IListItemInfoProvider<Sound>
    {
        public delegate string GetTextHandler(Column column, int itemIndex, Sound item);

        private IDictionary<string, GetTextHandler> getTextHandlers = new Dictionary<string, GetTextHandler>();

        private List<UserParameterStructureSetting> settings;

        public SoundInfoProvider(List<UserParameterStructureSetting> settings)
        {
            this.settings = settings;
            InitializeGetTextHandlers();
        }

        /// <summary>
        /// 指定したアイテムが有効かどうかを調べます。
        /// </summary>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>アイテムが有効なら true、無効なら false を返します。</returns>
        public bool IsEnabled(Sound item)
        {
            Assertion.Argument.NotNull(item);
            return item.IsHierarchyEnabled;
        }

        /// <summary>
        /// 指定したアイテムのカラーインデックスを取得します。
        /// </summary>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>アイテムのカラーインデックスを返します。</returns>
        public int GetColorIndex(Sound item)
        {
            Assertion.Argument.NotNull(item);
            return item.ColorIndex;
        }

        /// <summary>
        /// 指定したカラムとアイテムに対応するテキストを取得します。
        /// </summary>
        /// <param name="column">対象カラムを指定します。</param>
        /// <param name="itemIndex">対象アイテムのインデックスを指定します。</param>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>取得したテキストを返します。</returns>
        public string GetText(Column column, int itemIndex, Sound item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            if (column.Name.StartsWith(SoundProjectReportTraits.ColumnName.UserParameter + "_"))
            {
                return GetUserParameterText(settings[0].Structures, column, item.UserParameter);
            }
            else if (column.Name.StartsWith(SoundProjectReportTraits.ColumnName.UserParameter1 + "_"))
            {
                return GetUserParameterText(settings[1].Structures, column, item.UserParameter1);
            }
            else if (column.Name.StartsWith(SoundProjectReportTraits.ColumnName.UserParameter2 + "_"))
            {
                return GetUserParameterText(settings[2].Structures, column, item.UserParameter2);
            }
            else if (column.Name.StartsWith(SoundProjectReportTraits.ColumnName.UserParameter3 + "_"))
            {
                return GetUserParameterText(settings[3].Structures, column, item.UserParameter3);
            }

            return getTextHandlers[column.Name](column, itemIndex, item);
        }

        private string GetUserParameterText(List<UserParameterStructure> structures, Column column, ulong userParameter)
        {
            foreach (UserParameterStructure structure in structures)
            {
                if (column.Name.EndsWith("_" + structure.Label) == true)
                {
                    return UserParameterUtility.GetStringValue(structure, userParameter);
                }
            }

            return string.Empty;
        }

        private void InitializeGetTextHandlers()
        {
            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.ActorPlayer,
                (Column column, int itemIndex, Sound item) => item.ActorPlayer.ToString());

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.FileName,
                (Column column, int itemIndex, Sound item) => Path.GetFileName(item.FilePath));

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.ItemIndex,
                (Column column, int itemIndex, Sound item) => (itemIndex < 0) ? "-" : itemIndex.ToString());

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.ItemType,
                GetItemType);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Name,
                (Column column, int itemIndex, Sound item) => item.Name);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.PlayerPriority,
                (Column column, int itemIndex, Sound item) => item.PlayerPriority.ToString());

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.PlayerReference,
                (Column column, int itemIndex, Sound item) => item.PlayerReference);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Volume,
                (Column column, int itemIndex, Sound item) => item.Volume.ToString());

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment,
                (Column column, int itemIndex, Sound item) => item.Comment);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment1,
                (Column column, int itemIndex, Sound item) => item.Comment1);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment2,
                (Column column, int itemIndex, Sound item) => item.Comment2);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment3,
                (Column column, int itemIndex, Sound item) => item.Comment3);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment4,
                (Column column, int itemIndex, Sound item) => item.Comment4);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment5,
                (Column column, int itemIndex, Sound item) => item.Comment5);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment6,
                (Column column, int itemIndex, Sound item) => item.Comment6);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment7,
                (Column column, int itemIndex, Sound item) => item.Comment7);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment8,
                (Column column, int itemIndex, Sound item) => item.Comment8);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.Comment9,
                (Column column, int itemIndex, Sound item) => item.Comment9);

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.UserParameter,
                (Column column, int itemIndex, Sound item) => item.UserParameter.ToString());

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.UserParameter1,
                (Column column, int itemIndex, Sound item) => item.UserParameter1.ToString());

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.UserParameter2,
                (Column column, int itemIndex, Sound item) => item.UserParameter2.ToString());

            getTextHandlers.Add(
                SoundProjectReportTraits.ColumnName.UserParameter3,
                (Column column, int itemIndex, Sound item) => item.UserParameter3.ToString());
        }

        /// <summary>
        /// 指定したアイテムの種類を取得します。
        /// </summary>
        /// <param name="column">対象カラムを指定します。</param>
        /// <param name="itemIndex">対象アイテムのインデックスを指定します。</param>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>取得したアイテムの種類を返します。</returns>
        private string GetItemType(Column column, int itemIndex, Sound item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            if (item is StreamSoundBase)
            {
                return "Stream";
            }

            if (item is WaveSoundBase)
            {
                return "Wave";
            }

            if (item is SequenceSoundBase)
            {
                return "Sequence";
            }

            return "Unknown";
        }
    }
}
