﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareReport
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Text;
    using System.Xml;
    using FileFormats.Wave;
    using Projects;
    using Resources;
    using ToolDevelopmentKit;

    internal class InstrumentInfoProvider : IListItemInfoProvider<Instrument>
    {
        public delegate string GetTextHandler(Column column, int itemIndex, Instrument item);

        private static string[] KeyLabels = new String[]
            {
                "cnm1", "csm1", "dnm1", "dsm1", "enm1", "fnm1", "fsm1", "gnm1", "gsm1", "anm1", "asm1", "bnm1",
                "cn0", "cs0", "dn0", "ds0", "en0", "fn0", "fs0", "gn0", "gs0", "an0", "as0", "bn0",
                "cn1", "cs1", "dn1", "ds1", "en1", "fn1", "fs1", "gn1", "gs1", "an1", "as1", "bn1",
                "cn2", "cs2", "dn2", "ds2", "en2", "fn2", "fs2", "gn2", "gs2", "an2", "as2", "bn2",
                "cn3", "cs3", "dn3", "ds3", "en3", "fn3", "fs3", "gn3", "gs3", "an3", "as3", "bn3",
                "cn4", "cs4", "dn4", "ds4", "en4", "fn4", "fs4", "gn4", "gs4", "an4", "as4", "bn4",
                "cn5", "cs5", "dn5", "ds5", "en5", "fn5", "fs5", "gn5", "gs5", "an5", "as5", "bn5",
                "cn6", "cs6", "dn6", "ds6", "en6", "fn6", "fs6", "gn6", "gs6", "an6", "as6", "bn6",
                "cn7", "cs7", "dn7", "ds7", "en7", "fn7", "fs7", "gn7", "gs7", "an7", "as7", "bn7",
                "cn8", "cs8", "dn8", "ds8", "en8", "fn8", "fs8", "gn8", "gs8", "an8", "as8", "bn8",
                "cn9", "cs9", "dn9", "ds9", "en9", "fn9", "fs9", "gn9",
            };

        private static IDictionary<string, GetTextHandler> getTextHandlers = new Dictionary<string, GetTextHandler>();

        static InstrumentInfoProvider()
        {
            InitializeGetTextHandlers();
        }

        /// <summary>
        /// 指定したアイテムが有効かどうかを調べます。
        /// </summary>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>アイテムが有効なら true、無効なら false を返します。</returns>
        public bool IsEnabled(Instrument item)
        {
            Assertion.Argument.NotNull(item);
            return item.IsEnabled;
        }

        /// <summary>
        /// 指定したアイテムのカラーインデックスを取得します。
        /// </summary>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>アイテムのカラーインデックスを返します。</returns>
        public int GetColorIndex(Instrument item)
        {
            Assertion.Argument.NotNull(item);
            return item.ColorIndex;
        }

        /// <summary>
        /// 指定したカラムとアイテムに対応するテキストを取得します。
        /// </summary>
        /// <param name="column">対象カラムを指定します。</param>
        /// <param name="itemIndex">対象アイテムのインデックスを指定します。</param>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>取得したテキストを返します。</returns>
        public string GetText(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);
            return getTextHandlers[column.Name](column, itemIndex, item);
        }

        private static void InitializeGetTextHandlers()
        {
            getTextHandlers.Add(
                BankReportTraits.ColumnName.Comment,
                (Column column, int itemIndex, Instrument item) => item.Comment);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.EnvelopeAttack,
                (Column column, int itemIndex, Instrument item) => item.Envelope.Attack.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.EnvelopeDecay,
                (Column column, int itemIndex, Instrument item) => item.Envelope.Decay.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.EnvelopeHold,
                (Column column, int itemIndex, Instrument item) => item.Envelope.Hold.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.EnvelopeRelease,
                (Column column, int itemIndex, Instrument item) => item.Envelope.Release.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.EnvelopeSustain,
                (Column column, int itemIndex, Instrument item) => item.Envelope.Sustain.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.FileName,
                GetFileName);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.ItemIndex,
                (Column column, int itemIndex, Instrument item) => (itemIndex < 0) ? "-" : itemIndex.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.Name,
                (Column column, int itemIndex, Instrument item) => item.Name);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.OriginalKey,
                GetOriginalKey);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.PitchSemitones,
                (Column column, int itemIndex, Instrument item) => item.PitchSemitones.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.PitchCents,
                (Column column, int itemIndex, Instrument item) => item.PitchCents.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.ProgramNo,
                (Column column, int itemIndex, Instrument item) => item.ProgramNo.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.Volume,
                (Column column, int itemIndex, Instrument item) => item.Volume.ToString());

            getTextHandlers.Add(
                BankReportTraits.ColumnName.WaveEncoding,
                GetWaveEncoding);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.WaveTime,
                GetWaveTime);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.WaveTick,
                GetWaveTick);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.SampleRate,
                GetSampleRate);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.WaveBitRate,
                GetWaveBitRate);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.WaveSampleBit,
                GetWaveSampleBit);

            getTextHandlers.Add(
                BankReportTraits.ColumnName.InterpolationType,
                GetInterpolationType);
        }

        /// <summary>
        /// 指定したアイテムのファイル名を取得します。
        /// </summary>
        /// <param name="column">対象カラムを指定します。</param>
        /// <param name="itemIndex">対象アイテムのインデックスを指定します。</param>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>取得したアイテムのファイル名を返します。</returns>
        private static string GetFileName(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            string filePath = null;

            foreach (KeyRegion keyRegion in item.Children)
            {
                foreach (VelocityRegion velocityRegion in keyRegion.Children)
                {
                    if (filePath != null && filePath != velocityRegion.FilePath)
                    {
                        return "*";
                    }

                    filePath = velocityRegion.FilePath;
                }
            }

            return (filePath == null) ? string.Empty : Path.GetFileName(filePath);
        }

        /// <summary>
        /// 指定したアイテムのオリジナルキーを取得します。
        /// </summary>
        /// <param name="column">対象カラムを指定します。</param>
        /// <param name="itemIndex">対象アイテムのインデックスを指定します。</param>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>取得したアイテムのオリジナルキーを返します。</returns>
        private static string GetOriginalKey(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            string filePath = GetFileName(column, itemIndex, item);
            if (filePath == "*")
            {
                return "*";
            }

            VelocityRegion target = GetTargetVelocityRegion(item);

            return (target == null) ? string.Empty : KeyLabels[target.OriginalKey];
        }

        /// <summary>
        /// 指定したアイテムのエンコーディングを取得します。
        /// </summary>
        /// <param name="column">対象カラムを指定します。</param>
        /// <param name="itemIndex">対象アイテムのインデックスを指定します。</param>
        /// <param name="item">対象アイテムを指定します。</param>
        /// <returns>取得したアイテムのエンコーディングを返します。</returns>
        private static string GetWaveEncoding(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            string filePath = GetFileName(column, itemIndex, item);
            if (filePath == "*")
            {
                return "*";
            }

            VelocityRegion target = GetTargetVelocityRegion(item);

            return (target == null) ? string.Empty : target.Encoding.ToText();
        }

        /// <summary>
        ///
        /// </summary>
        private static WaveFile GetWaveFile(Instrument instrument)
        {
            VelocityRegion target = GetSingleTargetVelocityRegion(instrument);
            if (target == null)
            {
                return null;
            }

            if (target.WaveFile == null)
            {
                using (WaveFileReader reader = WaveFileReader.CreateInstance(target.FilePath))
                {
                    target.WaveFile = reader.Open(target.FilePath);
                }
            }
            return target.WaveFile;
        }

        /// <summary>
        ///
        /// </summary>
        private static string GetWaveTime(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            WaveFile waveFile = GetWaveFile(item);
            return waveFile != null ? WaveFileUtility.GetWaveTimeStringWithLoop(waveFile) : "*";
        }

        /// <summary>
        ///
        /// </summary>
        private static string GetWaveTick(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            WaveFile waveFile = GetWaveFile(item);
            return waveFile != null ? WaveFileUtility.GetWaveTickString(waveFile) : "*";
        }

        /// <summary>
        ///
        /// </summary>
        private static string GetSampleRate(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            WaveFile waveFile = GetWaveFile(item);
            return waveFile != null ? WaveFileUtility.GetSampleRateString(waveFile) : "*";
        }

        /// <summary>
        ///
        /// </summary>
        private static string GetWaveBitRate(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            WaveFile waveFile = GetWaveFile(item);
            return waveFile != null ? WaveFileUtility.GetWaveBitRateString(waveFile) : "*";
        }

        /// <summary>
        ///
        /// </summary>
        private static string GetWaveSampleBit(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            WaveFile waveFile = GetWaveFile(item);
            return waveFile != null ? WaveFileUtility.GetWaveSampleBitString(waveFile) : "*";
        }

        /// <summary>
        ///
        /// </summary>
        private static string GetInterpolationType(Column column, int itemIndex, Instrument item)
        {
            Assertion.Argument.NotNull(column);
            Assertion.Argument.NotNull(item);

            VelocityRegion target = GetSingleTargetVelocityRegion(item);
            if (target == null)
            {
                return "*";
            }

            switch (target.InterpolationType)
            {
                case InterpolationType.Polyphase:
                    return MessageResource.Label_InterpolationType_Polyphase;

                case InterpolationType.Linear:
                    return MessageResource.Label_InterpolationType_Linear;

                case InterpolationType.None:
                    return MessageResource.Label_InterpolationType_None;

            }

            Debug.Assert(false, "An unexpected value");
            return String.Empty;
        }

        private static string GetWaveTimeString(int waveTime)
        {
            double timed = 0.0;
            long timel = (long)(timed * 10000000); // 秒を100ナノ秒単位に変換
            TimeSpan timeSpan = new TimeSpan((long)timel);
            string times = string.Format(
                                         "{0}:{1:00}.{2:00}",
                                         timeSpan.Minutes,
                                         timeSpan.Seconds,
                                         timeSpan.Milliseconds / 10
                                         );
            return times;
        }

        private static string GetWaveTimeString(string filePath)
        {
            double timed = 0.0;

            if (TryGetWaveTime(filePath, ref timed) == false)
            {
                return "N/A";
            }

            long timel = (long)(timed * 10000000); // 秒を100ナノ秒単位に変換
            TimeSpan timeSpan = new TimeSpan((long)timel);
            string times = string.Format(
                                         "{0}:{1:00}.{2:00}",
                                         timeSpan.Minutes,
                                         timeSpan.Seconds,
                                         timeSpan.Milliseconds / 10
                                         );
            return times;
        }

        private static string GetWaveTickString(string filePath)
        {
            double timed = 0.0;
            int smfConvertTimeBase = 48; // 48,四分音符分解能デフォルト値

            if (TryGetWaveTime(filePath, ref timed) == false)
            {
                return "N/A";
            }

            long tick = (long)(timed * (120 / 60) * smfConvertTimeBase);
            return tick.ToString();
        }

        private static bool TryGetWaveTime(string filePath, ref double result)
        {
            if (File.Exists(filePath) == false)
            {
                return false;
            }

            using (WaveFileReader reader = WaveFileReader.CreateInstance(filePath))
            {
                WaveFile waveFile = reader.Open(filePath);
                double timed = ((double)waveFile.FrameCount) / waveFile.SampleRate;
                timed += 0.005; // 下３桁目を四捨五入

                result = timed;
                return true;
            }
        }

        private static VelocityRegion GetSingleTargetVelocityRegion(Instrument item)
        {
            if (item.Children.Count == 1 &&
                item.Children[0].Children.Count == 1)
            {
                return item.Children[0].Children[0] as VelocityRegion;
            }
            return null;
        }

        /// <summary>
        /// 情報の表示対象となる最初のベロシティリージョンを取得します。
        /// </summary>
        /// <param name="item">インストルメントを指定します。</param>
        /// <returns>
        /// 対象のベロシティリージョンを返します。
        /// ベロシティリージョンが１つもない場合は、nullを返します。
        /// </returns>
        private static VelocityRegion GetTargetVelocityRegion(Instrument item)
        {
            Assertion.Argument.NotNull(item);

            foreach (KeyRegion keyRegion in item.Children)
            {
                foreach (VelocityRegion velocityRegion in keyRegion.Children)
                {
                    return velocityRegion;
                }
            }

            return null;
        }
    }
}
