﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;

namespace NintendoWare.SoundFoundation.Commands
{
    /// <summary>
    /// キーバインディングを管理します。
    /// </summary>
    public class KeyBindingManager : IEnumerable<KeyBinding>, IEnumerable
    {
        // ディクショナリ
        private Dictionary<KeyStroke, KeyBindingMap> _keyBindingMaps = new Dictionary<KeyStroke, KeyBindingMap>();

        /// <summary>
        /// キーストロークの IEnumerable を取得します。
        /// </summary>
        public IEnumerable<KeyStroke> FirstKeys
        {
            get { return _keyBindingMaps.Keys; }
        }

        /// <summary>
        /// キーストロークからキーバインディングマップを取得します。
        /// </summary>
        /// <param name="key">キー。</param>
        /// <returns>キーバインディングマップ。</returns>
        public IEnumerable<KeyBinding> this[KeyStroke firstKey]
        {
            get { return FindKeyBindings(firstKey); }
        }

        /// <summary>
        /// キーストロークとコマンドの関連付けを追加します。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <param name="secondKey">第２キー。</param>
        /// <param name="command">コマンド。</param>
        public void Add(KeyStroke firstKey, KeyStroke secondKey, Command command)
        {
            if (KeyStroke.NullKeyStroke == secondKey)
            {
                throw new ArgumentException("secondKey must not be NullKeyStroke", "secondKey");
            }

            if (KeyStroke.NullKeyStroke == firstKey && ContainsKey(secondKey))
            {
                throw new ArgumentException("secondKey must not be registerd as firstKey", "firstKey");
            }

            if (!ContainsKey(firstKey))
            {
                _keyBindingMaps.Add(firstKey, new KeyBindingMap());
            }

            KeyBindingMap targetKeyBindingMap = _keyBindingMaps[firstKey];

            if (!targetKeyBindingMap.ContainsKey(secondKey))
            {
                targetKeyBindingMap.Add(secondKey, command);
            }
            else
            {
                targetKeyBindingMap[secondKey] = command;
            }
        }

        /// <summary>
        /// 指定キーバインディングを削除します。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <param name="secondKey">第２キー。</param>
        /// <returns>成功した場合は true、失敗した場合は false。</returns>
        public bool Remove(KeyStroke firstKey, KeyStroke secondKey)
        {
            if (!ContainsKey(firstKey, secondKey)) { return false; }
            return _keyBindingMaps[firstKey].Remove(secondKey);
        }

        /// <summary>
        /// 指定キーバインディングマップを削除します。
        /// </summary>
        /// <param name="firstKey">削除する第１キー。</param>
        /// <returns>成功した場合は true、失敗した場合は false。</returns>
        public bool RemoveKeyBindingMap(KeyStroke firstKey)
        {
            return _keyBindingMaps.Remove(firstKey);
        }

        /// <summary>
        /// 第１キーが存在するかどうか調べます。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <returns>存在する場合は true、存在しない場合は false。</returns>
        public bool ContainsKey(KeyStroke firstKey)
        {
            return _keyBindingMaps.ContainsKey(firstKey);
        }

        /// <summary>
        /// キーバインディングが存在するかどうか調べます。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <param name="secondKey">第２キー。</param>
        /// <returns>存在する場合は true、存在しない場合は false。</returns>
        public bool ContainsKey(KeyStroke firstKey, KeyStroke secondKey)
        {
            if (!ContainsKey(firstKey)) { return false; }
            return _keyBindingMaps[firstKey].ContainsKey(secondKey);
        }

        /// <summary>
        /// 指定キーに関連付けられたコマンドを検索します。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <param name="secondKey">第２キー。</param>
        /// <returns>コマンド。</returns>
        public Command Find(KeyStroke firstKey, KeyStroke secondKey)
        {
            if (!ContainsKey(firstKey, secondKey)) { return null; }
            return _keyBindingMaps[firstKey][secondKey];
        }

        /// <summary>
        /// 第１キーの後に続くキーバインディングの IEnumerable を取得します。
        /// </summary>
        /// <param name="firstKey">第１キー。</param>
        /// <returns>キーバインディングの IEnumerable。</returns>
        public IEnumerable<KeyBinding> FindKeyBindings(KeyStroke firstKey)
        {
            KeyBindingMap keyBindingMap = _keyBindingMaps[firstKey];

            foreach (KeyStroke secondKey in keyBindingMap.Keys)
            {

                yield return new KeyBinding(new KeyStroke[] { firstKey, secondKey },
                                             keyBindingMap[secondKey]);

            }
        }

        /// <summary>
        /// キーストロークとキーバインディングマップの IEnumerator を取得します。
        /// </summary>
        /// <returns>キーストロークとキーバインディングマップの IEnumerator。</returns>
        public IEnumerator<KeyBinding> GetEnumerator()
        {
            return FindKeyBindings().GetEnumerator();
        }

        /// <summary>
        /// キーストロークとキーバインディングマップの IEnumerator を取得します。
        /// </summary>
        /// <returns>キーストロークとキーバインディングマップの IEnumerator。</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return FindKeyBindings().GetEnumerator();
        }

        private IEnumerable<KeyBinding> FindKeyBindings()
        {
            foreach (KeyStroke firstKey in FirstKeys)
            {

                foreach (KeyBinding keyBinding in FindKeyBindings(firstKey))
                {
                    yield return keyBinding;
                }

            }
        }
    }
}
