﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using NintendoWare.SoundFoundation.Commands;

namespace NintendoWare.SoundFoundation.CommandHandlers
{
    public abstract class CommandHandler
    {
        private IQueryCommandParameter _extendedQueryParameter;

        public CommandHandler() : this(null) { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="extendedQueryParameter">パラメータ拡張。</param>
        public CommandHandler(IQueryCommandParameter extendedQueryParameter)
        {
            _extendedQueryParameter = extendedQueryParameter;
        }

        public abstract string TargetCommandID { get; }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        public CommandStatus QueryStatus(Command command)
        {
            ParametersProxy proxy = new ParametersProxy();
            proxy.QueryParameters.Add(command.Parameters);

            if (null != _extendedQueryParameter)
            {
                proxy.QueryParameters.Add(_extendedQueryParameter);
            }

            return QueryStatusInternal(command, proxy);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        public bool Execute(Command command)
        {
            ParametersProxy proxy = new ParametersProxy();
            proxy.QueryParameters.Add(command.Parameters);

            if (null != _extendedQueryParameter)
            {
                proxy.QueryParameters.Add(_extendedQueryParameter);
            }

            return ExecuteInternal(command, proxy);
        }

        protected abstract CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters);

        protected abstract bool ExecuteInternal(Command command, IQueryCommandParameter parameters);

        private class ParametersProxy : IQueryCommandParameter
        {
            private HashSet<IQueryCommandParameter> _queryParameters = new HashSet<IQueryCommandParameter>();

            public ICollection<IQueryCommandParameter> QueryParameters
            {
                get { return _queryParameters; }
            }

            /// <summary>
            /// 指定された名前のパラメータを取得します。
            /// </summary>
            /// <param name="name">パラメータの名前。</param>
            /// <returns>指定キーに関連付けられた値。</returns>
            public object this[string name]
            {
                get
                {
                    foreach (IQueryCommandParameter queryParameter in _queryParameters)
                    {
                        if (!queryParameter.ContainsParameter(name)) { continue; }
                        return queryParameter.GetParameter(name);
                    }

                    throw new KeyNotFoundException();
                }
            }

            /// <summary>
            /// 指定された名前がコレクションに含まれているかどうかを調べます。
            /// </summary>
            /// <param name="name">パラメータの名前。</param>
            /// <returns>含まれている場合は true、含まれていない場合は false。</returns>
            public bool ContainsKey(string name)
            {
                foreach (IQueryCommandParameter queryParameter in _queryParameters)
                {
                    if (queryParameter.ContainsParameter(name))
                    {
                        return true;
                    }
                }

                return false;
            }

            /// <summary>
            /// 指定パラメータを取得できるかどうか調べます。
            /// </summary>
            /// <param name="command">対象コマンド。</param>
            /// <param name="parameterName">パラメータ名。</param>
            /// <returns>取得できる場合は true、できない場合は false。</returns>
            bool IQueryCommandParameter.ContainsParameter(string parameterName)
            {
                return ContainsKey(parameterName);
            }

            /// <summary>
            /// 指定パラメータを取得します。
            /// </summary>
            /// <param name="command">対象コマンド。</param>
            /// <param name="parameterName">パラメータ名。</param>
            /// <returns>コマンドのパラメータ。</returns>
            object IQueryCommandParameter.GetParameter(string parameterName)
            {
                return this[parameterName];
            }
        }
    }
}
