﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Codecs
{
    using System;
    using System.IO;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// パケットを処理するフィルタの基本クラスです。
    /// </summary>
    public abstract class Filter
    {
        private InputConnectionPoint[] inputs = new InputConnectionPoint[0];
        private OutputConnectionPoint[] outputs = new OutputConnectionPoint[0];

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        protected Filter()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        protected Filter(InputConnectionPoint input)
            : this(new InputConnectionPoint[] { input })
        {
            Ensure.Argument.NotNull(input);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="inputs">入力接続ポイントを指定します。</param>
        protected Filter(InputConnectionPoint[] inputs)
        {
            Ensure.Argument.NotNull(inputs);
            this.SetInputs(inputs);
        }

        /// <summary>
        /// 入力接続ポイントの配列を取得します。
        /// </summary>
        public InputConnectionPoint[] Inputs
        {
            get { return this.inputs; }
        }

        /// <summary>
        /// 出力接続ポイントの配列を取得します。
        /// </summary>
        public OutputConnectionPoint[] Outputs
        {
            get { return this.outputs; }
        }

        /// <summary>
        /// 入力接続ポイントが全て接続済みかどうかを調べます。
        /// </summary>
        public bool IsInputsConnected
        {
            get
            {
                foreach (InputConnectionPoint point in this.Inputs)
                {
                    if (!point.IsConnected)
                    {
                        return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// 出力接続ポイントが全て接続済みかどうかを調べます。
        /// </summary>
        public bool IsOutputsConnected
        {
            get
            {
                foreach (OutputConnectionPoint point in this.Outputs)
                {
                    if (!point.IsConnected)
                    {
                        return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// ストリーム終端に達したかどうかを調べます。
        /// </summary>
        public virtual bool IsEndOfStream
        {
            get
            {
                foreach (InputConnectionPoint input in this.inputs)
                {
                    if (!input.IsEndOfStream)
                    {
                        return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// 入力接続ポイントを設定します。
        /// </summary>
        /// <returns>入力接続ポイントの配列を返します。</returns>
        protected void SetInputs(InputConnectionPoint[] inputs)
        {
            Assertion.Argument.NotNull(inputs);
            Assertion.Operation.True(this.inputs.Length == 0);

            this.inputs = inputs;

            foreach (InputConnectionPoint input in inputs)
            {
                input.Connected += this.OnInputPointConnected;
            }
        }

        /// <summary>
        /// 出力接続ポイントを設定します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected void SetOutputs(OutputConnectionPoint[] outputs)
        {
            Assertion.Argument.NotNull(outputs);
            this.outputs = outputs;
        }

        /// <summary>
        /// 入力接続ポイントを検証します。
        /// </summary>
        /// <param name="inputs">入力接続ポイントの配列を指定します。</param>
        protected virtual void ValidateInputs(InputConnectionPoint[] inputs)
        {
        }

        /// <summary>
        /// 出力接続ポイントを生成します。
        /// </summary>
        /// <returns>出力接続ポイントの配列を返します。</returns>
        protected virtual OutputConnectionPoint[] CreateOutputs()
        {
            return new OutputConnectionPoint[0];
        }

        /// <summary>
        /// パケットを処理します。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        /// <param name="packet">パケットを指定します。</param>
        protected virtual void Process(InputConnectionPoint input, Packet packet)
        {
        }

        /// <summary>
        /// 単一の入力ストリームが終端に達すると実行されます。
        /// </summary>
        /// <param name="input">入力接続ポイントを指定します。</param>
        protected virtual void ProcessEndOfStream(InputConnectionPoint input)
        {
        }

        /// <summary>
        /// 全ての入力ストリームが終端に達した場合に実行されます。
        /// </summary>
        protected virtual void ProcessEndOfAllStreams()
        {
            this.SendToAll(Packet.CreateEndOfStreamPacket());
        }

        /// <summary>
        /// パケットを全ての出力に送信します。
        /// </summary>
        /// <param name="packet">パケットを指定します。</param>
        protected void SendToAll(Packet packet)
        {
            Assertion.Argument.NotNull(packet);

            foreach (OutputConnectionPoint output in this.Outputs)
            {
                if (!output.IsConnected) { continue; }
                output.Send(packet);
            }
        }

        /// <summary>
        /// 入力接続ポイントが接続されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントデータを指定します。</param>
        private void OnInputPointConnected(object sender, EventArgs e)
        {
            // 入力接続ポイントが全て接続済みになったら初期化処理を行います。
            if (!this.IsInputsConnected)
            {
                return;
            }

            this.ValidateInputs(this.Inputs);

            this.outputs = this.CreateOutputs();

            foreach (InputConnectionPoint input in this.inputs)
            {
                input.PacketReceived -= this.OnPacketReceived;
                input.PacketReceived += this.OnPacketReceived;
            }
        }

        /// <summary>
        /// パケットを受信すると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントデータを指定します。</param>
        private void OnPacketReceived(object sender, PacketEventArgs e)
        {
            if (e.Packet.IsEndOfStream)
            {
                this.ProcessEndOfStream(sender as InputConnectionPoint);

                if (this.IsEndOfStream)
                {
                    this.ProcessEndOfAllStreams();
                }
            }
            else
            {
                this.Process(sender as InputConnectionPoint, e.Packet);
            }
        }
    }
}
