﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Binarization
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using ToolDevelopmentKit;

    public class ObjectReferenceResolver
    {
        private DomWriterContext context;

        private Dictionary<string, Stack<DomObject>> tagIndex = new Dictionary<string, Stack<DomObject>>();
        private Dictionary<object, ObjectReference> objectReferences = new Dictionary<object, ObjectReference>();
        private Dictionary<DomObject, ObjectReference> domObjectReferences = new Dictionary<DomObject, ObjectReference>();
        private HashSet<ObjectReference> endObjectReferences = new HashSet<ObjectReference>();
        private HashSet<ObjectReference> endDomObjectReferences = new HashSet<ObjectReference>();
        private HashSet<AddressWriter> addressCommiters = new HashSet<AddressWriter>();

        public ObjectReferenceResolver(DomWriterContext context)
        {
            if (null == context) { throw new ArgumentNullException("context"); }

            context.ObjectProcessing += OnObjectProcessing;
            context.ObjectProcessed += OnObjectProcessed;
            context.ProcessCompleted += OnProcessCompleted;

            this.context = context;
        }

        public DomObject GetDomObject(string tag)
        {
            Ensure.Argument.NotNull(tag);

            if (this.tagIndex.ContainsKey(tag))
            {
                Stack<DomObject> stack = this.tagIndex[tag];
                if (stack.Count == 0)
                {
                    throw new Exception("internal error");
                }

                return this.tagIndex[tag].Peek();
            }

            return null;
        }

        public ObjectReference GetObjectReference(object value)
        {
            if (null == value) { throw new ArgumentNullException("value"); }

            if (!this.objectReferences.ContainsKey(value))
            {
                this.objectReferences.Add(
                    value,
                    new ObjectReference()
                    {
                        Object = value
                    });
            }

            return this.objectReferences[value];
        }

        public ObjectReference GetDomObjectReference(DomObject domObject)
        {
            if (null == domObject) { throw new ArgumentNullException("domObject"); }

            if (!this.domObjectReferences.ContainsKey(domObject))
            {
                this.domObjectReferences.Add(
                    domObject,
                    new ObjectReference()
                    {
                        Object = domObject
                    });
            }

            return this.domObjectReferences[domObject];
        }

        public void RegisterAddressCommiter(AddressWriter commiter)
        {
            if (null == commiter) { throw new ArgumentNullException("commiter"); }
            this.addressCommiters.Add(commiter);
        }

        private void ElementProcessing(DomElement element)
        {
            Debug.Assert(null != element, "invalid argument");

            // ルート要素の場合は、コンテキスト情報をクリアします。
            if (null == element.Parent)
            {
                Clear();
            }

            RegisterTag(element.Tags, element);

            RegisterObjectStartAddress(element.Value, (ulong)this.context.Writer.Position);
            RegisterDomObjectStartAddress(element, (ulong)this.context.Writer.Position);
        }

        private void ElementProcessed(DomElement element)
        {
            Debug.Assert(null != element, "invalid argument");

            UnregisterTag(element.Tags);

            RegisterObjectEndAddress(element.Value, (ulong)this.context.Writer.Position);
            RegisterDomObjectEndAddress(element, (ulong)this.context.Writer.Position);
        }

        private void ElementFieldProcessing(DomElementField field)
        {
            Debug.Assert(null != field, "invalid argument");
            if (IsObjectReference(field)) { return; }

            RegisterTag(field.Tags, field);

            if (field.Value == null || field.Value.GetType().IsPrimitive)
            {
                return;
            }

            RegisterObjectStartAddress(field.Value, (ulong)this.context.Writer.Position);
            RegisterDomObjectStartAddress(field, (ulong)this.context.Writer.Position);
        }

        private void ElementFieldProcessed(DomElementField field)
        {
            Debug.Assert(null != field, "invalid argument");
            if (IsObjectReference(field)) { return; }

            UnregisterTag(field.Tags);

            if (field.Value == null || field.Value.GetType().IsPrimitive)
            {
                return;
            }

            RegisterObjectEndAddress(field.Value, (ulong)this.context.Writer.Position);
            RegisterDomObjectEndAddress(field, (ulong)this.context.Writer.Position);
        }

        private bool IsObjectReference(DomElementField field)
        {
            Debug.Assert(null != field, "invalid argument");
            return field.HasAttribute<DomObjectReferenceAttribute>();
        }

        private void RegisterTag(string[] tags, DomObject domObject)
        {
            Assertion.Argument.NotNull(domObject);
            Assertion.Argument.NotNull(tags);

            if (tags.Length == 0) { return; }

            foreach (string tag in tags)
            {
                Stack<DomObject> stack = null;
                this.tagIndex.TryGetValue(tag, out stack);

                if (stack == null)
                {
                    stack = new Stack<DomObject>();
                    this.tagIndex.Add(tag, stack);
                }
                else
                {
                    Assertion.Operation.True(stack.Count > 0);
                }

                stack.Push(domObject);
            }
        }

        private void UnregisterTag(string[] tags)
        {
            Assertion.Argument.NotNull(tags);

            if (tags.Length == 0) { return; }

            foreach (string tag in tags)
            {
                Stack<DomObject> stack = null;
                this.tagIndex.TryGetValue(tag, out stack);

                if (stack == null) { return; }

                if (stack.Count > 1)
                {
                    stack.Pop();
                }
                else
                {
                    this.tagIndex.Remove(tag);
                }
            }
        }

        private void RegisterObjectStartAddress(object value, ulong position)
        {
            Debug.Assert(null != value, "invalid argument");

            ObjectReference objectReference = this.GetObjectReference(value);
            objectReference.StartAddress = position;

            this.RegisterObjectNextAddress(position);
        }

        private void RegisterObjectEndAddress(object value, ulong position)
        {
            Debug.Assert(null != value, "invalid argument");

            ObjectReference objectReference = null;
            this.objectReferences.TryGetValue(value, out objectReference);

            if (objectReference == null)
            {
                throw new Exception("internal error.");
            }

            objectReference.EndAddress = position;

            if (!this.endObjectReferences.Contains(objectReference))
            {
                this.endObjectReferences.Add(objectReference);
            }
        }

        private void RegisterObjectNextAddress(ulong position)
        {
            foreach (ObjectReference endObjectReference in this.endObjectReferences)
            {
                endObjectReference.NextAddress = position;
            }

            this.endObjectReferences.Clear();
        }

        private void RegisterDomObjectStartAddress(DomObject domObject, ulong position)
        {
            Debug.Assert(null != domObject, "invalid argument");

            ObjectReference reference = this.GetDomObjectReference(domObject);
            reference.StartAddress = position;

            this.RegisterDomObjectNextAddress(position);
        }

        private void RegisterDomObjectEndAddress(DomObject domObject, ulong position)
        {
            Debug.Assert(null != domObject, "invalid argument");

            ObjectReference objectReference = null;
            this.domObjectReferences.TryGetValue(domObject, out objectReference);

            if (objectReference == null)
            {
                throw new Exception("internal error.");
            }

            objectReference.EndAddress = position;

            if (!this.endDomObjectReferences.Contains(objectReference))
            {
                this.endDomObjectReferences.Add(objectReference);
            }
        }

        private void RegisterDomObjectNextAddress(ulong position)
        {
            foreach (ObjectReference endDomObjectReference in this.endDomObjectReferences)
            {
                endDomObjectReference.NextAddress = position;
            }

            this.endDomObjectReferences.Clear();
        }

        private void CommitAddresses()
        {
            foreach (AddressWriter commiter in this.addressCommiters)
            {
                commiter.Commit(this.context);
            }
        }

        private void Clear()
        {
            this.tagIndex.Clear();
            this.objectReferences.Clear();
        }

        private void OnObjectProcessing(object sender, DomObjectEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (e.Object is DomElement)
            {
                ElementProcessing(e.Object as DomElement);
                return;
            }

            if (e.Object is DomElementField)
            {
                ElementFieldProcessing(e.Object as DomElementField);
                return;
            }

            throw new Exception("internal error");
        }

        private void OnObjectProcessed(object sender, DomObjectEventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (e.Object is DomElement)
            {
                ElementProcessed(e.Object as DomElement);
                return;
            }

            if (e.Object is DomElementField)
            {
                ElementFieldProcessed(e.Object as DomElementField);
                return;
            }

            throw new Exception("internal error");
        }

        private void OnProcessCompleted(object sender, EventArgs e)
        {
            this.RegisterObjectNextAddress((ulong)this.context.Writer.Position);
            this.RegisterDomObjectNextAddress((ulong)this.context.Writer.Position);

            this.CommitAddresses();
        }
    }
}
