﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.IO;
using NintendoWare.SoundAdditionalInfoConverter.Resources;

namespace NintendoWare.SoundAdditionalInfoConverter
{
    internal class ConsoleApplication
    {
        // 設定
        private Settings _settings = null;

        public bool Run(string[] args)
        {
            try
            {
                CommandLineParserSettings commandLineParserSettings = new CommandLineParserSettings()
                {
                    ApplicationDescription = Messages.CommandLine_ApplicationDescription,
                };
                if (new Nintendo.ToolFoundation.IO.CommandLineParser(commandLineParserSettings).ParseArgs(args, out this._settings) == false)
                {
                    return true;
                }
            }
            catch (Exception e)
            {
                Logs.Logger.GetInstance().OutputError(e.Message);
                return false;
            }

            Logs.Logger.GetInstance().IsSilent = _settings.IsSilent;

            // 出力ディレクトリを指定していない場合は入力ファイルと同じ場所に出す
            if (string.IsNullOrEmpty(this._settings.OutputDirectory))
            {
                string fullFilePath = Path.GetFullPath(this._settings.FilePath);
                this._settings.OutputDirectory = Path.GetDirectoryName(fullFilePath);
            }

            try
            {
                AdditionalInfoConverter converter = new AdditionalInfoConverter();
                if (!converter.Convert(this._settings.FilePath, this._settings.OutputDirectory))
                {
                    return false;
                }
            }
            catch (Exception e)
            {
                Logs.Logger.GetInstance().OutputError(e.Message);
                return false;
            }

            return true;
        }

        private class Settings
        {
            private string filePath = string.Empty;

            public Settings()
            {
                this.IsSilent = false;
                this.OutputDirectory = string.Empty;
            }

            [CommandLineValue(
                0,
                IsRequired = true,
                ValueName = "input-filepath",
                Description = nameof(Messages.CommandLine_InputDescription),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public string FilePath
            {
                get { return this.filePath; }
                set
                {
                    Ensure.Argument.NotNull(value);
                    this.filePath = value;
                }
            }

            [CommandLineOption(
                's',
                "silent",
                Description = nameof(Messages.CommandLine_UsageMessage_Option_s),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public bool IsSilent { get; set; }

            [CommandLineOption(
                'o',
                "output",
                ValueName = "output-dirpath",
                Description = nameof(Messages.CommandLine_UsageMessage_Option_output),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public string OutputDirectory { get; set; }

            /// <summary>
            /// ローカライズしたオプションの説明を取得します。
            /// </summary>
            /// <param name="description">Descriptionプロパティの値</param>
            /// <param name="valueName">引数名またはオプション名</param>
            /// <returns>ローカライズされたコマンドラインオプションの説明を返します。</returns>
            public static string LocalizeDescription(string description, string valueName)
            {
                string result = Messages.ResourceManager.GetString(description, Messages.Culture);
                System.Diagnostics.Debug.Assert(result != null, "コマンドオプションの Description が null");

                return result;
            }
        }
    }
}
