﻿namespace Opal.Plugins
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.ComponentModel.Composition;
    using System.ComponentModel.Composition.Hosting;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.ComponentModel;

    /// <summary>
    /// プラグイン管理クラスです。
    /// </summary>
    public sealed class PluginManager : DisposableObject
    {
        private readonly Dictionary<string, IPlugin> typeNameToPlugin =
            new Dictionary<string, IPlugin>();

        private readonly Assembly defaultAssembly;
        private readonly string[] pluginPaths;
        private readonly PluginImport pluginImport = new PluginImport();
        private CompositionContainer pluginContainer = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="defaultAssembly">プラグインがある標準アセンブリです。</param>
        /// <param name="pluginPaths">追加するプラグインのパスです。</param>
        public PluginManager(Assembly defaultAssembly, params string[] pluginPaths)
        {
            Debug.Assert(defaultAssembly != null);

            this.defaultAssembly = defaultAssembly;

            if (pluginPaths != null)
            {
                foreach (var path in pluginPaths)
                {
                    Debug.Assert(!string.IsNullOrWhiteSpace(path));
                }

                this.pluginPaths = pluginPaths.ToArray();
            }
        }

        /// <summary>
        /// プラグインを取得します。
        /// </summary>
        /// <typeparam name="T">プラグインのテンプレートの型です。</typeparam>
        /// <param name="typeName">プラグインタイプ名です。</param>
        /// <returns>指定した型でプラグインのインスタンスを返します。</returns>
        public T GetPlugin<T>(string typeName)
            where T : class, IPlugin
        {
            var plugin = this.typeNameToPlugin[typeName];
            return plugin as T;
        }

        /// <summary>
        /// プラグインを取得します。
        /// </summary>
        /// <typeparam name="T">プラグインのテンプレートの型です。</typeparam>
        /// <param name="typeName">プラグインタイプ名です。</param>
        /// <returns>指定した型でプラグインのインスタンスを返します。</returns>
        public IEnumerable<T> GetPlugins<T>()
            where T : class, IPlugin
        {
            foreach (var plugin in this.typeNameToPlugin.Values)
            {
                T p = plugin as T;
                if (p != null)
                {
                    yield return p;
                }
            }
        }

        /// <summary>
        /// プラグインを初期化します。
        /// </summary>
        /// <returns>非同期処理のタスクを返します。</returns>
        public async Task InitializePlugins()
        {
            List<CatalogExportProvider> providers = new List<CatalogExportProvider>();

            // デフォルト機能のプラグイン登録
            AssemblyCatalog defaultCatalog = new AssemblyCatalog(this.defaultAssembly);
            CatalogExportProvider defaultProvider = new CatalogExportProvider(defaultCatalog);

            providers.Add(defaultProvider);

            // 拡張機能のプラグイン登録
            if (this.pluginPaths != null)
            {
                foreach (var path in this.pluginPaths)
                {
                    // ディレクトリパスが有効でない場合はスキップ
                    if (!Directory.Exists(path))
                    {
                        continue;
                    }

                    DirectoryCatalog pluginCatalog = new DirectoryCatalog(path);
                    CatalogExportProvider pluginProvider = new CatalogExportProvider(pluginCatalog);

                    providers.Add(pluginProvider);
                }
            }

            this.pluginContainer = new CompositionContainer(providers.ToArray());

            foreach (var provider in providers)
            {
                provider.SourceProvider = this.pluginContainer;
            }

            await Task.Run(() =>
            {
                this.pluginContainer.ComposeParts(this.pluginImport);
            });

            await this.InitializePluginAsync();
        }

        /// <summary>
        /// プラグインを非同期で初期化します。
        /// </summary>
        /// <returns>タスクのインスタンスを返します。</returns>
        public Task InitializePluginAsync()
        {
            return Task.Run(() =>
            {
                foreach (var plugin in this.pluginImport.Plugins)
                {
                    string fullName = plugin.Value.GetType().FullName;

                    if (this.typeNameToPlugin.ContainsKey(fullName))
                    {
                        this.typeNameToPlugin[fullName] = plugin.Value;
                    }
                    else
                    {
                            this.typeNameToPlugin.Add(fullName, plugin.Value);
                    }
                }
            });
        }

        /// <summary>
        /// 廃棄します。内部処理です。
        /// </summary>
        protected sealed override void DisposeInternal()
        {
            this.pluginContainer.Dispose();
        }
    }
}
