﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Linq.Expressions;
    using System.Reflection;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Resources;
    using Opal.Utilities;

    /// <summary>
    /// テンプレートプロパティ編集オペレーションです。
    /// </summary>
    /// <typeparam name="TTarget">編集対象となるプロパティを保持するテンプレート型です。</typeparam>
    /// <typeparam name="TObject">編集対象となるプロパティのテンプレート型です。</typeparam>
    public sealed class PropertyEditOperation<TTarget, TObject> : Operation where TTarget : class
    {
        private readonly Tuple<TTarget, PropertyInfo> targetAndProperty;
        private readonly int mergeableHashCode;
        private TObject data;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="propertyName">編集対象名です。</param>
        /// <param name="data">設定するデータです。</param>
        public PropertyEditOperation(
            TTarget target,
            string propertyName,
            TObject data)
            : this(target, target.GetType().GetProperty(propertyName), data)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="propertyInfo">編集対象の PropertyInfo です。</param>
        /// <param name="data">設定するデータです。</param>
        public PropertyEditOperation(
            TTarget target,
            PropertyInfo propertyInfo,
            TObject data)
        {
            Debug.Assert(target != null);
            Debug.Assert(propertyInfo != null);

            this.targetAndProperty =
                Tuple.Create(target, propertyInfo);
            this.mergeableHashCode = this.targetAndProperty.GetHashCode();
            this.data = data;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="propertyExpression">対象プロパティを示す式です。</param>
        /// <param name="data">設定するデータです。</param>
        public PropertyEditOperation(
            TTarget target,
            Expression<Func<TTarget, TObject>> propertyExpression,
            TObject data)
        {
            Debug.Assert(target != null);
            Debug.Assert(propertyExpression != null);

            var memberExpression = propertyExpression.Body as MemberExpression;
            Debug.Assert(memberExpression != null);

            var propertyInfo =
                target.GetType().GetProperty(memberExpression.Member.Name);

            this.targetAndProperty =
                Tuple.Create(target, propertyInfo);
            this.mergeableHashCode = this.targetAndProperty.GetHashCode();
            this.data = data;
        }

        /// <summary>
        /// マージ可能か判定します。
        /// </summary>
        public override bool IsMergeable
        {
            get
            {
                return true;
            }
        }

        /// <summary>
        /// マージ対象であるか判定するためのハッシュコードを取得します。
        /// </summary>
        /// <returns>マージ対象判定用のハッシュコードを返します。</returns>
        public override int GetMergeableHashCode()
        {
            return this.mergeableHashCode;
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>実行時に返される Operation です。</returns>
        public override Operation Execute()
        {
            TTarget target = this.targetAndProperty.Item1;

            TObject temp = (TObject)this.targetAndProperty.Item2.GetValue(target);
            this.targetAndProperty.Item2.SetValue(target, this.data, null);
            this.data = temp;

            return this;
        }
    }
}
