﻿namespace Opal.Logs
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Runtime.CompilerServices;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// ログクラスです。
    /// </summary>
    public sealed class Log
    {
        private readonly LogLevel level;
        private readonly string message;
        private readonly string exceptionMessage = string.Empty;
        private readonly Action associatedAction = null;

#if DEBUG
        private readonly string fileName;
        private readonly int fileLine;
        private readonly string memberName;
#endif

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="level">ログレベルです。</param>
        /// <param name="message">ログに登録するメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        internal Log(LogLevel level, string message, string fileName = null, int fileLine = 0, string memberName = null)
        {
            this.level = level;
            this.message = message;
#if DEBUG
            this.fileName = fileName;
            this.fileLine = fileLine;
            this.memberName = memberName;
#endif
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="level">ログレベルです。</param>
        /// <param name="message">ログに登録するメッセージです。</param>
        /// <param name="exception">例外処理のインスタンスです。ログに例外時の情報を入力したい場合に使用します。</param>
        /// <param name="associatedAction">ログに関連付ける処理です。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        internal Log(LogLevel level, string message, Exception exception, Action associatedAction, string fileName = null, int fileLine = 0, string memberName = null)
            : this(level, message, fileName, fileLine, memberName)
        {
            if (exception != null)
            {
                if (exception.InnerException != null)
                {
                    this.exceptionMessage += exception.InnerException.ToString() + "\n";
                }

                this.exceptionMessage += exception.ToString();
            }

            this.associatedAction = associatedAction;
        }

        /// <summary>
        /// ログレベルを取得します。
        /// </summary>
        public LogLevel Level
        {
            get
            {
                return this.level;
            }
        }

        /// <summary>
        /// メッセージを取得します。
        /// </summary>
        public string Message
        {
            get
            {
                return this.message;
            }
        }

        /// <summary>
        /// 例外メッセージを取得します。
        /// </summary>
        public string ExceptionMessage
        {
            get
            {
                return this.exceptionMessage;
            }
        }

#if DEBUG
        /// <summary>
        /// ファイル名を取得します。
        /// </summary>
        public string FileName
        {
            get
            {
                return this.fileName;
            }
        }

        /// <summary>
        /// ファイル行を取得します。
        /// </summary>
        public int FileLine
        {
            get
            {
                return this.fileLine;
            }
        }

        /// <summary>
        /// メンバー名を取得します。
        /// </summary>
        public string MemberName
        {
            get
            {
                return this.memberName;
            }
        }
#endif

        /// <summary>
        /// ログに関連付けられた処理を実行します。
        /// </summary>
        public void InvokeAssociatedAction()
        {
            if (this.associatedAction != null)
            {
                this.associatedAction.Invoke();
            }
        }
    }
}
