﻿namespace Opal.Events
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// 弱参照イベントハンドラクラスです。
    /// </summary>
    /// <typeparam name="TTarget">イベントハンドラ対象のテンプレートの型です。</typeparam>
    /// <typeparam name="TEventArgs">イベント引数のテンプレートの型です。</typeparam>
    public class WeakEventHandler<TTarget, TEventArgs> : IWeakEventHandler<TEventArgs>
        where TTarget : class
        where TEventArgs : EventArgs
    {
        private WeakReference<TTarget> targetRef;
        private OpenEventHandler openHandler;
        private EventHandler<TEventArgs> handler;
        private UnregisterCallback<TEventArgs> unregisterCallback;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="handler">イベントハンドラです。</param>
        /// <param name="unregisterCallback">イベントの登録を取り消すコールバックです。</param>
        public WeakEventHandler(EventHandler<TEventArgs> handler, UnregisterCallback<TEventArgs> unregisterCallback)
        {
            this.targetRef = new WeakReference<TTarget>((TTarget)handler.Target);
            this.openHandler = (OpenEventHandler)Delegate.CreateDelegate(typeof(OpenEventHandler), null, handler.Method);
            this.handler = this.Invoke;
            this.unregisterCallback = unregisterCallback;
        }

        private delegate void OpenEventHandler(TTarget @this, object sender, TEventArgs e);

        /// <summary>
        /// イベントハンドラを取得します。
        /// </summary>
        public EventHandler<TEventArgs> Handler
        {
            get
            {
                return this.handler;
            }
        }

        /// <summary>
        /// 弱参照イベントハンドラをイベントハンドラに暗黙的に変換するオペレータです。
        /// </summary>
        /// <param name="weakHandler">変換対象の弱参照イベントハンドラです。</param>
        /// <returns>イベントハンドラを返します。</returns>
        public static implicit operator EventHandler<TEventArgs>(WeakEventHandler<TTarget, TEventArgs> weakHandler)
        {
            return weakHandler.handler;
        }

        /// <summary>
        /// イベントを起動します。
        /// </summary>
        /// <param name="sender">送信元です。</param>
        /// <param name="e">イベント引数です。</param>
        public void Invoke(object sender, TEventArgs e)
        {
            TTarget target = null;
            if (this.targetRef.TryGetTarget(out target))
            {
                this.openHandler.Invoke(target, sender, e);
            }
            else if (this.unregisterCallback != null)
            {
                this.unregisterCallback(this.handler);
                this.unregisterCallback = null;
            }
        }
    }
}
