﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Variation : ObservableEntity<variationType>, IDeepCopyable<Variation>, IDeepCopyFrom<Variation>
    {
        private string @id = string.Empty;
        private readonly ShaderOptionChoice @choice = new ShaderOptionChoice();
        private string @default = string.Empty;
        private bool @branch;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Variation()
        {
            this.@choice.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Variation(Variation source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Variation(variationType source)
        {
            this.@id = source.@id;
            this.@choice = new ShaderOptionChoice(source.choice);
            this.@default = source.@default;
            this.@branch = source.@branch;
            this.@choice.PropertyChanged += this.OnPropertyChanged;
        }

        public string Id
        {
            get
            {
                return this.@id;
            }

            set
            {
                if (this.@id == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@id, value, () => this.CalcCRC());
            }
        }

        public ShaderOptionChoice Choice
        {
            get
            {
                return this.@choice;
            }
        }

        public string Default
        {
            get
            {
                return this.@default;
            }

            set
            {
                if (this.@default == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@default, value, () => this.CalcCRC());
            }
        }

        public bool Branch
        {
            get
            {
                return this.@branch;
            }

            set
            {
                if (this.@branch == value)
                {
                    return;
                }

                this.SetProperty(ref this.@branch, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override variationType CreateSerializableData()
        {
            var writeData = new variationType();
            writeData.@id = this.Id;
            writeData.@choice = this.@choice.CreateSerializableData() as String;
            writeData.@default = this.Default;
            writeData.@branch = this.Branch;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Variation IDeepCopyable<Variation>.DeepCopy()
        {
            return new Variation(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Variation source)
        {
            this.@id = source.@id;
            this.@choice.DeepCopyFrom(source.@choice);
            this.@default = source.@default;
            this.@branch = source.@branch;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Id)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Choice.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Default)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Branch)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@choice.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@choice.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@choice.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
