﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class UserDataStream : UserData, IStreamIndex, IDeepCopyable<UserDataStream>, IDeepCopyFrom<UserDataStream>
    {
        private int @streamIndex;
        private StreamByte @stream = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserDataStream()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public UserDataStream(UserDataStream source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public UserDataStream(user_dataType source)
            : base(source)
        {
            this.@streamIndex = (source.Item as user_streamType).stream_index;
        }

        public int Size
        {
            get
            {
                if (this.Stream == null)
                {
                    return 0;
                }
                else
                {
                    return this.Stream.Values.Count;
                }
            }
        }

        int IStreamIndex.StreamIndex
        {
            get
            {
                if (this.Stream != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                    if (rootEntity == null)
                    {
                        return -1;
                    }

                    var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                    return sourceProp.IndexOf(this.Stream);
                }
                return this.@streamIndex;
            }
        }

        public StreamByte Stream
        {
            get
            {
                return this.@stream;
            }

            set
            {
                if (this.@stream == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@stream, value, () => this.CalcCRC());
                this.RaisePropertyChanged("Size");
            }
        }

        public override UserDataType Type
        {
            get
            {
                return UserDataType.Stream;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override user_dataType CreateSerializableData()
        {
            var writeData = base.CreateSerializableData();
            if (writeData.Item == null)
            {
                writeData.Item = new user_streamType();
            }
            (writeData.Item as user_streamType).size = this.Size;
            (writeData.Item as user_streamType).stream_index = (this as IStreamIndex).StreamIndex;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        UserDataStream IDeepCopyable<UserDataStream>.DeepCopy()
        {
            return new UserDataStream(this);
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected override UserData DeepCopy()
        {
            return new UserDataStream(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(UserDataStream source)
        {
            base.DeepCopyFrom(source);
            this.@streamIndex = source.@streamIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32((this as IStreamIndex).StreamIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
