﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class TextureInfo : ObservableEntity<texture_infoType>, IChildEntity, IStreamIndex, IDeepCopyable<TextureInfo>, IDeepCopyFrom<TextureInfo>
    {
        private string @dccPreset = string.Empty;
        private string @hint = string.Empty;
        private readonly Bool4 @linear = new Bool4(false, false, false, false);
        private texture_info_dimensionType @dimension;
        private texture_info_quantize_typeType @quantizeType;
        private int @width = 1;
        private int @height = 1;
        private int @depth = 1;
        private int @mipLevel = 1;
        private string @mipGenFilter = string.Empty;
        private int @size = 1;
        private readonly ComponentSelector @compSel = new ComponentSelector(texture_info_comp_selValue.r, texture_info_comp_selValue.r, texture_info_comp_selValue.r, texture_info_comp_selValue.Item1);
        private bool @weightedCompress;
        private int @initialSwizzle;
        private readonly ObservableList<uint> @levelOffsets = new ObservableList<uint>();
        private string @originalImageHash = string.Empty;
        private int @streamIndex;
        private Texture parent = null;
        private Stream @stream = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public TextureInfo()
        {
            this.@linear.PropertyChanged += this.OnPropertyChanged;
            this.@compSel.PropertyChanged += this.OnPropertyChanged;
            this.@levelOffsets.Add(0);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public TextureInfo(TextureInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public TextureInfo(texture_infoType source)
        {
            this.@levelOffsets.Add(0);
            this.@dccPreset = source.@dcc_preset;
            this.@hint = source.@hint;
            Utility.CopyArray(source.linear, this.@linear.Values);
            this.@dimension = source.@dimension;
            this.@quantizeType = source.@quantize_type;
            this.@width = source.@width;
            this.@height = source.@height;
            this.@depth = source.@depth;
            this.@mipLevel = source.@mip_level;
            this.@mipGenFilter = source.@mip_gen_filter;
            this.@size = source.@size;
            Utility.CopyArray(source.comp_sel, this.@compSel.Values);
            this.@weightedCompress = source.@weighted_compress;
            this.@initialSwizzle = source.@initial_swizzle;
            if (source.@level_offset != null)
            {
                this.@levelOffsets.Clear();
                this.@levelOffsets.Add(source.@level_offset);
            }
            this.@originalImageHash = source.@original_image_hash;
            this.@streamIndex = source.@stream_index;
            this.@linear.PropertyChanged += this.OnPropertyChanged;
            this.@compSel.PropertyChanged += this.OnPropertyChanged;
        }

        public string DccPreset
        {
            get
            {
                return this.@dccPreset;
            }

            set
            {
                if (this.@dccPreset == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@dccPreset, value, () => this.CalcCRC());
            }
        }

        public string Hint
        {
            get
            {
                return this.@hint;
            }

            set
            {
                if (this.@hint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@hint, value, () => this.CalcCRC());
            }
        }

        public Bool4 Linear
        {
            get
            {
                return this.@linear;
            }
        }

        public texture_info_dimensionType Dimension
        {
            get
            {
                return this.@dimension;
            }

            set
            {
                if (this.@dimension == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dimension, value, () => this.CalcCRC());
            }
        }

        public texture_info_quantize_typeType QuantizeType
        {
            get
            {
                return this.@quantizeType;
            }

            set
            {
                if (this.@quantizeType == value)
                {
                    return;
                }

                this.SetProperty(ref this.@quantizeType, value, () => this.CalcCRC());
            }
        }

        public int Width
        {
            get
            {
                return this.@width;
            }

            set
            {
                if (this.@width == value)
                {
                    return;
                }

                this.SetProperty(ref this.@width, value, () => this.CalcCRC());
            }
        }

        public int Height
        {
            get
            {
                return this.@height;
            }

            set
            {
                if (this.@height == value)
                {
                    return;
                }

                this.SetProperty(ref this.@height, value, () => this.CalcCRC());
            }
        }

        public int Depth
        {
            get
            {
                return this.@depth;
            }

            set
            {
                if (this.@depth == value)
                {
                    return;
                }

                this.SetProperty(ref this.@depth, value, () => this.CalcCRC());
            }
        }

        public int MipLevel
        {
            get
            {
                return this.@mipLevel;
            }

            set
            {
                if (this.@mipLevel == value)
                {
                    return;
                }

                this.SetProperty(ref this.@mipLevel, value, () => this.CalcCRC());
            }
        }

        public string MipGenFilter
        {
            get
            {
                return this.@mipGenFilter;
            }

            set
            {
                if (this.@mipGenFilter == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@mipGenFilter, value, () => this.CalcCRC());
            }
        }

        public int Size
        {
            get
            {
                return this.@size;
            }

            set
            {
                if (this.@size == value)
                {
                    return;
                }

                this.SetProperty(ref this.@size, value, () => this.CalcCRC());
            }
        }

        public ComponentSelector CompSel
        {
            get
            {
                return this.@compSel;
            }
        }

        public bool WeightedCompress
        {
            get
            {
                return this.@weightedCompress;
            }

            set
            {
                if (this.@weightedCompress == value)
                {
                    return;
                }

                this.SetProperty(ref this.@weightedCompress, value, () => this.CalcCRC());
            }
        }

        public int InitialSwizzle
        {
            get
            {
                return this.@initialSwizzle;
            }

            set
            {
                if (this.@initialSwizzle == value)
                {
                    return;
                }

                this.SetProperty(ref this.@initialSwizzle, value, () => this.CalcCRC());
            }
        }

        public ObservableList<uint> LevelOffsets
        {
            get
            {
                return this.@levelOffsets;
            }
        }

        public string OriginalImageHash
        {
            get
            {
                return this.@originalImageHash;
            }

            set
            {
                if (this.@originalImageHash == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@originalImageHash, value, () => this.CalcCRC());
            }
        }

        int IStreamIndex.StreamIndex
        {
            get
            {
                if (this.Stream != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                    if (rootEntity == null)
                    {
                        return -1;
                    }

                    var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                    return sourceProp.IndexOf(this.Stream);
                }
                return this.@streamIndex;
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Texture : true);
                this.SetProperty(ref this.parent, value as Texture, () => this.CalcCRC());
            }
        }

        public Texture Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Texture);
            }
        }

        public Stream Stream
        {
            get
            {
                return this.@stream;
            }

            set
            {
                if (this.@stream == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@stream, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override texture_infoType CreateSerializableData()
        {
            var writeData = new texture_infoType();
            writeData.@dcc_preset = this.DccPreset;
            writeData.@hint = this.Hint;
            writeData.linear = new bool[4];
            Utility.CopyArray(this.Linear.Values, writeData.linear);
            writeData.@dimension = this.Dimension;
            writeData.@quantize_type = this.QuantizeType;
            writeData.@width = this.Width;
            writeData.@height = this.Height;
            writeData.@depth = this.Depth;
            writeData.@mip_level = this.MipLevel;
            writeData.@mip_gen_filter = this.MipGenFilter;
            writeData.@size = this.Size;
            writeData.comp_sel = new texture_info_comp_selValue[4];
            Utility.CopyArray(this.CompSel.Values, writeData.comp_sel);
            writeData.@weighted_compress = this.WeightedCompress;
            writeData.@initial_swizzle = this.InitialSwizzle;
            writeData.@level_offset = this.LevelOffsets.ToArray();
            writeData.@original_image_hash = this.OriginalImageHash;
            {
                IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                Ensure.Operation.NotNull(rootEntity);
                var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                // 読み取り高速化のために不要なデータを読み飛ばす場合があるので、参照先が存在しない場合を許容します
                Ensure.Operation.True((sourceProp.Count == 0) || (Stream != null));
                writeData.stream_index = (this as IStreamIndex).StreamIndex;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        TextureInfo IDeepCopyable<TextureInfo>.DeepCopy()
        {
            return new TextureInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(TextureInfo source)
        {
            this.@dccPreset = source.@dccPreset;
            this.@hint = source.@hint;
            this.@linear.DeepCopyFrom(source.@linear);
            this.@dimension = source.@dimension;
            this.@quantizeType = source.@quantizeType;
            this.@width = source.@width;
            this.@height = source.@height;
            this.@depth = source.@depth;
            this.@mipLevel = source.@mipLevel;
            this.@mipGenFilter = source.@mipGenFilter;
            this.@size = source.@size;
            this.@compSel.DeepCopyFrom(source.@compSel);
            this.@weightedCompress = source.@weightedCompress;
            this.@initialSwizzle = source.@initialSwizzle;
            CopyValues<uint>(source.@levelOffsets, this.@levelOffsets);
            this.@originalImageHash = source.@originalImageHash;
            this.@streamIndex = source.@streamIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccPreset)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Hint)));
            buffers.AddRange(BitConverter.GetBytes(this.Linear.HashValue));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Dimension)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.QuantizeType)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Width)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Height)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Depth)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MipLevel)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MipGenFilter)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Size)));
            buffers.AddRange(BitConverter.GetBytes(this.CompSel.HashValue));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.WeightedCompress)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.InitialSwizzle)));
            foreach (var elem in this.LevelOffsets)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OriginalImageHash)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32((this as IStreamIndex).StreamIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@linear.AutoCalc = this.AutoCalc;
            this.@compSel.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@linear.Reset();
            this.@compSel.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@linear.Refresh();
            this.@compSel.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
