﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Textblock : ObservableEntity<textblockType>, IUiElementHolder, IDeepCopyable<Textblock>, IDeepCopyFrom<Textblock>, IDisposable
    {
        private UiGroup @uiGroup = null;
        private UiOrder @uiOrder = null;
        private string @id = string.Empty;
        private string @text = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Textblock()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Textblock(Textblock source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Textblock(textblockType source)
        {
            if (source.@ui_group != null)
            {
                this.UiGroup = new UiGroup(source.@ui_group);
            }
            else
            {
                this.UiGroup = null;
            }
            if (source.@ui_order != null)
            {
                this.UiOrder = new UiOrder(source.@ui_order);
            }
            else
            {
                this.UiOrder = null;
            }
            this.@id = source.@id;
            this.@text = source.@text;
        }

        public void Dispose()
        {
            if (this.@uiGroup != null)
            {
                this.@uiGroup.PropertyChanged -= this.OnPropertyChanged;
            }
            if (this.@uiOrder != null)
            {
                this.@uiOrder.PropertyChanged -= this.OnPropertyChanged;
            }
        }

        public UiGroup UiGroup
        {
            get
            {
                return this.@uiGroup;
            }

            set
            {
                if (this.@uiGroup == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(UiGroup).IsAssignableFrom(value.GetType()) : true);
                if (this.@uiGroup != null)
                {
                    (this.@uiGroup as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@uiGroup != null)
                {
                    this.@uiGroup.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@uiGroup, value, () => this.CalcCRC());
            }
        }

        public UiOrder UiOrder
        {
            get
            {
                return this.@uiOrder;
            }

            set
            {
                if (this.@uiOrder == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(UiOrder).IsAssignableFrom(value.GetType()) : true);
                if (this.@uiOrder != null)
                {
                    (this.@uiOrder as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@uiOrder != null)
                {
                    this.@uiOrder.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@uiOrder, value, () => this.CalcCRC());
            }
        }

        public string Id
        {
            get
            {
                return this.@id;
            }

            set
            {
                if (this.@id == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@id, value, () => this.CalcCRC());
            }
        }

        public string Text
        {
            get
            {
                return this.@text;
            }

            set
            {
                if (this.@text == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@text, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override textblockType CreateSerializableData()
        {
            var writeData = new textblockType();
            if (this.@uiGroup != null)
            {
                writeData.@ui_group = this.UiGroup.CreateSerializableData() as ui_groupType;
            }
            if (this.@uiOrder != null)
            {
                writeData.@ui_order = this.UiOrder.CreateSerializableData() as ui_orderType;
            }
            writeData.@id = this.Id;
            writeData.@text = this.Text;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Textblock IDeepCopyable<Textblock>.DeepCopy()
        {
            return new Textblock(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Textblock source)
        {
            if (source.@uiGroup == null)
            {
                this.UiGroup = null;
            }
            else
            {
                this.UiGroup.DeepCopyFrom(source.@uiGroup);
            }
            if (source.@uiOrder == null)
            {
                this.UiOrder = null;
            }
            else
            {
                this.UiOrder.DeepCopyFrom(source.@uiOrder);
            }
            this.@id = source.@id;
            this.@text = source.@text;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            if (this.UiGroup != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UiGroup.HashValue)));
            }
            if (this.UiOrder != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UiOrder.HashValue)));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Id)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Text)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            if (this.@uiGroup != null)
            {
                this.@uiGroup.AutoCalc = this.AutoCalc;
            }
            if (this.@uiOrder != null)
            {
                this.@uiOrder.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            if (this.@uiGroup != null)
            {
                this.@uiGroup.Reset();
            }
            if (this.@uiOrder != null)
            {
                this.@uiOrder.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            if (this.@uiGroup != null)
            {
                this.@uiGroup.Refresh();
            }
            if (this.@uiOrder != null)
            {
                this.@uiOrder.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
